package com.alibaba.schedulerx.worker.timer;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.lang.management.OperatingSystemMXBean;
import java.math.BigDecimal;

import com.alibaba.schedulerx.common.domain.LimitedQueue;
import com.alibaba.schedulerx.common.domain.Metrics;
import com.alibaba.schedulerx.common.monitor.MetricsCollector;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.metrics.CgroupMetrics;

/**
 * @author zhaibian
 * @version $Id: MetricsCollectorTimer.java, v 0.1 2019年01月15日 21:22 zhaibian Exp $
 */
public class MetricsCollectorTimer extends AbstractTimerTask {
    private static int QUEUE_LIMIT = 5;
    private static final LimitedQueue<Double> LOAD_QUEUE = new LimitedQueue<>(QUEUE_LIMIT);
    private static final LimitedQueue<Double> HEAP_QUEUE = new LimitedQueue<>(QUEUE_LIMIT);
    private boolean enableCgroup = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.CGROUP_MERTRICS_ENABLE,
            WorkerConstants.CGROUP_MERTRICS_ENABLE_DEFAULT);

    @Override
    public String getName() {
        return "MetricsCollectorTimer";
    }

    @Override
    public long getInitialDelay() {
        return 10;
    }

    @Override
    public long getPeriod() {
        return 60;
    }

    @Override
    public void run() {
        try {
            Metrics metrics = new Metrics();
            this.buildCPUInfo(metrics);
            this.buildJvmHeapInfo(metrics);
            this.buildDiskInfo(metrics);
            MetricsCollector.setMetrics(metrics);
            LOGGER.info("enableCgroup={}, cpuUsage={}, jvmHeapUsage={}, diskUsage={}", enableCgroup, 
                    metrics.getCpuLoad5(), metrics.getHeap1Usage(), metrics.getDiskUsage());
        } catch (Throwable ex) {
            LOGGER.error("MetricsCollectorTimer error.", ex);
        }
    }


    private void buildDiskInfo(Metrics metrics) {
        File[] roots = File.listRoots();
        int free = 0;
        int total = 0;
        for (File file : roots) {
            if (file != null) {
                free += file.getFreeSpace() / 1024 / 1024;
                total += file.getTotalSpace() / 1024 / 1024;
            }
        }
        metrics.setDiskUsed(total - free);
        metrics.setDiskMax(total);
        if (total != 0 ) {
            metrics.setDiskUsage((double)metrics.getDiskUsed() / total);
        } else {
            metrics.setDiskUsage(0.0);
        }
    }

    private void buildCPUInfo(Metrics metrics) {
        if (enableCgroup) {
            CgroupMetrics cGroupMetrics = CgroupMetrics.getInstance();
            metrics.setCpuProcessors(cGroupMetrics.getCpuCores());
            BigDecimal bd = new BigDecimal(cGroupMetrics.getCpuUsagePercent()); 
            double cpuLoad = bd.setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue(); 
            metrics.setCpuLoad1(cpuLoad);
            metrics.setCpuLoad5(cpuLoad);
        } else {
            OperatingSystemMXBean operatingSystemMXBean = ManagementFactory
                .getOperatingSystemMXBean();
            metrics.setCpuProcessors(operatingSystemMXBean.getAvailableProcessors());
            LOAD_QUEUE.offer(operatingSystemMXBean.getSystemLoadAverage());
    
            Object[] loads = LOAD_QUEUE.toArray();
            double totalLoads = 0D;
            for (Object o : loads) {
                totalLoads += (Double) o;
            }
            metrics.setCpuLoad1((Double) loads[loads.length - 1]);
            metrics.setCpuLoad5(totalLoads / loads.length);
        }
    }

    private void buildJvmHeapInfo(Metrics metrics) {
        MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
        MemoryUsage memoryUsage = memoryMXBean.getHeapMemoryUsage();

        int used = (int) (memoryUsage.getUsed() / 1024 / 1024);
        int max = (int) (memoryUsage.getMax() / 1024 / 1024);

        metrics.setHeap1Used(used);
        metrics.setHeapMax(max);
        metrics.setHeap1Usage((double) used / max);
        HEAP_QUEUE.offer(metrics.getHeap1Usage());

        Object[] heaps = HEAP_QUEUE.toArray();
        double totalHeap = 0;
        for (Object o : heaps) {
            totalHeap += (Double) o;
        }
        metrics.setHeap5Usage(totalHeap / heaps.length);
    }
}