package com.alterioncorp.requestlogger;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.List;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.sql.rowset.serial.SerialClob;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersisterJdbcImpl implements Persister, Constants {
	
	private final Logger log = LoggerFactory.getLogger(this.getClass());
	
	private final DataSource dataSource;
	private final String tableName;

	public PersisterJdbcImpl() {

		super();
		
		tableName = System.getProperty(
				PROPERTY_NAME_JDBC_TABLE_NAME, PROPERTY_DEFAULT_JDBC_TABLE_NAME);

		String dataSourceJndi = System.getProperty(
				PROPERTY_NAME_JDBC_DATASOURCE_JNDI, PROPERTY_DEFAULT_JDBC_DATASOURCE_JNDI);

		try {
			InitialContext initialContext = new InitialContext();
			dataSource = (DataSource)initialContext.lookup(dataSourceJndi);
		}
		catch (NamingException e) {
			throw new RuntimeException(e);
		}
	}

	public PersisterJdbcImpl(DataSource dataSource, String tableName) {
		super();
		this.dataSource = dataSource;
		this.tableName = tableName;
	}

	@Override
	public void saveRequests(List<Request> requests, PropertyRegistry propertyRegistry) {
		
		if (requests != null && ! requests.isEmpty()) {

			StringBuilder sql = new StringBuilder();
			sql.append("insert into ");
			sql.append(tableName);
			sql.append(" (");
			for (int i = 0; i < propertyRegistry.getRegisteredProperties().size(); i++) {
				if (i > 0) {
					sql.append(", ");
				}
				sql.append(propertyRegistry.getRegisteredProperties().get(i));
			}
			sql.append(" ) values ( ");
			for (int i = 0; i < propertyRegistry.getRegisteredProperties().size(); i++) {
				if (i > 0) {
					sql.append(", ");
				}
				sql.append("?");
			}
			sql.append(")");

			Connection connection = null;
			PreparedStatement statement = null;
			
			try {
				
				connection = dataSource.getConnection();
				connection.setAutoCommit(false);
				
				statement = connection.prepareStatement(sql.toString());
				
				for (Request request : requests) {

					for (int i = 0; i < propertyRegistry.getRegisteredProperties().size(); i++) {
						
						String property = propertyRegistry.getRegisteredProperties().get(i);
						int type = propertyRegistry.getDataType(property);
						Object value = request.getData().get(property);
						
						if (type == Types.CLOB && value instanceof String) {
							value = new SerialClob(((String)value).toCharArray());
						}
						
						if (value == null) {
							statement.setNull(i + 1, type);
						}
						else {
							statement.setObject(i + 1, value, type);
						}
					}
					
					statement.addBatch();
				}
				
				statement.executeBatch();
				connection.commit();
			}
			catch (SQLException e) {
				try {
					if (connection != null) {
						connection.rollback();
					}
				}
				catch (SQLException e2) {
					log.error("Error rolling back transaction", e2);
				}
				throw new RuntimeException(e);
			}
			finally {
				close(statement);
				close(connection);
			}
		}
	}
	
	public String getTableName() {
		return tableName;
	}

	DataSource getDataSource() {
		return dataSource;
	}
	
	static void close(ResultSet closeable) {
		if (closeable != null) {
			try {
				closeable.close();
			}
			catch (Exception e) {
				// ignore
			}
		}
	}

	static void close(Statement closeable) {
		if (closeable != null) {
			try {
				closeable.close();
			}
			catch (Exception e) {
				// ignore
			}
		}
	}
	
	static void close(Connection closeable) {
		if (closeable != null) {
			try {
				closeable.close();
			}
			catch (Exception e) {
				// ignore
			}
		}
	}
}
