/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.dynamodb;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSCredentialsProviderChain;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDBClient;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemRequest;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemResult;
import com.amazonaws.services.dynamodbv2.model.Condition;
import com.amazonaws.services.dynamodbv2.model.ConsumedCapacity;
import com.amazonaws.services.dynamodbv2.model.DeleteRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableResult;
import com.amazonaws.services.dynamodbv2.model.PutRequest;
import com.amazonaws.services.dynamodbv2.model.QueryRequest;
import com.amazonaws.services.dynamodbv2.model.QueryResult;
import com.amazonaws.services.dynamodbv2.model.ReturnConsumedCapacity;
import com.amazonaws.services.dynamodbv2.model.ScanRequest;
import com.amazonaws.services.dynamodbv2.model.ScanResult;
import com.amazonaws.services.dynamodbv2.model.TableDescription;
import com.amazonaws.services.dynamodbv2.model.WriteRequest;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.primitives.Ints;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Callable;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.dynamodb.CredentialPairName;
import org.apache.hadoop.dynamodb.DynamoDBFibonacciRetryer;
import org.apache.hadoop.dynamodb.DynamoDBUtil;
import org.apache.hadoop.dynamodb.PrintCounter;
import org.apache.hadoop.dynamodb.filter.DynamoDBQueryFilter;
import org.apache.hadoop.mapred.Reporter;
import org.apache.hadoop.util.ReflectionUtils;
import org.joda.time.Duration;

public class DynamoDBClient {
    private static final Log log = LogFactory.getLog(DynamoDBClient.class);
    private static final int DEFAULT_RETRY_DURATION = 10;
    private static final long MAX_BACKOFF_IN_MILLISECONDS = 3000L;
    private static final CredentialPairName DYNAMODB_CREDENTIAL_PAIR_NAME = new CredentialPairName("dynamodb.awsAccessKeyId", "dynamodb.awsSecretAccessKey");
    private static final CredentialPairName DEFAULT_CREDENTIAL_PAIR_NAME = new CredentialPairName("fs.s3.awsAccessKeyId", "fs.s3.awsSecretAccessKey");
    private final Map<String, List<WriteRequest>> writeBatchMap = new HashMap<String, List<WriteRequest>>();
    private final AmazonDynamoDBClient dynamoDB;
    private int writeBatchMapSizeBytes;
    private int batchWriteRetries;
    private final Configuration config;
    private final long maxBatchSize;
    private final long maxItemByteSize;

    public DynamoDBClient() {
        this.dynamoDB = null;
        this.config = null;
        this.maxBatchSize = 0x1000000L;
        this.maxItemByteSize = 409600L;
    }

    public DynamoDBClient(Configuration conf) {
        this(conf, null);
    }

    public DynamoDBClient(Configuration conf, String region) {
        Preconditions.checkNotNull((Object)conf, (Object)"conf cannot be null.");
        this.config = conf;
        this.dynamoDB = this.getDynamoDBClient(conf);
        this.dynamoDB.setEndpoint(DynamoDBUtil.getDynamoDBEndpoint(conf, region));
        this.maxBatchSize = this.config.getLong("dynamodb.max.batch.size", 0x1000000L);
        this.maxItemByteSize = this.config.getLong("dynamodb.max.item.size", 409600L);
    }

    public final Map<String, List<WriteRequest>> getWriteBatchMap() {
        return this.writeBatchMap;
    }

    public TableDescription describeTable(String tableName) {
        final DescribeTableRequest describeTablesRequest = new DescribeTableRequest().withTableName(tableName);
        try {
            DynamoDBFibonacciRetryer.RetryResult<DescribeTableResult> describeResult = this.getRetryDriver().runWithRetry(new Callable<DescribeTableResult>(){

                @Override
                public DescribeTableResult call() {
                    DescribeTableResult result = DynamoDBClient.this.dynamoDB.describeTable(describeTablesRequest);
                    log.info((Object)("Describe table output: " + result));
                    return result;
                }
            }, null, null);
            return ((DescribeTableResult)describeResult.result).getTable();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not lookup table " + tableName + " in DynamoDB.", e);
        }
    }

    public DynamoDBFibonacciRetryer.RetryResult<ScanResult> scanTable(String tableName, DynamoDBQueryFilter dynamoDBQueryFilter, Integer segment, Integer totalSegments, Map<String, AttributeValue> exclusiveStartKey, long limit, Reporter reporter) {
        Map<String, Condition> scanFilter;
        final ScanRequest scanRequest = new ScanRequest(tableName).withExclusiveStartKey(exclusiveStartKey).withLimit(Integer.valueOf(Ints.checkedCast((long)limit))).withSegment(segment).withTotalSegments(totalSegments).withReturnConsumedCapacity(ReturnConsumedCapacity.TOTAL);
        if (dynamoDBQueryFilter != null && !(scanFilter = dynamoDBQueryFilter.getScanFilter()).isEmpty()) {
            scanRequest.setScanFilter(scanFilter);
        }
        DynamoDBFibonacciRetryer.RetryResult<ScanResult> retryResult = this.getRetryDriver().runWithRetry(new Callable<ScanResult>(){

            @Override
            public ScanResult call() {
                log.debug((Object)("Executing DynamoDB scan: " + scanRequest));
                return DynamoDBClient.this.dynamoDB.scan(scanRequest);
            }
        }, reporter, PrintCounter.DynamoDBReadThrottle);
        return retryResult;
    }

    public DynamoDBFibonacciRetryer.RetryResult<QueryResult> queryTable(String tableName, DynamoDBQueryFilter dynamoDBQueryFilter, Map<String, AttributeValue> exclusiveStartKey, long limit, Reporter reporter) {
        final QueryRequest queryRequest = new QueryRequest().withTableName(tableName).withExclusiveStartKey(exclusiveStartKey).withKeyConditions(dynamoDBQueryFilter.getKeyConditions()).withLimit(Integer.valueOf(Ints.checkedCast((long)limit))).withReturnConsumedCapacity(ReturnConsumedCapacity.TOTAL);
        DynamoDBFibonacciRetryer.RetryResult<QueryResult> retryResult = this.getRetryDriver().runWithRetry(new Callable<QueryResult>(){

            @Override
            public QueryResult call() {
                log.debug((Object)("Executing DynamoDB query: " + queryRequest));
                return DynamoDBClient.this.dynamoDB.query(queryRequest);
            }
        }, reporter, PrintCounter.DynamoDBReadThrottle);
        return retryResult;
    }

    public BatchWriteItemResult putBatch(String tableName, Map<String, AttributeValue> item, long maxItemsPerBatch, Reporter reporter, boolean deletionMode) throws UnsupportedEncodingException {
        List<Object> writeBatchList;
        int itemSizeBytes = DynamoDBUtil.getItemSizeBytes(item);
        if ((long)itemSizeBytes > this.maxItemByteSize) {
            throw new RuntimeException("Cannot pass items with size greater than " + this.maxItemByteSize + ". Item with size of " + itemSizeBytes + " was given.");
        }
        maxItemsPerBatch = DynamoDBUtil.getBoundedBatchLimit(this.config, maxItemsPerBatch);
        BatchWriteItemResult result = null;
        if (this.writeBatchMap.containsKey(tableName)) {
            boolean totalSizeOfWriteBatchesOverLimit;
            boolean writeRequestsForTableAtLimit = (long)this.writeBatchMap.get(tableName).size() >= maxItemsPerBatch;
            boolean bl = totalSizeOfWriteBatchesOverLimit = (long)(this.writeBatchMapSizeBytes + itemSizeBytes) > this.maxBatchSize;
            if (writeRequestsForTableAtLimit || totalSizeOfWriteBatchesOverLimit) {
                result = this.writeBatch(reporter, itemSizeBytes);
            }
        }
        if (!this.writeBatchMap.containsKey(tableName)) {
            writeBatchList = new ArrayList((int)maxItemsPerBatch);
            this.writeBatchMap.put(tableName, writeBatchList);
        } else {
            writeBatchList = this.writeBatchMap.get(tableName);
        }
        log.debug((Object)("BatchWriteItem deletionMode " + deletionMode));
        if (deletionMode) {
            writeBatchList.add(new WriteRequest().withDeleteRequest(new DeleteRequest().withKey(item)));
        } else {
            writeBatchList.add(new WriteRequest().withPutRequest(new PutRequest().withItem(item)));
        }
        this.writeBatchMapSizeBytes += itemSizeBytes;
        return result;
    }

    public void close() {
        while (!this.writeBatchMap.isEmpty()) {
            this.writeBatch(Reporter.NULL, 0);
        }
        if (this.dynamoDB != null) {
            this.dynamoDB.shutdown();
        }
    }

    private BatchWriteItemResult writeBatch(Reporter reporter, final int roomNeeded) {
        final BatchWriteItemRequest batchWriteItemRequest = new BatchWriteItemRequest().withRequestItems(this.writeBatchMap).withReturnConsumedCapacity(ReturnConsumedCapacity.TOTAL);
        DynamoDBFibonacciRetryer.RetryResult<BatchWriteItemResult> retryResult = this.getRetryDriver().runWithRetry(new Callable<BatchWriteItemResult>(){

            @Override
            public BatchWriteItemResult call() throws UnsupportedEncodingException, InterruptedException {
                DynamoDBClient.this.pauseExponentially(DynamoDBClient.this.batchWriteRetries);
                BatchWriteItemResult result = DynamoDBClient.this.dynamoDB.batchWriteItem(batchWriteItemRequest);
                Map unprocessedItems = result.getUnprocessedItems();
                if (unprocessedItems == null || unprocessedItems.isEmpty()) {
                    DynamoDBClient.this.batchWriteRetries = 0;
                } else {
                    DynamoDBClient.this.batchWriteRetries++;
                    int unprocessedItemCount = 0;
                    for (List unprocessedWriteRequests : unprocessedItems.values()) {
                        unprocessedItemCount += unprocessedWriteRequests.size();
                        int batchSizeBytes = 0;
                        for (WriteRequest request : unprocessedWriteRequests) {
                            batchSizeBytes += DynamoDBUtil.getItemSizeBytes(request.getPutRequest().getItem());
                        }
                        long maxItemsPerBatch = DynamoDBClient.this.config.getLong("dynamodb.max.batch.items", 25L);
                        long maxBatchSize = DynamoDBClient.this.config.getLong("dynamodb.max.batch.size", 0x1000000L);
                        if ((long)unprocessedWriteRequests.size() < maxItemsPerBatch && maxBatchSize - (long)batchSizeBytes >= (long)roomNeeded) continue;
                        throw new AmazonClientException("Full list of write requests not processed");
                    }
                    double consumed = 0.0;
                    for (ConsumedCapacity consumedCapacity : result.getConsumedCapacity()) {
                        consumed = consumedCapacity.getCapacityUnits();
                    }
                    int batchSize = 0;
                    for (List writeRequests : batchWriteItemRequest.getRequestItems().values()) {
                        batchSize += writeRequests.size();
                    }
                    log.debug((Object)("BatchWriteItem attempted " + batchSize + " items, consumed " + consumed + " wcu, left unprocessed " + unprocessedItemCount + " items, now at " + DynamoDBClient.this.batchWriteRetries + " retries"));
                }
                return result;
            }
        }, reporter, PrintCounter.DynamoDBWriteThrottle);
        this.writeBatchMap.clear();
        this.writeBatchMapSizeBytes = 0;
        Map unprocessedItems = ((BatchWriteItemResult)retryResult.result).getUnprocessedItems();
        for (Map.Entry entry : unprocessedItems.entrySet()) {
            String key = (String)entry.getKey();
            List requests = (List)entry.getValue();
            for (WriteRequest request : requests) {
                this.writeBatchMapSizeBytes += DynamoDBUtil.getItemSizeBytes(request.getPutRequest().getItem());
            }
            this.writeBatchMap.put(key, requests);
        }
        return (BatchWriteItemResult)retryResult.result;
    }

    private DynamoDBFibonacciRetryer getRetryDriver() {
        return new DynamoDBFibonacciRetryer(Duration.standardMinutes((long)10L));
    }

    private void pauseExponentially(int retries) throws InterruptedException {
        if (retries == 0) {
            return;
        }
        long scaleFactor = 500 + new Random().nextInt(100);
        long delay = (long)(Math.pow(2.0, retries) * (double)scaleFactor) / 4L;
        delay = Math.min(delay, 3000L);
        log.info((Object)("Pausing " + delay + " ms at retry " + retries));
        Thread.sleep(delay);
    }

    private AmazonDynamoDBClient getDynamoDBClient(Configuration conf) {
        ClientConfiguration clientConfig = new ClientConfiguration().withMaxErrorRetry(1);
        this.applyProxyConfiguration(clientConfig, conf);
        return new AmazonDynamoDBClient(this.getAWSCredentialsProvider(conf), clientConfig);
    }

    @VisibleForTesting
    void applyProxyConfiguration(ClientConfiguration clientConfig, Configuration conf) {
        String proxyHost = conf.get("dynamodb.proxy.hostname");
        int proxyPort = conf.getInt("dynamodb.proxy.port", 0);
        String proxyUsername = conf.get("dynamodb.proxy.username");
        String proxyPassword = conf.get("dynamodb.proxy.password");
        boolean proxyHostAndPortPresent = false;
        if (!Strings.isNullOrEmpty((String)proxyHost) && proxyPort > 0) {
            clientConfig.setProxyHost(proxyHost);
            clientConfig.setProxyPort(proxyPort);
            proxyHostAndPortPresent = true;
        } else if (Strings.isNullOrEmpty((String)proxyHost) ^ proxyPort <= 0) {
            throw new RuntimeException("Only one of proxy host and port are set, when both are required");
        }
        if (!Strings.isNullOrEmpty((String)proxyUsername) && !Strings.isNullOrEmpty((String)proxyPassword)) {
            if (!proxyHostAndPortPresent) {
                throw new RuntimeException("Proxy host and port must be supplied if proxy username and password are present");
            }
            clientConfig.setProxyUsername(proxyUsername);
            clientConfig.setProxyPassword(proxyPassword);
        } else if (Strings.isNullOrEmpty((String)proxyUsername) ^ Strings.isNullOrEmpty((String)proxyPassword)) {
            throw new RuntimeException("Only one of proxy username and password are set, when both are required");
        }
    }

    protected AWSCredentialsProvider getAWSCredentialsProvider(Configuration conf) {
        String secretKey;
        String accessKey;
        ArrayList<Object> providersList = new ArrayList<Object>();
        String providerClass = conf.get("dynamodb.customAWSCredentialsProvider");
        if (!Strings.isNullOrEmpty((String)providerClass)) {
            try {
                providersList.add((AWSCredentialsProvider)ReflectionUtils.newInstance(Class.forName(providerClass), (Configuration)conf));
            }
            catch (ClassNotFoundException e) {
                throw new RuntimeException("Custom AWSCredentialsProvider not found: " + providerClass, e);
            }
        }
        if (Strings.isNullOrEmpty((String)(accessKey = conf.get(DYNAMODB_CREDENTIAL_PAIR_NAME.getAccessKeyName())))) {
            accessKey = conf.get(DEFAULT_CREDENTIAL_PAIR_NAME.getAccessKeyName());
            secretKey = conf.get(DEFAULT_CREDENTIAL_PAIR_NAME.getSecretKeyName());
        } else {
            secretKey = conf.get(DYNAMODB_CREDENTIAL_PAIR_NAME.getSecretKeyName());
        }
        if (Strings.isNullOrEmpty((String)accessKey) || Strings.isNullOrEmpty((String)secretKey)) {
            providersList.add(new InstanceProfileCredentialsProvider());
        } else {
            BasicAWSCredentials credentials = new BasicAWSCredentials(accessKey, secretKey);
            providersList.add(new AWSCredentialsProvider((AWSCredentials)credentials){
                final /* synthetic */ AWSCredentials val$credentials;
                {
                    this.val$credentials = aWSCredentials;
                }

                public AWSCredentials getCredentials() {
                    return this.val$credentials;
                }

                public void refresh() {
                }
            });
        }
        AWSCredentialsProvider[] providerArray = providersList.toArray(new AWSCredentialsProvider[providersList.size()]);
        AWSCredentialsProviderChain providerChain = new AWSCredentialsProviderChain(providerArray);
        providerChain.setReuseLastProvider(true);
        return providerChain;
    }
}

