/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.kinesis.agg;

import com.amazonaws.annotation.NotThreadSafe;
import com.amazonaws.services.kinesis.clientlibrary.types.Messages;
import com.amazonaws.services.kinesis.model.PutRecordRequest;
import com.amazonaws.services.kinesis.model.PutRecordsRequestEntry;
import com.google.protobuf.ByteString;
import java.io.ByteArrayOutputStream;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.apache.commons.lang.StringUtils;

@NotThreadSafe
public class AggRecord {
    private static final byte[] AGGREGATED_RECORD_MAGIC = new byte[]{-13, -119, -102, -62};
    protected static final String MESSAGE_DIGEST_NAME = "MD5";
    private static final BigInteger UINT_128_MAX = new BigInteger(StringUtils.repeat((String)"FF", (int)16), 16);
    protected static final int MAX_BYTES_PER_RECORD = 0x100000;
    protected static final int AGGREGATION_OVERHEAD_BYTES = 256;
    protected static final int PARTITION_KEY_MIN_LENGTH = 1;
    protected static final int PARTITION_KEY_MAX_LENGTH = 256;
    private int aggregatedMessageSizeBytes = 0;
    private final KeySet explicitHashKeys;
    private final KeySet partitionKeys;
    private Messages.AggregatedRecord.Builder aggregatedRecordBuilder = Messages.AggregatedRecord.newBuilder();
    private final MessageDigest md5;
    private String aggPartitionKey = "";
    private String aggExplicitHashKey = "";

    public AggRecord() {
        this.explicitHashKeys = new KeySet();
        this.partitionKeys = new KeySet();
        try {
            this.md5 = MessageDigest.getInstance(MESSAGE_DIGEST_NAME);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Could not create an MD5 message digest.", e);
        }
    }

    public int getNumUserRecords() {
        return this.aggregatedRecordBuilder.getRecordsCount();
    }

    public int getSizeBytes() {
        if (this.getNumUserRecords() == 0) {
            return 0;
        }
        return AGGREGATED_RECORD_MAGIC.length + this.aggregatedMessageSizeBytes + this.md5.getDigestLength();
    }

    public byte[] toRecordBytes() {
        if (this.getNumUserRecords() == 0) {
            return new byte[0];
        }
        byte[] messageBody = this.aggregatedRecordBuilder.build().toByteArray();
        this.md5.reset();
        byte[] messageDigest = this.md5.digest(messageBody);
        ByteArrayOutputStream baos = new ByteArrayOutputStream(this.getSizeBytes());
        baos.write(AGGREGATED_RECORD_MAGIC, 0, AGGREGATED_RECORD_MAGIC.length);
        baos.write(messageBody, 0, messageBody.length);
        baos.write(messageDigest, 0, messageDigest.length);
        return baos.toByteArray();
    }

    public void clear() {
        this.md5.reset();
        this.aggExplicitHashKey = "";
        this.aggPartitionKey = "";
        this.aggregatedMessageSizeBytes = 0;
        this.explicitHashKeys.clear();
        this.partitionKeys.clear();
        this.aggregatedRecordBuilder = Messages.AggregatedRecord.newBuilder();
    }

    public String getPartitionKey() {
        if (this.getNumUserRecords() == 0) {
            return null;
        }
        return this.aggPartitionKey;
    }

    public String getExplicitHashKey() {
        if (this.getNumUserRecords() == 0) {
            return null;
        }
        return this.aggExplicitHashKey;
    }

    private int calculateRecordSize(String partitionKey, String explicitHashKey, byte[] data) {
        int messageSize = 0;
        if (!this.partitionKeys.contains(partitionKey)) {
            int pkLength = partitionKey.length();
            ++messageSize;
            messageSize += this.calculateVarintSize(pkLength);
            messageSize += pkLength;
        }
        if (!this.explicitHashKeys.contains(explicitHashKey)) {
            int ehkLength = explicitHashKey.length();
            ++messageSize;
            messageSize += this.calculateVarintSize(ehkLength);
            messageSize += ehkLength;
        }
        long innerRecordSize = 0L;
        ++innerRecordSize;
        innerRecordSize += (long)this.calculateVarintSize(this.partitionKeys.getPotentialIndex(partitionKey));
        if (explicitHashKey != null) {
            ++innerRecordSize;
            innerRecordSize += (long)this.calculateVarintSize(this.explicitHashKeys.getPotentialIndex(explicitHashKey));
        }
        ++innerRecordSize;
        innerRecordSize += (long)this.calculateVarintSize(data.length);
        ++messageSize;
        messageSize += this.calculateVarintSize(innerRecordSize += (long)data.length);
        messageSize = (int)((long)messageSize + innerRecordSize);
        return messageSize;
    }

    private int calculateVarintSize(long value) {
        if (value < 0L) {
            throw new IllegalArgumentException("Size values should not be negative.");
        }
        int numBitsNeeded = 0;
        if (value == 0L) {
            numBitsNeeded = 1;
        } else {
            while (value > 0L) {
                ++numBitsNeeded;
                value >>= 1;
            }
        }
        int numVarintBytes = numBitsNeeded / 7;
        if (numBitsNeeded % 7 > 0) {
            ++numVarintBytes;
        }
        return numVarintBytes;
    }

    public boolean addUserRecord(String partitionKey, String explicitHashKey, byte[] data) {
        explicitHashKey = explicitHashKey != null ? explicitHashKey : this.createExplicitHashKey(partitionKey);
        this.validatePartitionKey(partitionKey);
        this.validateExplicitHashKey(explicitHashKey);
        this.validateData(data);
        int sizeOfNewRecord = this.calculateRecordSize(partitionKey, explicitHashKey, data);
        if (this.getSizeBytes() + sizeOfNewRecord > 0x100000) {
            return false;
        }
        if (sizeOfNewRecord > 0x100000) {
            throw new IllegalArgumentException("Input record (PK=" + partitionKey + ", EHK=" + explicitHashKey + ", SizeBytes=" + sizeOfNewRecord + ") is larger than the maximum size before Aggregation encoding of " + 1048320 + " bytes");
        }
        Messages.Record.Builder newRecord = Messages.Record.newBuilder().setData(data != null ? ByteString.copyFrom((byte[])data) : ByteString.EMPTY);
        ExistenceIndexPair pkAddResult = this.partitionKeys.add(partitionKey);
        if (pkAddResult.getFirst().booleanValue()) {
            this.aggregatedRecordBuilder.addPartitionKeyTable(partitionKey);
        }
        newRecord.setPartitionKeyIndex(pkAddResult.getSecond().longValue());
        ExistenceIndexPair ehkAddResult = this.explicitHashKeys.add(explicitHashKey);
        if (ehkAddResult.getFirst().booleanValue()) {
            this.aggregatedRecordBuilder.addExplicitHashKeyTable(explicitHashKey);
        }
        newRecord.setExplicitHashKeyIndex(ehkAddResult.getSecond().longValue());
        this.aggregatedMessageSizeBytes += sizeOfNewRecord;
        this.aggregatedRecordBuilder.addRecords(newRecord.build());
        if (this.aggregatedRecordBuilder.getRecordsCount() == 1) {
            this.aggPartitionKey = partitionKey;
            this.aggExplicitHashKey = explicitHashKey;
        }
        return true;
    }

    public PutRecordRequest toPutRecordRequest(String streamName) {
        byte[] recordBytes = this.toRecordBytes();
        ByteBuffer bb = ByteBuffer.wrap(recordBytes);
        return new PutRecordRequest().withStreamName(streamName).withExplicitHashKey(this.getExplicitHashKey()).withPartitionKey(this.getPartitionKey()).withData(bb);
    }

    public PutRecordsRequestEntry toPutRecordsRequestEntry() {
        return new PutRecordsRequestEntry().withExplicitHashKey(this.getExplicitHashKey()).withPartitionKey(this.getPartitionKey()).withData(ByteBuffer.wrap(this.toRecordBytes()));
    }

    private void validateData(byte[] data) {
        int maxAllowableDataLength = 0x100000 - AGGREGATED_RECORD_MAGIC.length - this.md5.getDigestLength();
        if (data != null && data.length > maxAllowableDataLength) {
            throw new IllegalArgumentException("Data must be less than or equal to " + maxAllowableDataLength + " bytes in size, got " + data.length + " bytes");
        }
    }

    private void validatePartitionKey(String partitionKey) {
        if (partitionKey == null) {
            throw new IllegalArgumentException("Partition key cannot be null");
        }
        if (partitionKey.length() < 1 || partitionKey.length() > 256) {
            throw new IllegalArgumentException("Invalid partition key. Length must be at least 1 and at most 256, got length of " + partitionKey.length());
        }
        try {
            partitionKey.getBytes(StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Partition key must be valid " + StandardCharsets.UTF_8.displayName());
        }
    }

    private void validateExplicitHashKey(String explicitHashKey) {
        if (explicitHashKey == null) {
            return;
        }
        BigInteger b = null;
        try {
            b = new BigInteger(explicitHashKey);
            if (b.compareTo(UINT_128_MAX) > 0 || b.compareTo(BigInteger.ZERO) < 0) {
                throw new IllegalArgumentException("Invalid explicitHashKey, must be greater or equal to zero and less than or equal to (2^128 - 1), got " + explicitHashKey);
            }
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid explicitHashKey, must be an integer, got " + explicitHashKey);
        }
    }

    private String createExplicitHashKey(String partitionKey) {
        BigInteger hashKey = BigInteger.ZERO;
        this.md5.reset();
        byte[] pkDigest = this.md5.digest(partitionKey.getBytes(StandardCharsets.UTF_8));
        for (int i = 0; i < this.md5.getDigestLength(); ++i) {
            BigInteger p = new BigInteger(String.valueOf(pkDigest[i] & 0xFF));
            BigInteger shifted = p.shiftLeft((16 - i - 1) * 8);
            hashKey = hashKey.add(shifted);
        }
        return hashKey.toString(10);
    }

    private class ExistenceIndexPair {
        private Boolean first;
        private Long second;

        public ExistenceIndexPair(Boolean first, Long second) {
            this.first = first;
            this.second = second;
        }

        public Boolean getFirst() {
            return this.first;
        }

        public Long getSecond() {
            return this.second;
        }
    }

    private class KeySet {
        private List<String> keys = new LinkedList<String>();
        private Map<String, Long> lookup = new TreeMap<String, Long>();

        public Long getPotentialIndex(String s) {
            Long it = this.lookup.get(s);
            if (it != null) {
                return it;
            }
            return this.keys.size();
        }

        public ExistenceIndexPair add(String s) {
            Long it = this.lookup.get(s);
            if (it != null) {
                return new ExistenceIndexPair(false, it);
            }
            if (!this.lookup.containsKey(s)) {
                this.lookup.put(s, Long.valueOf(this.keys.size()));
            }
            this.keys.add(s);
            return new ExistenceIndexPair(true, Long.valueOf(this.keys.size() - 1));
        }

        public boolean contains(String s) {
            return s != null && this.lookup.containsKey(s);
        }

        public void clear() {
            this.keys.clear();
            this.lookup.clear();
        }
    }
}

