/*
 * Copyright 2019-2024 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package com.amazonaws.services.accessanalyzer;

import org.w3c.dom.*;

import java.net.*;
import java.util.*;

import javax.annotation.Generated;

import org.apache.commons.logging.*;

import com.amazonaws.*;
import com.amazonaws.annotation.SdkInternalApi;
import com.amazonaws.auth.*;

import com.amazonaws.handlers.*;
import com.amazonaws.http.*;
import com.amazonaws.internal.*;
import com.amazonaws.internal.auth.*;
import com.amazonaws.metrics.*;
import com.amazonaws.regions.*;
import com.amazonaws.transform.*;
import com.amazonaws.util.*;
import com.amazonaws.protocol.json.*;
import com.amazonaws.util.AWSRequestMetrics.Field;
import com.amazonaws.annotation.ThreadSafe;
import com.amazonaws.client.AwsSyncClientParams;
import com.amazonaws.client.builder.AdvancedConfig;

import com.amazonaws.services.accessanalyzer.AWSAccessAnalyzerClientBuilder;

import com.amazonaws.AmazonServiceException;

import com.amazonaws.services.accessanalyzer.model.*;

import com.amazonaws.services.accessanalyzer.model.transform.*;

/**
 * Client for accessing Access Analyzer. All service calls made using this client are blocking, and will not return
 * until the service call completes.
 * <p>
 * <p>
 * Identity and Access Management Access Analyzer helps you to set, verify, and refine your IAM policies by providing a
 * suite of capabilities. Its features include findings for external and unused access, basic and custom policy checks
 * for validating policies, and policy generation to generate fine-grained policies. To start using IAM Access Analyzer
 * to identify external or unused access, you first need to create an analyzer.
 * </p>
 * <p>
 * <b>External access analyzers</b> help identify potential risks of accessing resources by enabling you to identify any
 * resource policies that grant access to an external principal. It does this by using logic-based reasoning to analyze
 * resource-based policies in your Amazon Web Services environment. An external principal can be another Amazon Web
 * Services account, a root user, an IAM user or role, a federated user, an Amazon Web Services service, or an anonymous
 * user. You can also use IAM Access Analyzer to preview public and cross-account access to your resources before
 * deploying permissions changes.
 * </p>
 * <p>
 * <b>Unused access analyzers</b> help identify potential identity access risks by enabling you to identify unused IAM
 * roles, unused access keys, unused console passwords, and IAM principals with unused service and action-level
 * permissions.
 * </p>
 * <p>
 * Beyond findings, IAM Access Analyzer provides basic and custom policy checks to validate IAM policies before
 * deploying permissions changes. You can use policy generation to refine permissions by attaching a policy generated
 * using access activity logged in CloudTrail logs.
 * </p>
 * <p>
 * This guide describes the IAM Access Analyzer operations that you can call programmatically. For general information
 * about IAM Access Analyzer, see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/what-is-access-analyzer.html">Identity and Access Management
 * Access Analyzer</a> in the <b>IAM User Guide</b>.
 * </p>
 */
@ThreadSafe
@Generated("com.amazonaws:aws-java-sdk-code-generator")
public class AWSAccessAnalyzerClient extends AmazonWebServiceClient implements AWSAccessAnalyzer {

    /** Provider for AWS credentials. */
    private final AWSCredentialsProvider awsCredentialsProvider;

    private static final Log log = LogFactory.getLog(AWSAccessAnalyzer.class);

    /** Default signing name for the service. */
    private static final String DEFAULT_SIGNING_NAME = "access-analyzer";

    /** Client configuration factory providing ClientConfigurations tailored to this client */
    protected static final ClientConfigurationFactory configFactory = new ClientConfigurationFactory();

    private final AdvancedConfig advancedConfig;

    private static final com.amazonaws.protocol.json.SdkJsonProtocolFactory protocolFactory = new com.amazonaws.protocol.json.SdkJsonProtocolFactory(
            new JsonClientMetadata()
                    .withProtocolVersion("1.1")
                    .withSupportsCbor(false)
                    .withSupportsIon(false)
                    .withContentTypeOverride("application/json")
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("InvalidParameterException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.InvalidParameterExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ThrottlingException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.ThrottlingExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ServiceQuotaExceededException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.ServiceQuotaExceededExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("InternalServerException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.InternalServerExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("AccessDeniedException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.AccessDeniedExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ConflictException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.ConflictExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.ResourceNotFoundExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("UnprocessableEntityException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.UnprocessableEntityExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ValidationException").withExceptionUnmarshaller(
                                    com.amazonaws.services.accessanalyzer.model.transform.ValidationExceptionUnmarshaller.getInstance()))
                    .withBaseServiceExceptionClass(com.amazonaws.services.accessanalyzer.model.AWSAccessAnalyzerException.class));

    public static AWSAccessAnalyzerClientBuilder builder() {
        return AWSAccessAnalyzerClientBuilder.standard();
    }

    /**
     * Constructs a new client to invoke service methods on Access Analyzer using the specified parameters.
     *
     * <p>
     * All service calls made using this new client object are blocking, and will not return until the service call
     * completes.
     *
     * @param clientParams
     *        Object providing client parameters.
     */
    AWSAccessAnalyzerClient(AwsSyncClientParams clientParams) {
        this(clientParams, false);
    }

    /**
     * Constructs a new client to invoke service methods on Access Analyzer using the specified parameters.
     *
     * <p>
     * All service calls made using this new client object are blocking, and will not return until the service call
     * completes.
     *
     * @param clientParams
     *        Object providing client parameters.
     */
    AWSAccessAnalyzerClient(AwsSyncClientParams clientParams, boolean endpointDiscoveryEnabled) {
        super(clientParams);
        this.awsCredentialsProvider = clientParams.getCredentialsProvider();
        this.advancedConfig = clientParams.getAdvancedConfig();
        init();
    }

    private void init() {
        setServiceNameIntern(DEFAULT_SIGNING_NAME);
        setEndpointPrefix(ENDPOINT_PREFIX);
        // calling this.setEndPoint(...) will also modify the signer accordingly
        setEndpoint("access-analyzer.us-east-1.amazonaws.com");
        HandlerChainFactory chainFactory = new HandlerChainFactory();
        requestHandler2s.addAll(chainFactory.newRequestHandlerChain("/com/amazonaws/services/accessanalyzer/request.handlers"));
        requestHandler2s.addAll(chainFactory.newRequestHandler2Chain("/com/amazonaws/services/accessanalyzer/request.handler2s"));
        requestHandler2s.addAll(chainFactory.getGlobalHandlers());
    }

    /**
     * <p>
     * Retroactively applies the archive rule to existing findings that meet the archive rule criteria.
     * </p>
     * 
     * @param applyArchiveRuleRequest
     *        Retroactively applies an archive rule.
     * @return Result of the ApplyArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ApplyArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ApplyArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ApplyArchiveRuleResult applyArchiveRule(ApplyArchiveRuleRequest request) {
        request = beforeClientExecution(request);
        return executeApplyArchiveRule(request);
    }

    @SdkInternalApi
    final ApplyArchiveRuleResult executeApplyArchiveRule(ApplyArchiveRuleRequest applyArchiveRuleRequest) {

        ExecutionContext executionContext = createExecutionContext(applyArchiveRuleRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ApplyArchiveRuleRequest> request = null;
        Response<ApplyArchiveRuleResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ApplyArchiveRuleRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(applyArchiveRuleRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ApplyArchiveRule");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ApplyArchiveRuleResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ApplyArchiveRuleResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Cancels the requested policy generation.
     * </p>
     * 
     * @param cancelPolicyGenerationRequest
     * @return Result of the CancelPolicyGeneration operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CancelPolicyGeneration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CancelPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelPolicyGenerationResult cancelPolicyGeneration(CancelPolicyGenerationRequest request) {
        request = beforeClientExecution(request);
        return executeCancelPolicyGeneration(request);
    }

    @SdkInternalApi
    final CancelPolicyGenerationResult executeCancelPolicyGeneration(CancelPolicyGenerationRequest cancelPolicyGenerationRequest) {

        ExecutionContext executionContext = createExecutionContext(cancelPolicyGenerationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CancelPolicyGenerationRequest> request = null;
        Response<CancelPolicyGenerationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CancelPolicyGenerationRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(cancelPolicyGenerationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CancelPolicyGeneration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CancelPolicyGenerationResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CancelPolicyGenerationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Checks whether the specified access isn't allowed by a policy.
     * </p>
     * 
     * @param checkAccessNotGrantedRequest
     * @return Result of the CheckAccessNotGranted operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws InvalidParameterException
     *         The specified parameter is invalid.
     * @throws UnprocessableEntityException
     *         The specified entity could not be processed.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CheckAccessNotGranted
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CheckAccessNotGranted"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckAccessNotGrantedResult checkAccessNotGranted(CheckAccessNotGrantedRequest request) {
        request = beforeClientExecution(request);
        return executeCheckAccessNotGranted(request);
    }

    @SdkInternalApi
    final CheckAccessNotGrantedResult executeCheckAccessNotGranted(CheckAccessNotGrantedRequest checkAccessNotGrantedRequest) {

        ExecutionContext executionContext = createExecutionContext(checkAccessNotGrantedRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CheckAccessNotGrantedRequest> request = null;
        Response<CheckAccessNotGrantedResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CheckAccessNotGrantedRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(checkAccessNotGrantedRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CheckAccessNotGranted");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CheckAccessNotGrantedResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new CheckAccessNotGrantedResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Checks whether new access is allowed for an updated policy when compared to the existing policy.
     * </p>
     * <p>
     * You can find examples for reference policies and learn how to set up and run a custom policy check for new access
     * in the <a href="https://github.com/aws-samples/iam-access-analyzer-custom-policy-check-samples">IAM Access
     * Analyzer custom policy checks samples</a> repository on GitHub. The reference policies in this repository are
     * meant to be passed to the <code>existingPolicyDocument</code> request parameter.
     * </p>
     * 
     * @param checkNoNewAccessRequest
     * @return Result of the CheckNoNewAccess operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws InvalidParameterException
     *         The specified parameter is invalid.
     * @throws UnprocessableEntityException
     *         The specified entity could not be processed.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CheckNoNewAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CheckNoNewAccess"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckNoNewAccessResult checkNoNewAccess(CheckNoNewAccessRequest request) {
        request = beforeClientExecution(request);
        return executeCheckNoNewAccess(request);
    }

    @SdkInternalApi
    final CheckNoNewAccessResult executeCheckNoNewAccess(CheckNoNewAccessRequest checkNoNewAccessRequest) {

        ExecutionContext executionContext = createExecutionContext(checkNoNewAccessRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CheckNoNewAccessRequest> request = null;
        Response<CheckNoNewAccessResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CheckNoNewAccessRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(checkNoNewAccessRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CheckNoNewAccess");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CheckNoNewAccessResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new CheckNoNewAccessResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Creates an access preview that allows you to preview IAM Access Analyzer findings for your resource before
     * deploying resource permissions.
     * </p>
     * 
     * @param createAccessPreviewRequest
     * @return Result of the CreateAccessPreview operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CreateAccessPreview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAccessPreviewResult createAccessPreview(CreateAccessPreviewRequest request) {
        request = beforeClientExecution(request);
        return executeCreateAccessPreview(request);
    }

    @SdkInternalApi
    final CreateAccessPreviewResult executeCreateAccessPreview(CreateAccessPreviewRequest createAccessPreviewRequest) {

        ExecutionContext executionContext = createExecutionContext(createAccessPreviewRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CreateAccessPreviewRequest> request = null;
        Response<CreateAccessPreviewResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CreateAccessPreviewRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(createAccessPreviewRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CreateAccessPreview");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CreateAccessPreviewResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new CreateAccessPreviewResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Creates an analyzer for your account.
     * </p>
     * 
     * @param createAnalyzerRequest
     *        Creates an analyzer.
     * @return Result of the CreateAnalyzer operation returned by the service.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CreateAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAnalyzerResult createAnalyzer(CreateAnalyzerRequest request) {
        request = beforeClientExecution(request);
        return executeCreateAnalyzer(request);
    }

    @SdkInternalApi
    final CreateAnalyzerResult executeCreateAnalyzer(CreateAnalyzerRequest createAnalyzerRequest) {

        ExecutionContext executionContext = createExecutionContext(createAnalyzerRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CreateAnalyzerRequest> request = null;
        Response<CreateAnalyzerResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CreateAnalyzerRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(createAnalyzerRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CreateAnalyzer");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CreateAnalyzerResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new CreateAnalyzerResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Creates an archive rule for the specified analyzer. Archive rules automatically archive new findings that meet
     * the criteria you define when you create the rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * 
     * @param createArchiveRuleRequest
     *        Creates an archive rule.
     * @return Result of the CreateArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.CreateArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateArchiveRuleResult createArchiveRule(CreateArchiveRuleRequest request) {
        request = beforeClientExecution(request);
        return executeCreateArchiveRule(request);
    }

    @SdkInternalApi
    final CreateArchiveRuleResult executeCreateArchiveRule(CreateArchiveRuleRequest createArchiveRuleRequest) {

        ExecutionContext executionContext = createExecutionContext(createArchiveRuleRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CreateArchiveRuleRequest> request = null;
        Response<CreateArchiveRuleResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CreateArchiveRuleRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(createArchiveRuleRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CreateArchiveRule");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CreateArchiveRuleResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new CreateArchiveRuleResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Deletes the specified analyzer. When you delete an analyzer, IAM Access Analyzer is disabled for the account or
     * organization in the current or specific Region. All findings that were generated by the analyzer are deleted. You
     * cannot undo this action.
     * </p>
     * 
     * @param deleteAnalyzerRequest
     *        Deletes an analyzer.
     * @return Result of the DeleteAnalyzer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.DeleteAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAnalyzerResult deleteAnalyzer(DeleteAnalyzerRequest request) {
        request = beforeClientExecution(request);
        return executeDeleteAnalyzer(request);
    }

    @SdkInternalApi
    final DeleteAnalyzerResult executeDeleteAnalyzer(DeleteAnalyzerRequest deleteAnalyzerRequest) {

        ExecutionContext executionContext = createExecutionContext(deleteAnalyzerRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<DeleteAnalyzerRequest> request = null;
        Response<DeleteAnalyzerResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new DeleteAnalyzerRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(deleteAnalyzerRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "DeleteAnalyzer");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<DeleteAnalyzerResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new DeleteAnalyzerResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Deletes the specified archive rule.
     * </p>
     * 
     * @param deleteArchiveRuleRequest
     *        Deletes an archive rule.
     * @return Result of the DeleteArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.DeleteArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteArchiveRuleResult deleteArchiveRule(DeleteArchiveRuleRequest request) {
        request = beforeClientExecution(request);
        return executeDeleteArchiveRule(request);
    }

    @SdkInternalApi
    final DeleteArchiveRuleResult executeDeleteArchiveRule(DeleteArchiveRuleRequest deleteArchiveRuleRequest) {

        ExecutionContext executionContext = createExecutionContext(deleteArchiveRuleRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<DeleteArchiveRuleRequest> request = null;
        Response<DeleteArchiveRuleResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new DeleteArchiveRuleRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(deleteArchiveRuleRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "DeleteArchiveRule");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<DeleteArchiveRuleResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new DeleteArchiveRuleResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about an access preview for the specified analyzer.
     * </p>
     * 
     * @param getAccessPreviewRequest
     * @return Result of the GetAccessPreview operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetAccessPreview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAccessPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAccessPreviewResult getAccessPreview(GetAccessPreviewRequest request) {
        request = beforeClientExecution(request);
        return executeGetAccessPreview(request);
    }

    @SdkInternalApi
    final GetAccessPreviewResult executeGetAccessPreview(GetAccessPreviewRequest getAccessPreviewRequest) {

        ExecutionContext executionContext = createExecutionContext(getAccessPreviewRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetAccessPreviewRequest> request = null;
        Response<GetAccessPreviewResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetAccessPreviewRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getAccessPreviewRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetAccessPreview");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetAccessPreviewResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetAccessPreviewResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about a resource that was analyzed.
     * </p>
     * 
     * @param getAnalyzedResourceRequest
     *        Retrieves an analyzed resource.
     * @return Result of the GetAnalyzedResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetAnalyzedResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAnalyzedResourceResult getAnalyzedResource(GetAnalyzedResourceRequest request) {
        request = beforeClientExecution(request);
        return executeGetAnalyzedResource(request);
    }

    @SdkInternalApi
    final GetAnalyzedResourceResult executeGetAnalyzedResource(GetAnalyzedResourceRequest getAnalyzedResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(getAnalyzedResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetAnalyzedResourceRequest> request = null;
        Response<GetAnalyzedResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetAnalyzedResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getAnalyzedResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetAnalyzedResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetAnalyzedResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetAnalyzedResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified analyzer.
     * </p>
     * 
     * @param getAnalyzerRequest
     *        Retrieves an analyzer.
     * @return Result of the GetAnalyzer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAnalyzerResult getAnalyzer(GetAnalyzerRequest request) {
        request = beforeClientExecution(request);
        return executeGetAnalyzer(request);
    }

    @SdkInternalApi
    final GetAnalyzerResult executeGetAnalyzer(GetAnalyzerRequest getAnalyzerRequest) {

        ExecutionContext executionContext = createExecutionContext(getAnalyzerRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetAnalyzerRequest> request = null;
        Response<GetAnalyzerResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetAnalyzerRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getAnalyzerRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetAnalyzer");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetAnalyzerResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetAnalyzerResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about an archive rule.
     * </p>
     * <p>
     * To learn about filter keys that you can use to create an archive rule, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * 
     * @param getArchiveRuleRequest
     *        Retrieves an archive rule.
     * @return Result of the GetArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetArchiveRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetArchiveRuleResult getArchiveRule(GetArchiveRuleRequest request) {
        request = beforeClientExecution(request);
        return executeGetArchiveRule(request);
    }

    @SdkInternalApi
    final GetArchiveRuleResult executeGetArchiveRule(GetArchiveRuleRequest getArchiveRuleRequest) {

        ExecutionContext executionContext = createExecutionContext(getArchiveRuleRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetArchiveRuleRequest> request = null;
        Response<GetArchiveRuleResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetArchiveRuleRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getArchiveRuleRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetArchiveRule");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetArchiveRuleResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetArchiveRuleResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified finding.
     * </p>
     * 
     * @param getFindingRequest
     *        Retrieves a finding.
     * @return Result of the GetFinding operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetFinding
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetFinding" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFindingResult getFinding(GetFindingRequest request) {
        request = beforeClientExecution(request);
        return executeGetFinding(request);
    }

    @SdkInternalApi
    final GetFindingResult executeGetFinding(GetFindingRequest getFindingRequest) {

        ExecutionContext executionContext = createExecutionContext(getFindingRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetFindingRequest> request = null;
        Response<GetFindingResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetFindingRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getFindingRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetFinding");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetFindingResult>> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                    .withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetFindingResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified finding.
     * </p>
     * 
     * @param getFindingV2Request
     * @return Result of the GetFindingV2 operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetFindingV2
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetFindingV2" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetFindingV2Result getFindingV2(GetFindingV2Request request) {
        request = beforeClientExecution(request);
        return executeGetFindingV2(request);
    }

    @SdkInternalApi
    final GetFindingV2Result executeGetFindingV2(GetFindingV2Request getFindingV2Request) {

        ExecutionContext executionContext = createExecutionContext(getFindingV2Request);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetFindingV2Request> request = null;
        Response<GetFindingV2Result> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetFindingV2RequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getFindingV2Request));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetFindingV2");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetFindingV2Result>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetFindingV2ResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves the policy that was generated using <code>StartPolicyGeneration</code>.
     * </p>
     * 
     * @param getGeneratedPolicyRequest
     * @return Result of the GetGeneratedPolicy operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.GetGeneratedPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetGeneratedPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGeneratedPolicyResult getGeneratedPolicy(GetGeneratedPolicyRequest request) {
        request = beforeClientExecution(request);
        return executeGetGeneratedPolicy(request);
    }

    @SdkInternalApi
    final GetGeneratedPolicyResult executeGetGeneratedPolicy(GetGeneratedPolicyRequest getGeneratedPolicyRequest) {

        ExecutionContext executionContext = createExecutionContext(getGeneratedPolicyRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetGeneratedPolicyRequest> request = null;
        Response<GetGeneratedPolicyResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetGeneratedPolicyRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getGeneratedPolicyRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetGeneratedPolicy");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetGeneratedPolicyResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetGeneratedPolicyResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of access preview findings generated by the specified access preview.
     * </p>
     * 
     * @param listAccessPreviewFindingsRequest
     * @return Result of the ListAccessPreviewFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListAccessPreviewFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviewFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewFindingsResult listAccessPreviewFindings(ListAccessPreviewFindingsRequest request) {
        request = beforeClientExecution(request);
        return executeListAccessPreviewFindings(request);
    }

    @SdkInternalApi
    final ListAccessPreviewFindingsResult executeListAccessPreviewFindings(ListAccessPreviewFindingsRequest listAccessPreviewFindingsRequest) {

        ExecutionContext executionContext = createExecutionContext(listAccessPreviewFindingsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListAccessPreviewFindingsRequest> request = null;
        Response<ListAccessPreviewFindingsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListAccessPreviewFindingsRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listAccessPreviewFindingsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListAccessPreviewFindings");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListAccessPreviewFindingsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListAccessPreviewFindingsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of access previews for the specified analyzer.
     * </p>
     * 
     * @param listAccessPreviewsRequest
     * @return Result of the ListAccessPreviews operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListAccessPreviews
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAccessPreviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccessPreviewsResult listAccessPreviews(ListAccessPreviewsRequest request) {
        request = beforeClientExecution(request);
        return executeListAccessPreviews(request);
    }

    @SdkInternalApi
    final ListAccessPreviewsResult executeListAccessPreviews(ListAccessPreviewsRequest listAccessPreviewsRequest) {

        ExecutionContext executionContext = createExecutionContext(listAccessPreviewsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListAccessPreviewsRequest> request = null;
        Response<ListAccessPreviewsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListAccessPreviewsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listAccessPreviewsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListAccessPreviews");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListAccessPreviewsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListAccessPreviewsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     * 
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return Result of the ListAnalyzedResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListAnalyzedResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnalyzedResourcesResult listAnalyzedResources(ListAnalyzedResourcesRequest request) {
        request = beforeClientExecution(request);
        return executeListAnalyzedResources(request);
    }

    @SdkInternalApi
    final ListAnalyzedResourcesResult executeListAnalyzedResources(ListAnalyzedResourcesRequest listAnalyzedResourcesRequest) {

        ExecutionContext executionContext = createExecutionContext(listAnalyzedResourcesRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListAnalyzedResourcesRequest> request = null;
        Response<ListAnalyzedResourcesResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListAnalyzedResourcesRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listAnalyzedResourcesRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListAnalyzedResources");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListAnalyzedResourcesResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new ListAnalyzedResourcesResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     * 
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return Result of the ListAnalyzers operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListAnalyzers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAnalyzersResult listAnalyzers(ListAnalyzersRequest request) {
        request = beforeClientExecution(request);
        return executeListAnalyzers(request);
    }

    @SdkInternalApi
    final ListAnalyzersResult executeListAnalyzers(ListAnalyzersRequest listAnalyzersRequest) {

        ExecutionContext executionContext = createExecutionContext(listAnalyzersRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListAnalyzersRequest> request = null;
        Response<ListAnalyzersResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListAnalyzersRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listAnalyzersRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListAnalyzers");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListAnalyzersResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListAnalyzersResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     * 
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return Result of the ListArchiveRules operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListArchiveRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListArchiveRulesResult listArchiveRules(ListArchiveRulesRequest request) {
        request = beforeClientExecution(request);
        return executeListArchiveRules(request);
    }

    @SdkInternalApi
    final ListArchiveRulesResult executeListArchiveRules(ListArchiveRulesRequest listArchiveRulesRequest) {

        ExecutionContext executionContext = createExecutionContext(listArchiveRulesRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListArchiveRulesRequest> request = null;
        Response<ListArchiveRulesResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListArchiveRulesRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listArchiveRulesRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListArchiveRules");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListArchiveRulesResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListArchiveRulesResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * 
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return Result of the ListFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFindingsResult listFindings(ListFindingsRequest request) {
        request = beforeClientExecution(request);
        return executeListFindings(request);
    }

    @SdkInternalApi
    final ListFindingsResult executeListFindings(ListFindingsRequest listFindingsRequest) {

        ExecutionContext executionContext = createExecutionContext(listFindingsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListFindingsRequest> request = null;
        Response<ListFindingsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListFindingsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listFindingsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListFindings");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListFindingsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListFindingsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <p>
     * To learn about filter keys that you can use to retrieve a list of findings, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-reference-filter-keys.html">IAM Access
     * Analyzer filter keys</a> in the <b>IAM User Guide</b>.
     * </p>
     * 
     * @param listFindingsV2Request
     * @return Result of the ListFindingsV2 operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListFindingsV2
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindingsV2" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFindingsV2Result listFindingsV2(ListFindingsV2Request request) {
        request = beforeClientExecution(request);
        return executeListFindingsV2(request);
    }

    @SdkInternalApi
    final ListFindingsV2Result executeListFindingsV2(ListFindingsV2Request listFindingsV2Request) {

        ExecutionContext executionContext = createExecutionContext(listFindingsV2Request);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListFindingsV2Request> request = null;
        Response<ListFindingsV2Result> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListFindingsV2RequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listFindingsV2Request));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListFindingsV2");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListFindingsV2Result>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListFindingsV2ResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists all of the policy generations requested in the last seven days.
     * </p>
     * 
     * @param listPolicyGenerationsRequest
     * @return Result of the ListPolicyGenerations operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListPolicyGenerations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListPolicyGenerations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPolicyGenerationsResult listPolicyGenerations(ListPolicyGenerationsRequest request) {
        request = beforeClientExecution(request);
        return executeListPolicyGenerations(request);
    }

    @SdkInternalApi
    final ListPolicyGenerationsResult executeListPolicyGenerations(ListPolicyGenerationsRequest listPolicyGenerationsRequest) {

        ExecutionContext executionContext = createExecutionContext(listPolicyGenerationsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListPolicyGenerationsRequest> request = null;
        Response<ListPolicyGenerationsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListPolicyGenerationsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listPolicyGenerationsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListPolicyGenerations");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListPolicyGenerationsResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new ListPolicyGenerationsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Retrieves a list of tags applied to the specified resource.
     * </p>
     * 
     * @param listTagsForResourceRequest
     *        Retrieves a list of tags applied to the specified resource.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResult listTagsForResource(ListTagsForResourceRequest request) {
        request = beforeClientExecution(request);
        return executeListTagsForResource(request);
    }

    @SdkInternalApi
    final ListTagsForResourceResult executeListTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(listTagsForResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListTagsForResourceRequest> request = null;
        Response<ListTagsForResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListTagsForResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listTagsForResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListTagsForResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListTagsForResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListTagsForResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Starts the policy generation request.
     * </p>
     * 
     * @param startPolicyGenerationRequest
     * @return Result of the StartPolicyGeneration operation returned by the service.
     * @throws ConflictException
     *         A conflict exception error.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ServiceQuotaExceededException
     *         Service quote met error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.StartPolicyGeneration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartPolicyGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartPolicyGenerationResult startPolicyGeneration(StartPolicyGenerationRequest request) {
        request = beforeClientExecution(request);
        return executeStartPolicyGeneration(request);
    }

    @SdkInternalApi
    final StartPolicyGenerationResult executeStartPolicyGeneration(StartPolicyGenerationRequest startPolicyGenerationRequest) {

        ExecutionContext executionContext = createExecutionContext(startPolicyGenerationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<StartPolicyGenerationRequest> request = null;
        Response<StartPolicyGenerationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new StartPolicyGenerationRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(startPolicyGenerationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "StartPolicyGeneration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<StartPolicyGenerationResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new StartPolicyGenerationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Immediately starts a scan of the policies applied to the specified resource.
     * </p>
     * 
     * @param startResourceScanRequest
     *        Starts a scan of the policies applied to the specified resource.
     * @return Result of the StartResourceScan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.StartResourceScan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartResourceScan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartResourceScanResult startResourceScan(StartResourceScanRequest request) {
        request = beforeClientExecution(request);
        return executeStartResourceScan(request);
    }

    @SdkInternalApi
    final StartResourceScanResult executeStartResourceScan(StartResourceScanRequest startResourceScanRequest) {

        ExecutionContext executionContext = createExecutionContext(startResourceScanRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<StartResourceScanRequest> request = null;
        Response<StartResourceScanResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new StartResourceScanRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(startResourceScanRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "StartResourceScan");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<StartResourceScanResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new StartResourceScanResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Adds a tag to the specified resource.
     * </p>
     * 
     * @param tagResourceRequest
     *        Adds a tag to the specified resource.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResult tagResource(TagResourceRequest request) {
        request = beforeClientExecution(request);
        return executeTagResource(request);
    }

    @SdkInternalApi
    final TagResourceResult executeTagResource(TagResourceRequest tagResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(tagResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<TagResourceRequest> request = null;
        Response<TagResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new TagResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(tagResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "TagResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<TagResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new TagResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Removes a tag from the specified resource.
     * </p>
     * 
     * @param untagResourceRequest
     *        Removes a tag from the specified resource.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResult untagResource(UntagResourceRequest request) {
        request = beforeClientExecution(request);
        return executeUntagResource(request);
    }

    @SdkInternalApi
    final UntagResourceResult executeUntagResource(UntagResourceRequest untagResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(untagResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UntagResourceRequest> request = null;
        Response<UntagResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UntagResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(untagResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UntagResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UntagResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new UntagResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Updates the criteria and values for the specified archive rule.
     * </p>
     * 
     * @param updateArchiveRuleRequest
     *        Updates the specified archive rule.
     * @return Result of the UpdateArchiveRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.UpdateArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateArchiveRuleResult updateArchiveRule(UpdateArchiveRuleRequest request) {
        request = beforeClientExecution(request);
        return executeUpdateArchiveRule(request);
    }

    @SdkInternalApi
    final UpdateArchiveRuleResult executeUpdateArchiveRule(UpdateArchiveRuleRequest updateArchiveRuleRequest) {

        ExecutionContext executionContext = createExecutionContext(updateArchiveRuleRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UpdateArchiveRuleRequest> request = null;
        Response<UpdateArchiveRuleResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UpdateArchiveRuleRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(updateArchiveRuleRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UpdateArchiveRule");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UpdateArchiveRuleResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new UpdateArchiveRuleResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Updates the status for the specified findings.
     * </p>
     * 
     * @param updateFindingsRequest
     *        Updates findings with the new values provided in the request.
     * @return Result of the UpdateFindings operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.UpdateFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFindingsResult updateFindings(UpdateFindingsRequest request) {
        request = beforeClientExecution(request);
        return executeUpdateFindings(request);
    }

    @SdkInternalApi
    final UpdateFindingsResult executeUpdateFindings(UpdateFindingsRequest updateFindingsRequest) {

        ExecutionContext executionContext = createExecutionContext(updateFindingsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UpdateFindingsRequest> request = null;
        Response<UpdateFindingsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UpdateFindingsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(updateFindingsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UpdateFindings");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UpdateFindingsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new UpdateFindingsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Requests the validation of a policy and returns a list of findings. The findings help you identify issues and
     * provide actionable recommendations to resolve the issue and enable you to author functional policies that meet
     * security best practices.
     * </p>
     * 
     * @param validatePolicyRequest
     * @return Result of the ValidatePolicy operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws InternalServerException
     *         Internal server error.
     * @throws ThrottlingException
     *         Throttling limit exceeded error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @sample AWSAccessAnalyzer.ValidatePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ValidatePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ValidatePolicyResult validatePolicy(ValidatePolicyRequest request) {
        request = beforeClientExecution(request);
        return executeValidatePolicy(request);
    }

    @SdkInternalApi
    final ValidatePolicyResult executeValidatePolicy(ValidatePolicyRequest validatePolicyRequest) {

        ExecutionContext executionContext = createExecutionContext(validatePolicyRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ValidatePolicyRequest> request = null;
        Response<ValidatePolicyResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ValidatePolicyRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(validatePolicyRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "AccessAnalyzer");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ValidatePolicy");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ValidatePolicyResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ValidatePolicyResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * Returns additional metadata for a previously executed successful, request, typically used for debugging issues
     * where a service isn't acting as expected. This data isn't considered part of the result data returned by an
     * operation, so it's available through this separate, diagnostic interface.
     * <p>
     * Response metadata is only cached for a limited period of time, so if you need to access this extra diagnostic
     * information for an executed request, you should use this method to retrieve it as soon as possible after
     * executing the request.
     *
     * @param request
     *        The originally executed request
     *
     * @return The response metadata for the specified request, or null if none is available.
     */
    public ResponseMetadata getCachedResponseMetadata(AmazonWebServiceRequest request) {
        return client.getResponseMetadataForRequest(request);
    }

    /**
     * Normal invoke with authentication. Credentials are required and may be overriden at the request level.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> invoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext) {

        return invoke(request, responseHandler, executionContext, null, null);
    }

    /**
     * Normal invoke with authentication. Credentials are required and may be overriden at the request level.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> invoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext, URI cachedEndpoint, URI uriFromEndpointTrait) {

        executionContext.setCredentialsProvider(CredentialUtils.getCredentialsProvider(request.getOriginalRequest(), awsCredentialsProvider));

        return doInvoke(request, responseHandler, executionContext, cachedEndpoint, uriFromEndpointTrait);
    }

    /**
     * Invoke with no authentication. Credentials are not required and any credentials set on the client or request will
     * be ignored for this operation.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> anonymousInvoke(Request<Y> request,
            HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler, ExecutionContext executionContext) {

        return doInvoke(request, responseHandler, executionContext, null, null);
    }

    /**
     * Invoke the request using the http client. Assumes credentials (or lack thereof) have been configured in the
     * ExecutionContext beforehand.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> doInvoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext, URI discoveredEndpoint, URI uriFromEndpointTrait) {

        if (discoveredEndpoint != null) {
            request.setEndpoint(discoveredEndpoint);
            request.getOriginalRequest().getRequestClientOptions().appendUserAgent("endpoint-discovery");
        } else if (uriFromEndpointTrait != null) {
            request.setEndpoint(uriFromEndpointTrait);
        } else {
            request.setEndpoint(endpoint);
        }

        request.setTimeOffset(timeOffset);

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());

        return client.execute(request, responseHandler, errorResponseHandler, executionContext);
    }

    @com.amazonaws.annotation.SdkInternalApi
    static com.amazonaws.protocol.json.SdkJsonProtocolFactory getProtocolFactory() {
        return protocolFactory;
    }

    @Override
    public void shutdown() {
        super.shutdown();
    }

}
