/*
 * Copyright 2010-2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.services.autoscaling.model;

import java.io.Serializable;

/**
 * <p>
 * Describes an Auto Scaling group.
 * </p>
 */
public class AutoScalingGroup implements Serializable, Cloneable {

    /**
     * <p>
     * The name of the group.
     * </p>
     */
    private String autoScalingGroupName;
    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group.
     * </p>
     */
    private String autoScalingGroupARN;
    /**
     * <p>
     * The name of the associated launch configuration.
     * </p>
     */
    private String launchConfigurationName;
    /**
     * <p>
     * The minimum size of the group.
     * </p>
     */
    private Integer minSize;
    /**
     * <p>
     * The maximum size of the group.
     * </p>
     */
    private Integer maxSize;
    /**
     * <p>
     * The desired size of the group.
     * </p>
     */
    private Integer desiredCapacity;
    /**
     * <p>
     * The number of seconds after a scaling activity completes before any
     * further scaling activities can start.
     * </p>
     */
    private Integer defaultCooldown;
    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<String> availabilityZones;
    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<String> loadBalancerNames;
    /**
     * <p>
     * The service of interest for the health status check, which can be either
     * <code>EC2</code> for Amazon EC2 or <code>ELB</code> for Elastic Load
     * Balancing.
     * </p>
     */
    private String healthCheckType;
    /**
     * <p>
     * The amount of time that Auto Scaling waits before checking an instance's
     * health status. The grace period begins when an instance comes into
     * service.
     * </p>
     */
    private Integer healthCheckGracePeriod;
    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<Instance> instances;
    /**
     * <p>
     * The date and time the group was created.
     * </p>
     */
    private java.util.Date createdTime;
    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<SuspendedProcess> suspendedProcesses;
    /**
     * <p>
     * The name of the placement group into which you'll launch your instances,
     * if any. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     * >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     * Guide</i>.
     * </p>
     */
    private String placementGroup;
    /**
     * <p>
     * One or more subnet IDs, if applicable, separated by commas.
     * </p>
     * <p>
     * If you specify <code>VPCZoneIdentifier</code> and
     * <code>AvailabilityZones</code>, ensure that the Availability Zones of the
     * subnets match the values for <code>AvailabilityZones</code>.
     * </p>
     */
    private String vPCZoneIdentifier;
    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<EnabledMetric> enabledMetrics;
    /**
     * <p>
     * The current state of the group when <a>DeleteAutoScalingGroup</a> is in
     * progress.
     * </p>
     */
    private String status;
    /**
     * <p>
     * The tags for the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<TagDescription> tags;
    /**
     * <p>
     * The termination policies for the group.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<String> terminationPolicies;

    /**
     * <p>
     * The name of the group.
     * </p>
     * 
     * @param autoScalingGroupName
     *        The name of the group.
     */
    public void setAutoScalingGroupName(String autoScalingGroupName) {
        this.autoScalingGroupName = autoScalingGroupName;
    }

    /**
     * <p>
     * The name of the group.
     * </p>
     * 
     * @return The name of the group.
     */
    public String getAutoScalingGroupName() {
        return this.autoScalingGroupName;
    }

    /**
     * <p>
     * The name of the group.
     * </p>
     * 
     * @param autoScalingGroupName
     *        The name of the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withAutoScalingGroupName(String autoScalingGroupName) {
        setAutoScalingGroupName(autoScalingGroupName);
        return this;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group.
     * </p>
     * 
     * @param autoScalingGroupARN
     *        The Amazon Resource Name (ARN) of the group.
     */
    public void setAutoScalingGroupARN(String autoScalingGroupARN) {
        this.autoScalingGroupARN = autoScalingGroupARN;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the group.
     */
    public String getAutoScalingGroupARN() {
        return this.autoScalingGroupARN;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group.
     * </p>
     * 
     * @param autoScalingGroupARN
     *        The Amazon Resource Name (ARN) of the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withAutoScalingGroupARN(String autoScalingGroupARN) {
        setAutoScalingGroupARN(autoScalingGroupARN);
        return this;
    }

    /**
     * <p>
     * The name of the associated launch configuration.
     * </p>
     * 
     * @param launchConfigurationName
     *        The name of the associated launch configuration.
     */
    public void setLaunchConfigurationName(String launchConfigurationName) {
        this.launchConfigurationName = launchConfigurationName;
    }

    /**
     * <p>
     * The name of the associated launch configuration.
     * </p>
     * 
     * @return The name of the associated launch configuration.
     */
    public String getLaunchConfigurationName() {
        return this.launchConfigurationName;
    }

    /**
     * <p>
     * The name of the associated launch configuration.
     * </p>
     * 
     * @param launchConfigurationName
     *        The name of the associated launch configuration.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withLaunchConfigurationName(
            String launchConfigurationName) {
        setLaunchConfigurationName(launchConfigurationName);
        return this;
    }

    /**
     * <p>
     * The minimum size of the group.
     * </p>
     * 
     * @param minSize
     *        The minimum size of the group.
     */
    public void setMinSize(Integer minSize) {
        this.minSize = minSize;
    }

    /**
     * <p>
     * The minimum size of the group.
     * </p>
     * 
     * @return The minimum size of the group.
     */
    public Integer getMinSize() {
        return this.minSize;
    }

    /**
     * <p>
     * The minimum size of the group.
     * </p>
     * 
     * @param minSize
     *        The minimum size of the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withMinSize(Integer minSize) {
        setMinSize(minSize);
        return this;
    }

    /**
     * <p>
     * The maximum size of the group.
     * </p>
     * 
     * @param maxSize
     *        The maximum size of the group.
     */
    public void setMaxSize(Integer maxSize) {
        this.maxSize = maxSize;
    }

    /**
     * <p>
     * The maximum size of the group.
     * </p>
     * 
     * @return The maximum size of the group.
     */
    public Integer getMaxSize() {
        return this.maxSize;
    }

    /**
     * <p>
     * The maximum size of the group.
     * </p>
     * 
     * @param maxSize
     *        The maximum size of the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withMaxSize(Integer maxSize) {
        setMaxSize(maxSize);
        return this;
    }

    /**
     * <p>
     * The desired size of the group.
     * </p>
     * 
     * @param desiredCapacity
     *        The desired size of the group.
     */
    public void setDesiredCapacity(Integer desiredCapacity) {
        this.desiredCapacity = desiredCapacity;
    }

    /**
     * <p>
     * The desired size of the group.
     * </p>
     * 
     * @return The desired size of the group.
     */
    public Integer getDesiredCapacity() {
        return this.desiredCapacity;
    }

    /**
     * <p>
     * The desired size of the group.
     * </p>
     * 
     * @param desiredCapacity
     *        The desired size of the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withDesiredCapacity(Integer desiredCapacity) {
        setDesiredCapacity(desiredCapacity);
        return this;
    }

    /**
     * <p>
     * The number of seconds after a scaling activity completes before any
     * further scaling activities can start.
     * </p>
     * 
     * @param defaultCooldown
     *        The number of seconds after a scaling activity completes before
     *        any further scaling activities can start.
     */
    public void setDefaultCooldown(Integer defaultCooldown) {
        this.defaultCooldown = defaultCooldown;
    }

    /**
     * <p>
     * The number of seconds after a scaling activity completes before any
     * further scaling activities can start.
     * </p>
     * 
     * @return The number of seconds after a scaling activity completes before
     *         any further scaling activities can start.
     */
    public Integer getDefaultCooldown() {
        return this.defaultCooldown;
    }

    /**
     * <p>
     * The number of seconds after a scaling activity completes before any
     * further scaling activities can start.
     * </p>
     * 
     * @param defaultCooldown
     *        The number of seconds after a scaling activity completes before
     *        any further scaling activities can start.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withDefaultCooldown(Integer defaultCooldown) {
        setDefaultCooldown(defaultCooldown);
        return this;
    }

    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     * 
     * @return One or more Availability Zones for the group.
     */
    public java.util.List<String> getAvailabilityZones() {
        if (availabilityZones == null) {
            availabilityZones = new com.amazonaws.internal.SdkInternalList<String>();
        }
        return availabilityZones;
    }

    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     * 
     * @param availabilityZones
     *        One or more Availability Zones for the group.
     */
    public void setAvailabilityZones(
            java.util.Collection<String> availabilityZones) {
        if (availabilityZones == null) {
            this.availabilityZones = null;
            return;
        }

        this.availabilityZones = new com.amazonaws.internal.SdkInternalList<String>(
                availabilityZones);
    }

    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setAvailabilityZones(java.util.Collection)} or
     * {@link #withAvailabilityZones(java.util.Collection)} if you want to
     * override the existing values.
     * </p>
     * 
     * @param availabilityZones
     *        One or more Availability Zones for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withAvailabilityZones(String... availabilityZones) {
        if (this.availabilityZones == null) {
            setAvailabilityZones(new com.amazonaws.internal.SdkInternalList<String>(
                    availabilityZones.length));
        }
        for (String ele : availabilityZones) {
            this.availabilityZones.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * One or more Availability Zones for the group.
     * </p>
     * 
     * @param availabilityZones
     *        One or more Availability Zones for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withAvailabilityZones(
            java.util.Collection<String> availabilityZones) {
        setAvailabilityZones(availabilityZones);
        return this;
    }

    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     * 
     * @return One or more load balancers associated with the group.
     */
    public java.util.List<String> getLoadBalancerNames() {
        if (loadBalancerNames == null) {
            loadBalancerNames = new com.amazonaws.internal.SdkInternalList<String>();
        }
        return loadBalancerNames;
    }

    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     * 
     * @param loadBalancerNames
     *        One or more load balancers associated with the group.
     */
    public void setLoadBalancerNames(
            java.util.Collection<String> loadBalancerNames) {
        if (loadBalancerNames == null) {
            this.loadBalancerNames = null;
            return;
        }

        this.loadBalancerNames = new com.amazonaws.internal.SdkInternalList<String>(
                loadBalancerNames);
    }

    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setLoadBalancerNames(java.util.Collection)} or
     * {@link #withLoadBalancerNames(java.util.Collection)} if you want to
     * override the existing values.
     * </p>
     * 
     * @param loadBalancerNames
     *        One or more load balancers associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withLoadBalancerNames(String... loadBalancerNames) {
        if (this.loadBalancerNames == null) {
            setLoadBalancerNames(new com.amazonaws.internal.SdkInternalList<String>(
                    loadBalancerNames.length));
        }
        for (String ele : loadBalancerNames) {
            this.loadBalancerNames.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * One or more load balancers associated with the group.
     * </p>
     * 
     * @param loadBalancerNames
     *        One or more load balancers associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withLoadBalancerNames(
            java.util.Collection<String> loadBalancerNames) {
        setLoadBalancerNames(loadBalancerNames);
        return this;
    }

    /**
     * <p>
     * The service of interest for the health status check, which can be either
     * <code>EC2</code> for Amazon EC2 or <code>ELB</code> for Elastic Load
     * Balancing.
     * </p>
     * 
     * @param healthCheckType
     *        The service of interest for the health status check, which can be
     *        either <code>EC2</code> for Amazon EC2 or <code>ELB</code> for
     *        Elastic Load Balancing.
     */
    public void setHealthCheckType(String healthCheckType) {
        this.healthCheckType = healthCheckType;
    }

    /**
     * <p>
     * The service of interest for the health status check, which can be either
     * <code>EC2</code> for Amazon EC2 or <code>ELB</code> for Elastic Load
     * Balancing.
     * </p>
     * 
     * @return The service of interest for the health status check, which can be
     *         either <code>EC2</code> for Amazon EC2 or <code>ELB</code> for
     *         Elastic Load Balancing.
     */
    public String getHealthCheckType() {
        return this.healthCheckType;
    }

    /**
     * <p>
     * The service of interest for the health status check, which can be either
     * <code>EC2</code> for Amazon EC2 or <code>ELB</code> for Elastic Load
     * Balancing.
     * </p>
     * 
     * @param healthCheckType
     *        The service of interest for the health status check, which can be
     *        either <code>EC2</code> for Amazon EC2 or <code>ELB</code> for
     *        Elastic Load Balancing.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withHealthCheckType(String healthCheckType) {
        setHealthCheckType(healthCheckType);
        return this;
    }

    /**
     * <p>
     * The amount of time that Auto Scaling waits before checking an instance's
     * health status. The grace period begins when an instance comes into
     * service.
     * </p>
     * 
     * @param healthCheckGracePeriod
     *        The amount of time that Auto Scaling waits before checking an
     *        instance's health status. The grace period begins when an instance
     *        comes into service.
     */
    public void setHealthCheckGracePeriod(Integer healthCheckGracePeriod) {
        this.healthCheckGracePeriod = healthCheckGracePeriod;
    }

    /**
     * <p>
     * The amount of time that Auto Scaling waits before checking an instance's
     * health status. The grace period begins when an instance comes into
     * service.
     * </p>
     * 
     * @return The amount of time that Auto Scaling waits before checking an
     *         instance's health status. The grace period begins when an
     *         instance comes into service.
     */
    public Integer getHealthCheckGracePeriod() {
        return this.healthCheckGracePeriod;
    }

    /**
     * <p>
     * The amount of time that Auto Scaling waits before checking an instance's
     * health status. The grace period begins when an instance comes into
     * service.
     * </p>
     * 
     * @param healthCheckGracePeriod
     *        The amount of time that Auto Scaling waits before checking an
     *        instance's health status. The grace period begins when an instance
     *        comes into service.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withHealthCheckGracePeriod(
            Integer healthCheckGracePeriod) {
        setHealthCheckGracePeriod(healthCheckGracePeriod);
        return this;
    }

    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     * 
     * @return The EC2 instances associated with the group.
     */
    public java.util.List<Instance> getInstances() {
        if (instances == null) {
            instances = new com.amazonaws.internal.SdkInternalList<Instance>();
        }
        return instances;
    }

    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     * 
     * @param instances
     *        The EC2 instances associated with the group.
     */
    public void setInstances(java.util.Collection<Instance> instances) {
        if (instances == null) {
            this.instances = null;
            return;
        }

        this.instances = new com.amazonaws.internal.SdkInternalList<Instance>(
                instances);
    }

    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setInstances(java.util.Collection)} or
     * {@link #withInstances(java.util.Collection)} if you want to override the
     * existing values.
     * </p>
     * 
     * @param instances
     *        The EC2 instances associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withInstances(Instance... instances) {
        if (this.instances == null) {
            setInstances(new com.amazonaws.internal.SdkInternalList<Instance>(
                    instances.length));
        }
        for (Instance ele : instances) {
            this.instances.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * The EC2 instances associated with the group.
     * </p>
     * 
     * @param instances
     *        The EC2 instances associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withInstances(
            java.util.Collection<Instance> instances) {
        setInstances(instances);
        return this;
    }

    /**
     * <p>
     * The date and time the group was created.
     * </p>
     * 
     * @param createdTime
     *        The date and time the group was created.
     */
    public void setCreatedTime(java.util.Date createdTime) {
        this.createdTime = createdTime;
    }

    /**
     * <p>
     * The date and time the group was created.
     * </p>
     * 
     * @return The date and time the group was created.
     */
    public java.util.Date getCreatedTime() {
        return this.createdTime;
    }

    /**
     * <p>
     * The date and time the group was created.
     * </p>
     * 
     * @param createdTime
     *        The date and time the group was created.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withCreatedTime(java.util.Date createdTime) {
        setCreatedTime(createdTime);
        return this;
    }

    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     * 
     * @return The suspended processes associated with the group.
     */
    public java.util.List<SuspendedProcess> getSuspendedProcesses() {
        if (suspendedProcesses == null) {
            suspendedProcesses = new com.amazonaws.internal.SdkInternalList<SuspendedProcess>();
        }
        return suspendedProcesses;
    }

    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     * 
     * @param suspendedProcesses
     *        The suspended processes associated with the group.
     */
    public void setSuspendedProcesses(
            java.util.Collection<SuspendedProcess> suspendedProcesses) {
        if (suspendedProcesses == null) {
            this.suspendedProcesses = null;
            return;
        }

        this.suspendedProcesses = new com.amazonaws.internal.SdkInternalList<SuspendedProcess>(
                suspendedProcesses);
    }

    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setSuspendedProcesses(java.util.Collection)} or
     * {@link #withSuspendedProcesses(java.util.Collection)} if you want to
     * override the existing values.
     * </p>
     * 
     * @param suspendedProcesses
     *        The suspended processes associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withSuspendedProcesses(
            SuspendedProcess... suspendedProcesses) {
        if (this.suspendedProcesses == null) {
            setSuspendedProcesses(new com.amazonaws.internal.SdkInternalList<SuspendedProcess>(
                    suspendedProcesses.length));
        }
        for (SuspendedProcess ele : suspendedProcesses) {
            this.suspendedProcesses.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * The suspended processes associated with the group.
     * </p>
     * 
     * @param suspendedProcesses
     *        The suspended processes associated with the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withSuspendedProcesses(
            java.util.Collection<SuspendedProcess> suspendedProcesses) {
        setSuspendedProcesses(suspendedProcesses);
        return this;
    }

    /**
     * <p>
     * The name of the placement group into which you'll launch your instances,
     * if any. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     * >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     * Guide</i>.
     * </p>
     * 
     * @param placementGroup
     *        The name of the placement group into which you'll launch your
     *        instances, if any. For more information, see <a href=
     *        "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     *        >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     *        Guide</i>.
     */
    public void setPlacementGroup(String placementGroup) {
        this.placementGroup = placementGroup;
    }

    /**
     * <p>
     * The name of the placement group into which you'll launch your instances,
     * if any. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     * >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     * Guide</i>.
     * </p>
     * 
     * @return The name of the placement group into which you'll launch your
     *         instances, if any. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     *         >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     *         Guide</i>.
     */
    public String getPlacementGroup() {
        return this.placementGroup;
    }

    /**
     * <p>
     * The name of the placement group into which you'll launch your instances,
     * if any. For more information, see <a href=
     * "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     * >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     * Guide</i>.
     * </p>
     * 
     * @param placementGroup
     *        The name of the placement group into which you'll launch your
     *        instances, if any. For more information, see <a href=
     *        "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html"
     *        >Placement Groups</a> in the <i>Amazon Elastic Compute Cloud User
     *        Guide</i>.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withPlacementGroup(String placementGroup) {
        setPlacementGroup(placementGroup);
        return this;
    }

    /**
     * <p>
     * One or more subnet IDs, if applicable, separated by commas.
     * </p>
     * <p>
     * If you specify <code>VPCZoneIdentifier</code> and
     * <code>AvailabilityZones</code>, ensure that the Availability Zones of the
     * subnets match the values for <code>AvailabilityZones</code>.
     * </p>
     * 
     * @param vPCZoneIdentifier
     *        One or more subnet IDs, if applicable, separated by commas.</p>
     *        <p>
     *        If you specify <code>VPCZoneIdentifier</code> and
     *        <code>AvailabilityZones</code>, ensure that the Availability Zones
     *        of the subnets match the values for <code>AvailabilityZones</code>.
     */
    public void setVPCZoneIdentifier(String vPCZoneIdentifier) {
        this.vPCZoneIdentifier = vPCZoneIdentifier;
    }

    /**
     * <p>
     * One or more subnet IDs, if applicable, separated by commas.
     * </p>
     * <p>
     * If you specify <code>VPCZoneIdentifier</code> and
     * <code>AvailabilityZones</code>, ensure that the Availability Zones of the
     * subnets match the values for <code>AvailabilityZones</code>.
     * </p>
     * 
     * @return One or more subnet IDs, if applicable, separated by commas.</p>
     *         <p>
     *         If you specify <code>VPCZoneIdentifier</code> and
     *         <code>AvailabilityZones</code>, ensure that the Availability
     *         Zones of the subnets match the values for
     *         <code>AvailabilityZones</code>.
     */
    public String getVPCZoneIdentifier() {
        return this.vPCZoneIdentifier;
    }

    /**
     * <p>
     * One or more subnet IDs, if applicable, separated by commas.
     * </p>
     * <p>
     * If you specify <code>VPCZoneIdentifier</code> and
     * <code>AvailabilityZones</code>, ensure that the Availability Zones of the
     * subnets match the values for <code>AvailabilityZones</code>.
     * </p>
     * 
     * @param vPCZoneIdentifier
     *        One or more subnet IDs, if applicable, separated by commas.</p>
     *        <p>
     *        If you specify <code>VPCZoneIdentifier</code> and
     *        <code>AvailabilityZones</code>, ensure that the Availability Zones
     *        of the subnets match the values for <code>AvailabilityZones</code>.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withVPCZoneIdentifier(String vPCZoneIdentifier) {
        setVPCZoneIdentifier(vPCZoneIdentifier);
        return this;
    }

    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     * 
     * @return The metrics enabled for the group.
     */
    public java.util.List<EnabledMetric> getEnabledMetrics() {
        if (enabledMetrics == null) {
            enabledMetrics = new com.amazonaws.internal.SdkInternalList<EnabledMetric>();
        }
        return enabledMetrics;
    }

    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     * 
     * @param enabledMetrics
     *        The metrics enabled for the group.
     */
    public void setEnabledMetrics(
            java.util.Collection<EnabledMetric> enabledMetrics) {
        if (enabledMetrics == null) {
            this.enabledMetrics = null;
            return;
        }

        this.enabledMetrics = new com.amazonaws.internal.SdkInternalList<EnabledMetric>(
                enabledMetrics);
    }

    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setEnabledMetrics(java.util.Collection)} or
     * {@link #withEnabledMetrics(java.util.Collection)} if you want to override
     * the existing values.
     * </p>
     * 
     * @param enabledMetrics
     *        The metrics enabled for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withEnabledMetrics(EnabledMetric... enabledMetrics) {
        if (this.enabledMetrics == null) {
            setEnabledMetrics(new com.amazonaws.internal.SdkInternalList<EnabledMetric>(
                    enabledMetrics.length));
        }
        for (EnabledMetric ele : enabledMetrics) {
            this.enabledMetrics.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * The metrics enabled for the group.
     * </p>
     * 
     * @param enabledMetrics
     *        The metrics enabled for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withEnabledMetrics(
            java.util.Collection<EnabledMetric> enabledMetrics) {
        setEnabledMetrics(enabledMetrics);
        return this;
    }

    /**
     * <p>
     * The current state of the group when <a>DeleteAutoScalingGroup</a> is in
     * progress.
     * </p>
     * 
     * @param status
     *        The current state of the group when <a>DeleteAutoScalingGroup</a>
     *        is in progress.
     */
    public void setStatus(String status) {
        this.status = status;
    }

    /**
     * <p>
     * The current state of the group when <a>DeleteAutoScalingGroup</a> is in
     * progress.
     * </p>
     * 
     * @return The current state of the group when <a>DeleteAutoScalingGroup</a>
     *         is in progress.
     */
    public String getStatus() {
        return this.status;
    }

    /**
     * <p>
     * The current state of the group when <a>DeleteAutoScalingGroup</a> is in
     * progress.
     * </p>
     * 
     * @param status
     *        The current state of the group when <a>DeleteAutoScalingGroup</a>
     *        is in progress.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withStatus(String status) {
        setStatus(status);
        return this;
    }

    /**
     * <p>
     * The tags for the group.
     * </p>
     * 
     * @return The tags for the group.
     */
    public java.util.List<TagDescription> getTags() {
        if (tags == null) {
            tags = new com.amazonaws.internal.SdkInternalList<TagDescription>();
        }
        return tags;
    }

    /**
     * <p>
     * The tags for the group.
     * </p>
     * 
     * @param tags
     *        The tags for the group.
     */
    public void setTags(java.util.Collection<TagDescription> tags) {
        if (tags == null) {
            this.tags = null;
            return;
        }

        this.tags = new com.amazonaws.internal.SdkInternalList<TagDescription>(
                tags);
    }

    /**
     * <p>
     * The tags for the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setTags(java.util.Collection)} or
     * {@link #withTags(java.util.Collection)} if you want to override the
     * existing values.
     * </p>
     * 
     * @param tags
     *        The tags for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withTags(TagDescription... tags) {
        if (this.tags == null) {
            setTags(new com.amazonaws.internal.SdkInternalList<TagDescription>(
                    tags.length));
        }
        for (TagDescription ele : tags) {
            this.tags.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * The tags for the group.
     * </p>
     * 
     * @param tags
     *        The tags for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withTags(java.util.Collection<TagDescription> tags) {
        setTags(tags);
        return this;
    }

    /**
     * <p>
     * The termination policies for the group.
     * </p>
     * 
     * @return The termination policies for the group.
     */
    public java.util.List<String> getTerminationPolicies() {
        if (terminationPolicies == null) {
            terminationPolicies = new com.amazonaws.internal.SdkInternalList<String>();
        }
        return terminationPolicies;
    }

    /**
     * <p>
     * The termination policies for the group.
     * </p>
     * 
     * @param terminationPolicies
     *        The termination policies for the group.
     */
    public void setTerminationPolicies(
            java.util.Collection<String> terminationPolicies) {
        if (terminationPolicies == null) {
            this.terminationPolicies = null;
            return;
        }

        this.terminationPolicies = new com.amazonaws.internal.SdkInternalList<String>(
                terminationPolicies);
    }

    /**
     * <p>
     * The termination policies for the group.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if
     * any). Use {@link #setTerminationPolicies(java.util.Collection)} or
     * {@link #withTerminationPolicies(java.util.Collection)} if you want to
     * override the existing values.
     * </p>
     * 
     * @param terminationPolicies
     *        The termination policies for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withTerminationPolicies(
            String... terminationPolicies) {
        if (this.terminationPolicies == null) {
            setTerminationPolicies(new com.amazonaws.internal.SdkInternalList<String>(
                    terminationPolicies.length));
        }
        for (String ele : terminationPolicies) {
            this.terminationPolicies.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * The termination policies for the group.
     * </p>
     * 
     * @param terminationPolicies
     *        The termination policies for the group.
     * @return Returns a reference to this object so that method calls can be
     *         chained together.
     */
    public AutoScalingGroup withTerminationPolicies(
            java.util.Collection<String> terminationPolicies) {
        setTerminationPolicies(terminationPolicies);
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getAutoScalingGroupName() != null)
            sb.append("AutoScalingGroupName: " + getAutoScalingGroupName()
                    + ",");
        if (getAutoScalingGroupARN() != null)
            sb.append("AutoScalingGroupARN: " + getAutoScalingGroupARN() + ",");
        if (getLaunchConfigurationName() != null)
            sb.append("LaunchConfigurationName: "
                    + getLaunchConfigurationName() + ",");
        if (getMinSize() != null)
            sb.append("MinSize: " + getMinSize() + ",");
        if (getMaxSize() != null)
            sb.append("MaxSize: " + getMaxSize() + ",");
        if (getDesiredCapacity() != null)
            sb.append("DesiredCapacity: " + getDesiredCapacity() + ",");
        if (getDefaultCooldown() != null)
            sb.append("DefaultCooldown: " + getDefaultCooldown() + ",");
        if (getAvailabilityZones() != null)
            sb.append("AvailabilityZones: " + getAvailabilityZones() + ",");
        if (getLoadBalancerNames() != null)
            sb.append("LoadBalancerNames: " + getLoadBalancerNames() + ",");
        if (getHealthCheckType() != null)
            sb.append("HealthCheckType: " + getHealthCheckType() + ",");
        if (getHealthCheckGracePeriod() != null)
            sb.append("HealthCheckGracePeriod: " + getHealthCheckGracePeriod()
                    + ",");
        if (getInstances() != null)
            sb.append("Instances: " + getInstances() + ",");
        if (getCreatedTime() != null)
            sb.append("CreatedTime: " + getCreatedTime() + ",");
        if (getSuspendedProcesses() != null)
            sb.append("SuspendedProcesses: " + getSuspendedProcesses() + ",");
        if (getPlacementGroup() != null)
            sb.append("PlacementGroup: " + getPlacementGroup() + ",");
        if (getVPCZoneIdentifier() != null)
            sb.append("VPCZoneIdentifier: " + getVPCZoneIdentifier() + ",");
        if (getEnabledMetrics() != null)
            sb.append("EnabledMetrics: " + getEnabledMetrics() + ",");
        if (getStatus() != null)
            sb.append("Status: " + getStatus() + ",");
        if (getTags() != null)
            sb.append("Tags: " + getTags() + ",");
        if (getTerminationPolicies() != null)
            sb.append("TerminationPolicies: " + getTerminationPolicies());
        sb.append("}");
        return sb.toString();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;

        if (obj instanceof AutoScalingGroup == false)
            return false;
        AutoScalingGroup other = (AutoScalingGroup) obj;
        if (other.getAutoScalingGroupName() == null
                ^ this.getAutoScalingGroupName() == null)
            return false;
        if (other.getAutoScalingGroupName() != null
                && other.getAutoScalingGroupName().equals(
                        this.getAutoScalingGroupName()) == false)
            return false;
        if (other.getAutoScalingGroupARN() == null
                ^ this.getAutoScalingGroupARN() == null)
            return false;
        if (other.getAutoScalingGroupARN() != null
                && other.getAutoScalingGroupARN().equals(
                        this.getAutoScalingGroupARN()) == false)
            return false;
        if (other.getLaunchConfigurationName() == null
                ^ this.getLaunchConfigurationName() == null)
            return false;
        if (other.getLaunchConfigurationName() != null
                && other.getLaunchConfigurationName().equals(
                        this.getLaunchConfigurationName()) == false)
            return false;
        if (other.getMinSize() == null ^ this.getMinSize() == null)
            return false;
        if (other.getMinSize() != null
                && other.getMinSize().equals(this.getMinSize()) == false)
            return false;
        if (other.getMaxSize() == null ^ this.getMaxSize() == null)
            return false;
        if (other.getMaxSize() != null
                && other.getMaxSize().equals(this.getMaxSize()) == false)
            return false;
        if (other.getDesiredCapacity() == null
                ^ this.getDesiredCapacity() == null)
            return false;
        if (other.getDesiredCapacity() != null
                && other.getDesiredCapacity().equals(this.getDesiredCapacity()) == false)
            return false;
        if (other.getDefaultCooldown() == null
                ^ this.getDefaultCooldown() == null)
            return false;
        if (other.getDefaultCooldown() != null
                && other.getDefaultCooldown().equals(this.getDefaultCooldown()) == false)
            return false;
        if (other.getAvailabilityZones() == null
                ^ this.getAvailabilityZones() == null)
            return false;
        if (other.getAvailabilityZones() != null
                && other.getAvailabilityZones().equals(
                        this.getAvailabilityZones()) == false)
            return false;
        if (other.getLoadBalancerNames() == null
                ^ this.getLoadBalancerNames() == null)
            return false;
        if (other.getLoadBalancerNames() != null
                && other.getLoadBalancerNames().equals(
                        this.getLoadBalancerNames()) == false)
            return false;
        if (other.getHealthCheckType() == null
                ^ this.getHealthCheckType() == null)
            return false;
        if (other.getHealthCheckType() != null
                && other.getHealthCheckType().equals(this.getHealthCheckType()) == false)
            return false;
        if (other.getHealthCheckGracePeriod() == null
                ^ this.getHealthCheckGracePeriod() == null)
            return false;
        if (other.getHealthCheckGracePeriod() != null
                && other.getHealthCheckGracePeriod().equals(
                        this.getHealthCheckGracePeriod()) == false)
            return false;
        if (other.getInstances() == null ^ this.getInstances() == null)
            return false;
        if (other.getInstances() != null
                && other.getInstances().equals(this.getInstances()) == false)
            return false;
        if (other.getCreatedTime() == null ^ this.getCreatedTime() == null)
            return false;
        if (other.getCreatedTime() != null
                && other.getCreatedTime().equals(this.getCreatedTime()) == false)
            return false;
        if (other.getSuspendedProcesses() == null
                ^ this.getSuspendedProcesses() == null)
            return false;
        if (other.getSuspendedProcesses() != null
                && other.getSuspendedProcesses().equals(
                        this.getSuspendedProcesses()) == false)
            return false;
        if (other.getPlacementGroup() == null
                ^ this.getPlacementGroup() == null)
            return false;
        if (other.getPlacementGroup() != null
                && other.getPlacementGroup().equals(this.getPlacementGroup()) == false)
            return false;
        if (other.getVPCZoneIdentifier() == null
                ^ this.getVPCZoneIdentifier() == null)
            return false;
        if (other.getVPCZoneIdentifier() != null
                && other.getVPCZoneIdentifier().equals(
                        this.getVPCZoneIdentifier()) == false)
            return false;
        if (other.getEnabledMetrics() == null
                ^ this.getEnabledMetrics() == null)
            return false;
        if (other.getEnabledMetrics() != null
                && other.getEnabledMetrics().equals(this.getEnabledMetrics()) == false)
            return false;
        if (other.getStatus() == null ^ this.getStatus() == null)
            return false;
        if (other.getStatus() != null
                && other.getStatus().equals(this.getStatus()) == false)
            return false;
        if (other.getTags() == null ^ this.getTags() == null)
            return false;
        if (other.getTags() != null
                && other.getTags().equals(this.getTags()) == false)
            return false;
        if (other.getTerminationPolicies() == null
                ^ this.getTerminationPolicies() == null)
            return false;
        if (other.getTerminationPolicies() != null
                && other.getTerminationPolicies().equals(
                        this.getTerminationPolicies()) == false)
            return false;
        return true;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = prime
                * hashCode
                + ((getAutoScalingGroupName() == null) ? 0
                        : getAutoScalingGroupName().hashCode());
        hashCode = prime
                * hashCode
                + ((getAutoScalingGroupARN() == null) ? 0
                        : getAutoScalingGroupARN().hashCode());
        hashCode = prime
                * hashCode
                + ((getLaunchConfigurationName() == null) ? 0
                        : getLaunchConfigurationName().hashCode());
        hashCode = prime * hashCode
                + ((getMinSize() == null) ? 0 : getMinSize().hashCode());
        hashCode = prime * hashCode
                + ((getMaxSize() == null) ? 0 : getMaxSize().hashCode());
        hashCode = prime
                * hashCode
                + ((getDesiredCapacity() == null) ? 0 : getDesiredCapacity()
                        .hashCode());
        hashCode = prime
                * hashCode
                + ((getDefaultCooldown() == null) ? 0 : getDefaultCooldown()
                        .hashCode());
        hashCode = prime
                * hashCode
                + ((getAvailabilityZones() == null) ? 0
                        : getAvailabilityZones().hashCode());
        hashCode = prime
                * hashCode
                + ((getLoadBalancerNames() == null) ? 0
                        : getLoadBalancerNames().hashCode());
        hashCode = prime
                * hashCode
                + ((getHealthCheckType() == null) ? 0 : getHealthCheckType()
                        .hashCode());
        hashCode = prime
                * hashCode
                + ((getHealthCheckGracePeriod() == null) ? 0
                        : getHealthCheckGracePeriod().hashCode());
        hashCode = prime * hashCode
                + ((getInstances() == null) ? 0 : getInstances().hashCode());
        hashCode = prime
                * hashCode
                + ((getCreatedTime() == null) ? 0 : getCreatedTime().hashCode());
        hashCode = prime
                * hashCode
                + ((getSuspendedProcesses() == null) ? 0
                        : getSuspendedProcesses().hashCode());
        hashCode = prime
                * hashCode
                + ((getPlacementGroup() == null) ? 0 : getPlacementGroup()
                        .hashCode());
        hashCode = prime
                * hashCode
                + ((getVPCZoneIdentifier() == null) ? 0
                        : getVPCZoneIdentifier().hashCode());
        hashCode = prime
                * hashCode
                + ((getEnabledMetrics() == null) ? 0 : getEnabledMetrics()
                        .hashCode());
        hashCode = prime * hashCode
                + ((getStatus() == null) ? 0 : getStatus().hashCode());
        hashCode = prime * hashCode
                + ((getTags() == null) ? 0 : getTags().hashCode());
        hashCode = prime
                * hashCode
                + ((getTerminationPolicies() == null) ? 0
                        : getTerminationPolicies().hashCode());
        return hashCode;
    }

    @Override
    public AutoScalingGroup clone() {
        try {
            return (AutoScalingGroup) super.clone();
        } catch (CloneNotSupportedException e) {
            throw new IllegalStateException(
                    "Got a CloneNotSupportedException from Object.clone() "
                            + "even though we're Cloneable!", e);
        }
    }
}