/*
 * Copyright 2010-2016 Amazon.com, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.services.cloudformation.model.transform;

import java.util.Map;
import java.util.Map.Entry;

import javax.xml.stream.events.XMLEvent;

import com.amazonaws.services.cloudformation.model.*;
import com.amazonaws.transform.Unmarshaller;
import com.amazonaws.transform.MapEntry;
import com.amazonaws.transform.StaxUnmarshallerContext;
import com.amazonaws.transform.SimpleTypeStaxUnmarshallers.*;

/**
 * ResourceChange StAX Unmarshaller
 */
public class ResourceChangeStaxUnmarshaller implements
        Unmarshaller<ResourceChange, StaxUnmarshallerContext> {

    public ResourceChange unmarshall(StaxUnmarshallerContext context)
            throws Exception {
        ResourceChange resourceChange = new ResourceChange();
        int originalDepth = context.getCurrentDepth();
        int targetDepth = originalDepth + 1;

        if (context.isStartOfDocument())
            targetDepth += 1;

        while (true) {
            XMLEvent xmlEvent = context.nextEvent();
            if (xmlEvent.isEndDocument())
                return resourceChange;

            if (xmlEvent.isAttribute() || xmlEvent.isStartElement()) {

                if (context.testExpression("Action", targetDepth)) {
                    resourceChange.setAction(StringStaxUnmarshaller
                            .getInstance().unmarshall(context));
                    continue;
                }

                if (context.testExpression("LogicalResourceId", targetDepth)) {
                    resourceChange.setLogicalResourceId(StringStaxUnmarshaller
                            .getInstance().unmarshall(context));
                    continue;
                }

                if (context.testExpression("PhysicalResourceId", targetDepth)) {
                    resourceChange.setPhysicalResourceId(StringStaxUnmarshaller
                            .getInstance().unmarshall(context));
                    continue;
                }

                if (context.testExpression("ResourceType", targetDepth)) {
                    resourceChange.setResourceType(StringStaxUnmarshaller
                            .getInstance().unmarshall(context));
                    continue;
                }

                if (context.testExpression("Replacement", targetDepth)) {
                    resourceChange.setReplacement(StringStaxUnmarshaller
                            .getInstance().unmarshall(context));
                    continue;
                }

                if (context.testExpression("Scope/member", targetDepth)) {
                    resourceChange.getScope().add(
                            StringStaxUnmarshaller.getInstance().unmarshall(
                                    context));
                    continue;
                }

                if (context.testExpression("Details/member", targetDepth)) {
                    resourceChange.getDetails().add(
                            ResourceChangeDetailStaxUnmarshaller.getInstance()
                                    .unmarshall(context));
                    continue;
                }

            } else if (xmlEvent.isEndElement()) {
                if (context.getCurrentDepth() < originalDepth) {
                    return resourceChange;
                }
            }
        }
    }

    private static ResourceChangeStaxUnmarshaller instance;

    public static ResourceChangeStaxUnmarshaller getInstance() {
        if (instance == null)
            instance = new ResourceChangeStaxUnmarshaller();
        return instance;
    }
}
