/*
 * Copyright 2018-2023 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package com.amazonaws.services.applicationdiscovery;

import javax.annotation.Generated;

import com.amazonaws.*;
import com.amazonaws.regions.*;

import com.amazonaws.services.applicationdiscovery.model.*;

/**
 * Interface for accessing AWS Application Discovery Service.
 * <p>
 * <b>Note:</b> Do not directly implement this interface, new methods are added to it regularly. Extend from
 * {@link com.amazonaws.services.applicationdiscovery.AbstractAWSApplicationDiscovery} instead.
 * </p>
 * <p>
 * <fullname>Amazon Web Services Application Discovery Service</fullname>
 * <p>
 * Amazon Web Services Application Discovery Service (Application Discovery Service) helps you plan application
 * migration projects. It automatically identifies servers, virtual machines (VMs), and network dependencies in your
 * on-premises data centers. For more information, see the <a
 * href="http://aws.amazon.com/application-discovery/faqs/">Amazon Web Services Application Discovery Service FAQ</a>.
 * </p>
 * <p>
 * Application Discovery Service offers three ways of performing discovery and collecting data about your on-premises
 * servers:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Agentless discovery</b> using Amazon Web Services Application Discovery Service Agentless Collector (Agentless
 * Collector), which doesn't require you to install an agent on each host.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Agentless Collector gathers server information regardless of the operating systems, which minimizes the time required
 * for initial on-premises infrastructure assessment.
 * </p>
 * </li>
 * <li>
 * <p>
 * Agentless Collector doesn't collect information about network dependencies, only agent-based discovery collects that
 * information.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <ul>
 * <li>
 * <p>
 * <b>Agent-based discovery</b> using the Amazon Web Services Application Discovery Agent (Application Discovery Agent)
 * collects a richer set of data than agentless discovery, which you install on one or more hosts in your data center.
 * </p>
 * <ul>
 * <li>
 * <p>
 * The agent captures infrastructure and application information, including an inventory of running processes, system
 * performance information, resource utilization, and network dependencies.
 * </p>
 * </li>
 * <li>
 * <p>
 * The information collected by agents is secured at rest and in transit to the Application Discovery Service database
 * in the Amazon Web Services cloud. For more information, see <a
 * href="https://docs.aws.amazon.com/application-discovery/latest/userguide/discovery-agent.html">Amazon Web Services
 * Application Discovery Agent</a>.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <ul>
 * <li>
 * <p>
 * <b>Amazon Web Services Partner Network (APN) solutions</b> integrate with Application Discovery Service, enabling you
 * to import details of your on-premises environment directly into Amazon Web Services Migration Hub (Migration Hub)
 * without using Agentless Collector or Application Discovery Agent.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Third-party application discovery tools can query Amazon Web Services Application Discovery Service, and they can
 * write to the Application Discovery Service database using the public API.
 * </p>
 * </li>
 * <li>
 * <p>
 * In this way, you can import data into Migration Hub and view it, so that you can associate applications with servers
 * and track migrations.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * <b>Working With This Guide</b>
 * </p>
 * <p>
 * This API reference provides descriptions, syntax, and usage examples for each of the actions and data types for
 * Application Discovery Service. The topic for each action shows the API request parameters and the response.
 * Alternatively, you can use one of the Amazon Web Services SDKs to access an API that is tailored to the programming
 * language or platform that you're using. For more information, see <a href="http://aws.amazon.com/tools/#SDKs">Amazon
 * Web Services SDKs</a>.
 * </p>
 * <note>
 * <ul>
 * <li>
 * <p>
 * Remember that you must set your Migration Hub home Region before you call any of these APIs.
 * </p>
 * </li>
 * <li>
 * <p>
 * You must make API calls for write actions (create, notify, associate, disassociate, import, or put) while in your
 * home Region, or a <code>HomeRegionNotSetException</code> error is returned.
 * </p>
 * </li>
 * <li>
 * <p>
 * API calls for read actions (list, describe, stop, and delete) are permitted outside of your home Region.
 * </p>
 * </li>
 * <li>
 * <p>
 * Although it is unlikely, the Migration Hub home Region could change. If you call APIs outside the home Region, an
 * <code>InvalidInputException</code> is returned.
 * </p>
 * </li>
 * <li>
 * <p>
 * You must call <code>GetHomeRegion</code> to obtain the latest Migration Hub home Region.
 * </p>
 * </li>
 * </ul>
 * </note>
 * <p>
 * This guide is intended for use with the <a
 * href="https://docs.aws.amazon.com/application-discovery/latest/userguide/">Amazon Web Services Application Discovery
 * Service User Guide</a>.
 * </p>
 * <important>
 * <p>
 * All data is handled according to the <a href="https://aws.amazon.com/privacy/">Amazon Web Services Privacy
 * Policy</a>. You can operate Application Discovery Service offline to inspect collected data before it is shared with
 * the service.
 * </p>
 * </important>
 */
@Generated("com.amazonaws:aws-java-sdk-code-generator")
public interface AWSApplicationDiscovery {

    /**
     * The region metadata service name for computing region endpoints. You can use this value to retrieve metadata
     * (such as supported regions) of the service.
     *
     * @see RegionUtils#getRegionsForService(String)
     */
    String ENDPOINT_PREFIX = "discovery";

    /**
     * Overrides the default endpoint for this client ("https://discovery.us-west-2.amazonaws.com"). Callers can use
     * this method to control which AWS region they want to work with.
     * <p>
     * Callers can pass in just the endpoint (ex: "discovery.us-west-2.amazonaws.com") or a full URL, including the
     * protocol (ex: "https://discovery.us-west-2.amazonaws.com"). If the protocol is not specified here, the default
     * protocol from this client's {@link ClientConfiguration} will be used, which by default is HTTPS.
     * <p>
     * For more information on using AWS regions with the AWS SDK for Java, and a complete list of all available
     * endpoints for all AWS services, see: <a href=
     * "https://docs.aws.amazon.com/sdk-for-java/v1/developer-guide/java-dg-region-selection.html#region-selection-choose-endpoint"
     * > https://docs.aws.amazon.com/sdk-for-java/v1/developer-guide/java-dg-region-selection.html#region-selection-
     * choose-endpoint</a>
     * <p>
     * <b>This method is not threadsafe. An endpoint should be configured when the client is created and before any
     * service requests are made. Changing it afterwards creates inevitable race conditions for any service requests in
     * transit or retrying.</b>
     *
     * @param endpoint
     *        The endpoint (ex: "discovery.us-west-2.amazonaws.com") or a full URL, including the protocol (ex:
     *        "https://discovery.us-west-2.amazonaws.com") of the region specific AWS endpoint this client will
     *        communicate with.
     * @deprecated use {@link AwsClientBuilder#setEndpointConfiguration(AwsClientBuilder.EndpointConfiguration)} for
     *             example:
     *             {@code builder.setEndpointConfiguration(new EndpointConfiguration(endpoint, signingRegion));}
     */
    @Deprecated
    void setEndpoint(String endpoint);

    /**
     * An alternative to {@link AWSApplicationDiscovery#setEndpoint(String)}, sets the regional endpoint for this
     * client's service calls. Callers can use this method to control which AWS region they want to work with.
     * <p>
     * By default, all service endpoints in all regions use the https protocol. To use http instead, specify it in the
     * {@link ClientConfiguration} supplied at construction.
     * <p>
     * <b>This method is not threadsafe. A region should be configured when the client is created and before any service
     * requests are made. Changing it afterwards creates inevitable race conditions for any service requests in transit
     * or retrying.</b>
     *
     * @param region
     *        The region this client will communicate with. See {@link Region#getRegion(com.amazonaws.regions.Regions)}
     *        for accessing a given region. Must not be null and must be a region where the service is available.
     *
     * @see Region#getRegion(com.amazonaws.regions.Regions)
     * @see Region#createClient(Class, com.amazonaws.auth.AWSCredentialsProvider, ClientConfiguration)
     * @see Region#isServiceSupported(String)
     * @deprecated use {@link AwsClientBuilder#setRegion(String)}
     */
    @Deprecated
    void setRegion(Region region);

    /**
     * <p>
     * Associates one or more configuration items with an application.
     * </p>
     * 
     * @param associateConfigurationItemsToApplicationRequest
     * @return Result of the AssociateConfigurationItemsToApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.AssociateConfigurationItemsToApplication
     */
    AssociateConfigurationItemsToApplicationResult associateConfigurationItemsToApplication(
            AssociateConfigurationItemsToApplicationRequest associateConfigurationItemsToApplicationRequest);

    /**
     * <p>
     * Deletes one or more agents or collectors as specified by ID. Deleting an agent or collector does not delete the
     * previously discovered data. To delete the data collected, use <code>StartBatchDeleteConfigurationTask</code>.
     * </p>
     * 
     * @param batchDeleteAgentsRequest
     * @return Result of the BatchDeleteAgents operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @sample AWSApplicationDiscovery.BatchDeleteAgents
     */
    BatchDeleteAgentsResult batchDeleteAgents(BatchDeleteAgentsRequest batchDeleteAgentsRequest);

    /**
     * <p>
     * Deletes one or more import tasks, each identified by their import ID. Each import task has a number of records
     * that can identify servers or applications.
     * </p>
     * <p>
     * Amazon Web Services Application Discovery Service has built-in matching logic that will identify when discovered
     * servers match existing entries that you've previously discovered, the information for the already-existing
     * discovered server is updated. When you delete an import task that contains records that were used to match, the
     * information in those matched records that comes from the deleted records will also be deleted.
     * </p>
     * 
     * @param batchDeleteImportDataRequest
     * @return Result of the BatchDeleteImportData operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.BatchDeleteImportData
     */
    BatchDeleteImportDataResult batchDeleteImportData(BatchDeleteImportDataRequest batchDeleteImportDataRequest);

    /**
     * <p>
     * Creates an application with the given name and description.
     * </p>
     * 
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.CreateApplication
     */
    CreateApplicationResult createApplication(CreateApplicationRequest createApplicationRequest);

    /**
     * <p>
     * Creates one or more tags for configuration items. Tags are metadata that help you categorize IT assets. This API
     * accepts a list of multiple configuration items.
     * </p>
     * <important>
     * <p>
     * Do not store sensitive information (like personal data) in tags.
     * </p>
     * </important>
     * 
     * @param createTagsRequest
     * @return Result of the CreateTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.CreateTags
     */
    CreateTagsResult createTags(CreateTagsRequest createTagsRequest);

    /**
     * <p>
     * Deletes a list of applications and their associations with configuration items.
     * </p>
     * 
     * @param deleteApplicationsRequest
     * @return Result of the DeleteApplications operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DeleteApplications
     */
    DeleteApplicationsResult deleteApplications(DeleteApplicationsRequest deleteApplicationsRequest);

    /**
     * <p>
     * Deletes the association between configuration items and one or more tags. This API accepts a list of multiple
     * configuration items.
     * </p>
     * 
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DeleteTags
     */
    DeleteTagsResult deleteTags(DeleteTagsRequest deleteTagsRequest);

    /**
     * <p>
     * Lists agents or collectors as specified by ID or other filters. All agents/collectors associated with your user
     * can be listed if you call <code>DescribeAgents</code> as is without passing any parameters.
     * </p>
     * 
     * @param describeAgentsRequest
     * @return Result of the DescribeAgents operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeAgents
     */
    DescribeAgentsResult describeAgents(DescribeAgentsRequest describeAgentsRequest);

    /**
     * <p>
     * Takes a unique deletion task identifier as input and returns metadata about a configuration deletion task.
     * </p>
     * 
     * @param describeBatchDeleteConfigurationTaskRequest
     * @return Result of the DescribeBatchDeleteConfigurationTask operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeBatchDeleteConfigurationTask
     */
    DescribeBatchDeleteConfigurationTaskResult describeBatchDeleteConfigurationTask(
            DescribeBatchDeleteConfigurationTaskRequest describeBatchDeleteConfigurationTaskRequest);

    /**
     * <p>
     * Retrieves attributes for a list of configuration item IDs.
     * </p>
     * <note>
     * <p>
     * All of the supplied IDs must be for the same asset type from one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * server
     * </p>
     * </li>
     * <li>
     * <p>
     * application
     * </p>
     * </li>
     * <li>
     * <p>
     * process
     * </p>
     * </li>
     * <li>
     * <p>
     * connection
     * </p>
     * </li>
     * </ul>
     * <p>
     * Output fields are specific to the asset type specified. For example, the output for a <i>server</i> configuration
     * item includes a list of attributes about the server, such as host name, operating system, number of network
     * cards, etc.
     * </p>
     * <p>
     * For a complete list of outputs for each asset type, see <a href=
     * "https://docs.aws.amazon.com/application-discovery/latest/userguide/discovery-api-queries.html#DescribeConfigurations"
     * >Using the DescribeConfigurations Action</a> in the <i>Amazon Web Services Application Discovery Service User
     * Guide</i>.
     * </p>
     * </note>
     * 
     * @param describeConfigurationsRequest
     * @return Result of the DescribeConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeConfigurations
     */
    DescribeConfigurationsResult describeConfigurations(DescribeConfigurationsRequest describeConfigurationsRequest);

    /**
     * <p>
     * Lists exports as specified by ID. All continuous exports associated with your user can be listed if you call
     * <code>DescribeContinuousExports</code> as is without passing any parameters.
     * </p>
     * 
     * @param describeContinuousExportsRequest
     * @return Result of the DescribeContinuousExports operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeContinuousExports
     */
    DescribeContinuousExportsResult describeContinuousExports(DescribeContinuousExportsRequest describeContinuousExportsRequest);

    /**
     * <p>
     * <code>DescribeExportConfigurations</code> is deprecated. Use <a
     * href="https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeExportTasks.html"
     * >DescribeExportTasks</a>, instead.
     * </p>
     * 
     * @param describeExportConfigurationsRequest
     * @return Result of the DescribeExportConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeExportConfigurations
     */
    @Deprecated
    DescribeExportConfigurationsResult describeExportConfigurations(DescribeExportConfigurationsRequest describeExportConfigurationsRequest);

    /**
     * <p>
     * Retrieve status of one or more export tasks. You can retrieve the status of up to 100 export tasks.
     * </p>
     * 
     * @param describeExportTasksRequest
     * @return Result of the DescribeExportTasks operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeExportTasks
     */
    DescribeExportTasksResult describeExportTasks(DescribeExportTasksRequest describeExportTasksRequest);

    /**
     * <p>
     * Returns an array of import tasks for your account, including status information, times, IDs, the Amazon S3 Object
     * URL for the import file, and more.
     * </p>
     * 
     * @param describeImportTasksRequest
     * @return Result of the DescribeImportTasks operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeImportTasks
     */
    DescribeImportTasksResult describeImportTasks(DescribeImportTasksRequest describeImportTasksRequest);

    /**
     * <p>
     * Retrieves a list of configuration items that have tags as specified by the key-value pairs, name and value,
     * passed to the optional parameter <code>filters</code>.
     * </p>
     * <p>
     * There are three valid tag filter names:
     * </p>
     * <ul>
     * <li>
     * <p>
     * tagKey
     * </p>
     * </li>
     * <li>
     * <p>
     * tagValue
     * </p>
     * </li>
     * <li>
     * <p>
     * configurationId
     * </p>
     * </li>
     * </ul>
     * <p>
     * Also, all configuration items associated with your user that have tags can be listed if you call
     * <code>DescribeTags</code> as is without passing any parameters.
     * </p>
     * 
     * @param describeTagsRequest
     * @return Result of the DescribeTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DescribeTags
     */
    DescribeTagsResult describeTags(DescribeTagsRequest describeTagsRequest);

    /**
     * <p>
     * Disassociates one or more configuration items from an application.
     * </p>
     * 
     * @param disassociateConfigurationItemsFromApplicationRequest
     * @return Result of the DisassociateConfigurationItemsFromApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.DisassociateConfigurationItemsFromApplication
     */
    DisassociateConfigurationItemsFromApplicationResult disassociateConfigurationItemsFromApplication(
            DisassociateConfigurationItemsFromApplicationRequest disassociateConfigurationItemsFromApplicationRequest);

    /**
     * <p>
     * Deprecated. Use <code>StartExportTask</code> instead.
     * </p>
     * <p>
     * Exports all discovered configuration data to an Amazon S3 bucket or an application that enables you to view and
     * evaluate the data. Data includes tags and tag associations, processes, connections, servers, and system
     * performance. This API returns an export ID that you can query using the <i>DescribeExportConfigurations</i> API.
     * The system imposes a limit of two configuration exports in six hours.
     * </p>
     * 
     * @param exportConfigurationsRequest
     * @return Result of the ExportConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.ExportConfigurations
     */
    @Deprecated
    ExportConfigurationsResult exportConfigurations(ExportConfigurationsRequest exportConfigurationsRequest);

    /**
     * <p>
     * Retrieves a short summary of discovered assets.
     * </p>
     * <p>
     * This API operation takes no request parameters and is called as is at the command prompt as shown in the example.
     * </p>
     * 
     * @param getDiscoverySummaryRequest
     * @return Result of the GetDiscoverySummary operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.GetDiscoverySummary
     */
    GetDiscoverySummaryResult getDiscoverySummary(GetDiscoverySummaryRequest getDiscoverySummaryRequest);

    /**
     * <p>
     * Retrieves a list of configuration items as specified by the value passed to the required parameter
     * <code>configurationType</code>. Optional filtering may be applied to refine search results.
     * </p>
     * 
     * @param listConfigurationsRequest
     * @return Result of the ListConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.ListConfigurations
     */
    ListConfigurationsResult listConfigurations(ListConfigurationsRequest listConfigurationsRequest);

    /**
     * <p>
     * Retrieves a list of servers that are one network hop away from a specified server.
     * </p>
     * 
     * @param listServerNeighborsRequest
     * @return Result of the ListServerNeighbors operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.ListServerNeighbors
     */
    ListServerNeighborsResult listServerNeighbors(ListServerNeighborsRequest listServerNeighborsRequest);

    /**
     * <p>
     * Takes a list of configurationId as input and starts an asynchronous deletion task to remove the
     * configurationItems. Returns a unique deletion task identifier.
     * </p>
     * 
     * @param startBatchDeleteConfigurationTaskRequest
     * @return Result of the StartBatchDeleteConfigurationTask operation returned by the service.
     * @throws LimitExceededException
     *         The limit of 200 configuration IDs per request has been exceeded.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @sample AWSApplicationDiscovery.StartBatchDeleteConfigurationTask
     */
    StartBatchDeleteConfigurationTaskResult startBatchDeleteConfigurationTask(StartBatchDeleteConfigurationTaskRequest startBatchDeleteConfigurationTaskRequest);

    /**
     * <p>
     * Start the continuous flow of agent's discovered data into Amazon Athena.
     * </p>
     * 
     * @param startContinuousExportRequest
     * @return Result of the StartContinuousExport operation returned by the service.
     * @throws ConflictErrorException
     *         Conflict error.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceInUseException
     *         This issue occurs when the same <code>clientRequestToken</code> is used with the
     *         <code>StartImportTask</code> action, but with different parameters. For example, you use the same request
     *         token but have two different import URLs, you can encounter this issue. If the import tasks are meant to
     *         be different, use a different <code>clientRequestToken</code>, and try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StartContinuousExport
     */
    StartContinuousExportResult startContinuousExport(StartContinuousExportRequest startContinuousExportRequest);

    /**
     * <p>
     * Instructs the specified agents to start collecting data.
     * </p>
     * 
     * @param startDataCollectionByAgentIdsRequest
     * @return Result of the StartDataCollectionByAgentIds operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StartDataCollectionByAgentIds
     */
    StartDataCollectionByAgentIdsResult startDataCollectionByAgentIds(StartDataCollectionByAgentIdsRequest startDataCollectionByAgentIdsRequest);

    /**
     * <p>
     * Begins the export of a discovered data report to an Amazon S3 bucket managed by Amazon Web Services.
     * </p>
     * <note>
     * <p>
     * Exports might provide an estimate of fees and savings based on certain information that you provide. Fee
     * estimates do not include any taxes that might apply. Your actual fees and savings depend on a variety of factors,
     * including your actual usage of Amazon Web Services services, which might vary from the estimates provided in this
     * report.
     * </p>
     * </note>
     * <p>
     * If you do not specify <code>preferences</code> or <code>agentIds</code> in the filter, a summary of all servers,
     * applications, tags, and performance is generated. This data is an aggregation of all server data collected
     * through on-premises tooling, file import, application grouping and applying tags.
     * </p>
     * <p>
     * If you specify <code>agentIds</code> in a filter, the task exports up to 72 hours of detailed data collected by
     * the identified Application Discovery Agent, including network, process, and performance details. A time range for
     * exported agent data may be set by using <code>startTime</code> and <code>endTime</code>. Export of detailed agent
     * data is limited to five concurrently running exports. Export of detailed agent data is limited to two exports per
     * day.
     * </p>
     * <p>
     * If you enable <code>ec2RecommendationsPreferences</code> in <code>preferences</code> , an Amazon EC2 instance
     * matching the characteristics of each server in Application Discovery Service is generated. Changing the
     * attributes of the <code>ec2RecommendationsPreferences</code> changes the criteria of the recommendation.
     * </p>
     * 
     * @param startExportTaskRequest
     * @return Result of the StartExportTask operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StartExportTask
     */
    StartExportTaskResult startExportTask(StartExportTaskRequest startExportTaskRequest);

    /**
     * <p>
     * Starts an import task, which allows you to import details of your on-premises environment directly into Amazon
     * Web Services Migration Hub without having to use the Amazon Web Services Application Discovery Service
     * (Application Discovery Service) tools such as the Amazon Web Services Application Discovery Service Agentless
     * Collector or Application Discovery Agent. This gives you the option to perform migration assessment and planning
     * directly from your imported data, including the ability to group your devices as applications and track their
     * migration status.
     * </p>
     * <p>
     * To start an import request, do this:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Download the specially formatted comma separated value (CSV) import template, which you can find here: <a
     * href="https://s3.us-west-2.amazonaws.com/templates-7cffcf56-bd96-4b1c-b45b-a5b42f282e46/import_template.csv"
     * >https://s3.us-west-2.amazonaws.com/templates-7cffcf56-bd96-4b1c-b45b-a5b42f282e46/import_template.csv</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Fill out the template with your server and application data.
     * </p>
     * </li>
     * <li>
     * <p>
     * Upload your import file to an Amazon S3 bucket, and make a note of it's Object URL. Your import file must be in
     * the CSV format.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the console or the <code>StartImportTask</code> command with the Amazon Web Services CLI or one of the Amazon
     * Web Services SDKs to import the records from your file.
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, including step-by-step procedures, see <a
     * href="https://docs.aws.amazon.com/application-discovery/latest/userguide/discovery-import.html">Migration Hub
     * Import</a> in the <i>Amazon Web Services Application Discovery Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * There are limits to the number of import tasks you can create (and delete) in an Amazon Web Services account. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/application-discovery/latest/userguide/ads_service_limits.html">Amazon Web
     * Services Application Discovery Service Limits</a> in the <i>Amazon Web Services Application Discovery Service
     * User Guide</i>.
     * </p>
     * </note>
     * 
     * @param startImportTaskRequest
     * @return Result of the StartImportTask operation returned by the service.
     * @throws ResourceInUseException
     *         This issue occurs when the same <code>clientRequestToken</code> is used with the
     *         <code>StartImportTask</code> action, but with different parameters. For example, you use the same request
     *         token but have two different import URLs, you can encounter this issue. If the import tasks are meant to
     *         be different, use a different <code>clientRequestToken</code>, and try again.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StartImportTask
     */
    StartImportTaskResult startImportTask(StartImportTaskRequest startImportTaskRequest);

    /**
     * <p>
     * Stop the continuous flow of agent's discovered data into Amazon Athena.
     * </p>
     * 
     * @param stopContinuousExportRequest
     * @return Result of the StopContinuousExport operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws ResourceInUseException
     *         This issue occurs when the same <code>clientRequestToken</code> is used with the
     *         <code>StartImportTask</code> action, but with different parameters. For example, you use the same request
     *         token but have two different import URLs, you can encounter this issue. If the import tasks are meant to
     *         be different, use a different <code>clientRequestToken</code>, and try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StopContinuousExport
     */
    StopContinuousExportResult stopContinuousExport(StopContinuousExportRequest stopContinuousExportRequest);

    /**
     * <p>
     * Instructs the specified agents to stop collecting data.
     * </p>
     * 
     * @param stopDataCollectionByAgentIdsRequest
     * @return Result of the StopDataCollectionByAgentIds operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.StopDataCollectionByAgentIds
     */
    StopDataCollectionByAgentIdsResult stopDataCollectionByAgentIds(StopDataCollectionByAgentIdsRequest stopDataCollectionByAgentIdsRequest);

    /**
     * <p>
     * Updates metadata about an application.
     * </p>
     * 
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The user does not have permission to perform the action. Check the IAM policy associated with this user.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws HomeRegionNotSetException
     *         The home Region is not set. Set the home Region to continue.
     * @sample AWSApplicationDiscovery.UpdateApplication
     */
    UpdateApplicationResult updateApplication(UpdateApplicationRequest updateApplicationRequest);

    /**
     * Shuts down this client object, releasing any resources that might be held open. This is an optional method, and
     * callers are not expected to call it, but can if they want to explicitly release any open resources. Once a client
     * has been shutdown, it should not be used to make any more requests.
     */
    void shutdown();

    /**
     * Returns additional metadata for a previously executed successful request, typically used for debugging issues
     * where a service isn't acting as expected. This data isn't considered part of the result data returned by an
     * operation, so it's available through this separate, diagnostic interface.
     * <p>
     * Response metadata is only cached for a limited period of time, so if you need to access this extra diagnostic
     * information for an executed request, you should use this method to retrieve it as soon as possible after
     * executing a request.
     *
     * @param request
     *        The originally executed request.
     *
     * @return The response metadata for the specified request, or null if none is available.
     */
    ResponseMetadata getCachedResponseMetadata(AmazonWebServiceRequest request);

}
