/*
 * Copyright 2011-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.math.BigInteger;

/**
 * Annotation for marking a property as an optimistic locking version attribute.
 * Applied to the getter method or the class field for the class's version
 * property. If the annotation is applied directly to the class field, the
 * corresponding getter and setter must be declared in the same class.
 * <p>
 * Only nullable, integral numeric types (e.g. Integer, Long) can be used as
 * version properties. On a save() operation, the {@link DynamoDBMapper} will
 * attempt to increment the version property and assert that the service's value
 * matches the client's. New objects will be assigned a version of 1 when saved.
 * <p>
 * Note that for batchWrite, and by extension batchSave and batchDelete, <b>no
 * version checks are performed</b>, as required by the
 * {@link com.amazonaws.services.dynamodbv2.AmazonDynamoDB#batchWriteItem(BatchWriteItemRequest)}
 * API.
 */
@DynamoDB
@DynamoDBAutoGenerated(generator=DynamoDBVersionAttribute.Generator.class)
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.FIELD, ElementType.METHOD})
public @interface DynamoDBVersionAttribute {

    /**
     * Optional parameter when the name of the attribute as stored in DynamoDB
     * should differ from the name used by the getter / setter.
     */
    String attributeName() default "";

    /**
     * Version auto-generator.
     */
    static final class Generator<T> implements DynamoDBAutoGenerator<T> {
        private final Sequence<T> sequence;

        public Generator(final Class<T> targetType, final DynamoDBVersionAttribute annotation) {
            this.sequence = Sequences.of(targetType);
        }

        @Override
        public final DynamoDBAutoGenerateStrategy getGenerateStrategy() {
            return DynamoDBAutoGenerateStrategy.ALWAYS;
        }

        @Override
        public final T generate(final T currentValue) {
            return currentValue == null ? sequence.initial() : sequence.next(currentValue);
        }

        private static class Sequence<T> {
            public T initial() {
                throw new DynamoDBMappingException("unsupported type for version");
            }
            public T next(final T currentValue) {
                throw new DynamoDBMappingException("unsupported type for version");
            }
        }

        private static enum Sequences {
            BIGINTEGER(StandardTypeConverters.STRING_TO_BIGINTEGER, new Sequence<BigInteger>() {
                @Override
                public final BigInteger initial() {
                    return BigInteger.ONE;
                }
                @Override
                public final BigInteger next(final BigInteger currentValue) {
                    return currentValue.add(BigInteger.ONE);
                }
            }),

            BYTE(StandardTypeConverters.STRING_TO_BYTE, new Sequence<Byte>() {
                @Override
                public final Byte initial() {
                    return Byte.valueOf((byte)1);
                }
                @Override
                public final Byte next(final Byte currentValue) {
                    return (byte)((currentValue + 1) % Byte.MAX_VALUE);
                }
            }),

            INTEGER(StandardTypeConverters.STRING_TO_INTEGER, new Sequence<Integer>() {
                @Override
                public final Integer initial() {
                    return Integer.valueOf(1);
                }
                @Override
                public final Integer next(final Integer currentValue) {
                    return currentValue + 1;
                }
            }),

            LONG(StandardTypeConverters.STRING_TO_LONG, new Sequence<Long>() {
                @Override
                public final Long initial() {
                    return Long.valueOf(1L);
                }
                @Override
                public final Long next(final Long currentValue) {
                    return currentValue + 1L;
                }
            }),

            SHORT(StandardTypeConverters.STRING_TO_SHORT, new Sequence<Short>() {
                @Override
                public final Short initial() {
                    return Short.valueOf((short)1);
                }
                @Override
                public final Short next(final Short currentValue) {
                    return (short)(currentValue + 1);
                }
            });

            private final StandardTypeConverters standard;
            private final Sequence<?> sequence;

            private Sequences(final StandardTypeConverters standard, final Sequence<?> sequence) {
                this.standard = standard;
                this.sequence = sequence;
            }

            private static final <T> Sequence<T> of(final Class<T> targetType) {
                for (final Sequences s : Sequences.values()) {
                    if (s.standard.isAssignableFrom(String.class, targetType, true)) {
                        return (Sequence<T>)s.sequence;
                    }
                }
                return new Sequence<T>(); //<- for backwards compatibility
            }
        }
    }

}
