/*
 * Copyright 2016-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation to assign a default value on creation if value is null.
 *
 * A minimal example using getter annotations,
 * <pre class="brush: java">
 * &#064;DynamoDBTable(tableName=&quot;TestTable&quot;)
 * public class TestClass {
 *     private String key;
 *     private String status;
 *
 *     &#064;DynamoDBHashKey
 *     public String getKey() { return key; }
 *     public void setKey(String key) { this.key = key; }
 *
 *     &#064;DynamoDBAutoGeneratedDefault(&quot;OPEN&quot;)
 *     public Sttring getStatus() { return status; }
 *     public void setStatus(String status) { this.status = status; }
 * }
 * </pre>
 *
 * Please note,
 *
 * Only compatible with properties of standard types: {@link String},
 * {@link java.math.BigDecimal}, {@link java.math.BigInteger}, {@link Boolean},
 * {@link Byte}, {@link java.util.Calendar}, {@link Character},
 * {@link java.util.Date}, {@link Double}, {@link Enum}, {@link Float},
 * {@link Integer}, {@link Long}, {@link Short}, and {@link java.util.UUID}.
 */
@DynamoDBAutoGenerated(generator=DynamoDBAutoGeneratedDefault.Generator.class)
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.FIELD, ElementType.METHOD})
public @interface DynamoDBAutoGeneratedDefault {

    /**
     * The default value.
     */
    String value();

    /**
     * Default generator.
     */
    static final class Generator<T> implements DynamoDBAutoGenerator<T> {
        private final DynamoDBTypeConverter<String,T> converter;
        private final String defaultValue;

        public Generator(final Class<T> targetType, final DynamoDBAutoGeneratedDefault annotation) {
            this.converter = StandardTypeConverters.Scalar.STRING.getConverter(targetType);
            this.defaultValue = annotation.value();
        }

        @Override
        public final DynamoDBAutoGenerateStrategy getGenerateStrategy() {
            return DynamoDBAutoGenerateStrategy.CREATE;
        }

        @Override
        public final T generate(final T currentValue) {
            return converter.unconvert(defaultValue);
        }
    }

}
