/*
 * Copyright 2016-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Date;
import java.util.TimeZone;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * Annotation to format a timestamp object using Java's standard date and time
 * patterns.
 *
 * A minimal example using getter annotations,
 * <pre class="brush: java">
 * &#064;DynamoDBTable(tableName=&quot;TestTable&quot;)
 * public class TestClass {
 *     private String key;
 *     private Date createdDate;
 *
 *     &#064;DynamoDBHashKey
 *     public String getKey() { return key; }
 *     public void setKey(String key) { this.key = key; }
 *
 *     &#064;DynamoDBTypeConvertedTimestamp(pattern=&quot;yyyyMMddHHmmssSSS&quot;, timeZone=&quot;UTC&quot;)
 *     public Date getCreatedDate() { return createdDate; }
 *     public void setCreatedDate(Date createdDate) { this.createdDate = createdDate; }
 * }
 * </pre>
 *
 * Please note,
 *
 * Only {@link java.util.Calendar}, {@link java.util.Date} and {@link Long} are
 * supported for now. Primitives such as {@code long} are not supported since
 * the unset (or null) state can't be detected.
 *
 * This annotation is also compatible with {@link DynamoDBAutoGeneratedTimestamp}.
 *
 * @see com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBAutoGeneratedTimestamp
 * @see com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBTypeConverted
 * @see java.text.SimpleDateFormat
 * @see java.util.TimeZone
 */
@DynamoDBTypeConverted(converter=DynamoDBTypeConvertedTimestamp.Converter.class)
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.TYPE, ElementType.FIELD, ElementType.METHOD})
public @interface DynamoDBTypeConvertedTimestamp {

    /**
     * The time zone.
     * @see java.util.TimeZone
     */
    String timeZone() default "UTC";

    /**
     * The pattern format.
     * @see java.text.SimpleDateFormat
     */
    String pattern();

    /**
     * Timestamp format converter.
     */
    static final class Converter<T> implements DynamoDBTypeConverter<String,T> {
        private final DynamoDBTypeConverter<Date,T> converter;
        private final DateTimeFormatter formatter;

        public Converter(final Class<T> targetType, final DynamoDBTypeConvertedTimestamp annotation) {
            this.formatter = DateTimeFormat.forPattern(annotation.pattern()).withZone(DateTimeZone.forTimeZone(TimeZone.getTimeZone(annotation.timeZone())));
            this.converter = StandardTypeConverters.Scalar.DATE.getConverter(targetType);
        }

        @Override
        public final String convert(final T object) {
            return formatter.print(new DateTime(converter.convert(object)));
        }

        @Override
        public final T unconvert(final String object) {
            return converter.unconvert(formatter.parseDateTime(object).toDate());
        }
    }

}
