/*
 * Copyright 2011-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import com.amazonaws.services.dynamodbv2.datamodeling.StandardTypeConverters.Scalar;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.math.BigInteger;
import java.util.Arrays;

/**
 * Annotation for marking a property as an optimistic locking version attribute.
 * Applied to the getter method or the class field for the class's version
 * property. If the annotation is applied directly to the class field, the
 * corresponding getter and setter must be declared in the same class.
 * <p>
 * Only nullable, integral numeric types (e.g. Integer, Long) can be used as
 * version properties. On a save() operation, the {@link DynamoDBMapper} will
 * attempt to increment the version property and assert that the service's value
 * matches the client's. New objects will be assigned a version of 1 when saved.
 * <p>
 * Note that for batchWrite, and by extension batchSave and batchDelete, <b>no
 * version checks are performed</b>, as required by the
 * {@link com.amazonaws.services.dynamodbv2.AmazonDynamoDB#batchWriteItem(BatchWriteItemRequest)}
 * API.
 */
@DynamoDB
@DynamoDBAutoGenerated(generator=DynamoDBVersionAttribute.Generator.class)
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.FIELD, ElementType.METHOD})
public @interface DynamoDBVersionAttribute {

    /**
     * Optional parameter when the name of the attribute as stored in DynamoDB
     * should differ from the name used by the getter / setter.
     */
    String attributeName() default "";

    /**
     * Version auto-generator.
     */
    static final class Generator<T> implements DynamoDBAutoGenerator<T> {
        private final Sequence<T> sequence;

        public Generator(final Class<T> targetType, final DynamoDBVersionAttribute annotation) {
            this.sequence = Sequences.of(targetType);
        }

        @Override
        public final DynamoDBAutoGenerateStrategy getGenerateStrategy() {
            return DynamoDBAutoGenerateStrategy.ALWAYS;
        }

        @Override
        public final T generate(final T currentValue) {
            return sequence.next(currentValue);
        }

        static interface Sequence<T> {
            public T next(final T o);
        }

        private static enum Sequences {
            BIG_INTEGER(Scalar.BIG_INTEGER, new Sequence<BigInteger>() {
                public final BigInteger next(final BigInteger o) {
                    return o == null ? BigInteger.ONE : o.add(BigInteger.ONE);
                }
            }),

            BYTE(Scalar.BYTE, new Sequence<Byte>() {
                public final Byte next(final Byte o) {
                    return o == null ? Byte.valueOf((byte)1) : (byte)((o + 1) % Byte.MAX_VALUE);
                }
            }),

            INTEGER(Scalar.INTEGER, new Sequence<Integer>() {
                public final Integer next(final Integer o) {
                    return o == null ? Integer.valueOf(1) : o + 1;
                }
            }),

            LONG(Scalar.LONG, new Sequence<Long>() {
                public final Long next(final Long o) {
                    return o == null ? Long.valueOf(1L) : o + 1L;
                }
            }),

            SHORT(Scalar.SHORT, new Sequence<Short>() {
                public final Short next(final Short o) {
                    return o == null ? Short.valueOf((short)1) : (short)(o + 1);
                }
            });

            private final Sequence<?> sequence;
            private final Scalar scalar;

            private Sequences(final Scalar scalar, final Sequence<?> sequence) {
                this.sequence = sequence;
                this.scalar = scalar;
            }

            private static final <T> Sequence<T> of(final Class<T> targetType) {
                final Scalar target = Scalar.of(targetType);
                for (final Sequences s : Sequences.values()) {
                    if (s.scalar == target) {
                        return (Sequence<T>)s.sequence;
                    }
                }
                return new Sequence<T>() { //<- for backwards compatibility
                    public T next(final T o) {
                        throw new DynamoDBMappingException("type [" + targetType + "] is not supported" +
                            "; only " + Arrays.toString(Sequences.values()) + " allowed");
                    }
                };
            }
        }
    }

}
