/*
 * Copyright 2016-2023 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import com.amazonaws.annotation.SdkInternalApi;

/**
 * Generator interface for auto-generating attribute values.
 *
 * <p>Auto-generation may be controlled by {@link DynamoDBAutoGenerateStrategy},

 * <p>{@link DynamoDBAutoGenerateStrategy#CREATE}, instructs to generate when
 * creating the item. The mapper, determines an item is new, or overwriting,
 * if it's current value is {@code null}. There is a limitiation when performing
 * partial updates using either,
 * {@link DynamoDBMapperConfig.SaveBehavior#UPDATE_SKIP_NULL_ATTRIBUTES}, or
 * {@link DynamoDBMapperConfig.SaveBehavior#APPEND_SET}. A new value will only
 * be generated if the mapper is also generating the key.</p>
 *
 * <p>{@link DynamoDBAutoGenerateStrategy#ALWAYS}, instructs to always generate
 * a new value, applied on any save or batch write operation.</li>
 *
 * <p>May be used in combination with {@link DynamoDBAutoGenerated}.</p>
 *
 * @param <T> The object's field/property value type.
 *
 * @see com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBAutoGenerated
 */
public interface DynamoDBAutoGenerator<T extends Object> {

    /**
     * Gets the auto-generate strategy.
     */
    public DynamoDBAutoGenerateStrategy getGenerateStrategy();

    /**
     * Generates a new value given the current value (or null) if applicable.
     */
    public T generate(T currentValue);

    /**
     * A generator which holds the {@link DynamoDBAutoGenerateStrategy}.
     */
    @SdkInternalApi
    static abstract class AbstractGenerator<T> implements DynamoDBAutoGenerator<T> {
        private final DynamoDBAutoGenerateStrategy strategy;

        protected AbstractGenerator(DynamoDBAutoGenerateStrategy strategy) {
            this.strategy = strategy;
        }

        @Override
        public DynamoDBAutoGenerateStrategy getGenerateStrategy() {
            return this.strategy;
        }
    }

}
