/*
 * Copyright 2020-2025 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package com.amazonaws.services.logs.model;

import java.io.Serializable;
import javax.annotation.Generated;
import com.amazonaws.protocol.StructuredPojo;
import com.amazonaws.protocol.ProtocolMarshaller;

/**
 * <p>
 * This structure represents one anomaly that has been found by a logs anomaly detector.
 * </p>
 * <p>
 * For more information about patterns and anomalies, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogAnomalyDetector.html"
 * >CreateLogAnomalyDetector</a>.
 * </p>
 * 
 * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/Anomaly" target="_top">AWS API Documentation</a>
 */
@Generated("com.amazonaws:aws-java-sdk-code-generator")
public class Anomaly implements Serializable, Cloneable, StructuredPojo {

    /**
     * <p>
     * The unique ID that CloudWatch Logs assigned to this anomaly.
     * </p>
     */
    private String anomalyId;
    /**
     * <p>
     * The ID of the pattern used to help identify this anomaly.
     * </p>
     */
    private String patternId;
    /**
     * <p>
     * The ARN of the anomaly detector that identified this anomaly.
     * </p>
     */
    private String anomalyDetectorArn;
    /**
     * <p>
     * The pattern used to help identify this anomaly, in string format.
     * </p>
     */
    private String patternString;
    /**
     * <p>
     * The pattern used to help identify this anomaly, in regular expression format.
     * </p>
     */
    private String patternRegex;
    /**
     * <p>
     * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log severity
     * labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the baseline. Possible
     * values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * </p>
     */
    private String priority;
    /**
     * <p>
     * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is the
     * number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     */
    private Long firstSeen;
    /**
     * <p>
     * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time, which
     * is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     */
    private Long lastSeen;
    /**
     * <p>
     * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * </p>
     */
    private String description;
    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     */
    private Boolean active;
    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     */
    private String state;
    /**
     * <p>
     * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
     * <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalMap<String, Long> histogram;
    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<LogEvent> logSamples;
    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<PatternToken> patternTokens;
    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     */
    private com.amazonaws.internal.SdkInternalList<String> logGroupArnList;
    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     */
    private Boolean suppressed;
    /**
     * <p>
     * If the anomaly is suppressed, this indicates when it was suppressed.
     * </p>
     */
    private Long suppressedDate;
    /**
     * <p>
     * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>, the
     * anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * </p>
     */
    private Long suppressedUntil;
    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     */
    private Boolean isPatternLevelSuppression;

    /**
     * <p>
     * The unique ID that CloudWatch Logs assigned to this anomaly.
     * </p>
     * 
     * @param anomalyId
     *        The unique ID that CloudWatch Logs assigned to this anomaly.
     */

    public void setAnomalyId(String anomalyId) {
        this.anomalyId = anomalyId;
    }

    /**
     * <p>
     * The unique ID that CloudWatch Logs assigned to this anomaly.
     * </p>
     * 
     * @return The unique ID that CloudWatch Logs assigned to this anomaly.
     */

    public String getAnomalyId() {
        return this.anomalyId;
    }

    /**
     * <p>
     * The unique ID that CloudWatch Logs assigned to this anomaly.
     * </p>
     * 
     * @param anomalyId
     *        The unique ID that CloudWatch Logs assigned to this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withAnomalyId(String anomalyId) {
        setAnomalyId(anomalyId);
        return this;
    }

    /**
     * <p>
     * The ID of the pattern used to help identify this anomaly.
     * </p>
     * 
     * @param patternId
     *        The ID of the pattern used to help identify this anomaly.
     */

    public void setPatternId(String patternId) {
        this.patternId = patternId;
    }

    /**
     * <p>
     * The ID of the pattern used to help identify this anomaly.
     * </p>
     * 
     * @return The ID of the pattern used to help identify this anomaly.
     */

    public String getPatternId() {
        return this.patternId;
    }

    /**
     * <p>
     * The ID of the pattern used to help identify this anomaly.
     * </p>
     * 
     * @param patternId
     *        The ID of the pattern used to help identify this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPatternId(String patternId) {
        setPatternId(patternId);
        return this;
    }

    /**
     * <p>
     * The ARN of the anomaly detector that identified this anomaly.
     * </p>
     * 
     * @param anomalyDetectorArn
     *        The ARN of the anomaly detector that identified this anomaly.
     */

    public void setAnomalyDetectorArn(String anomalyDetectorArn) {
        this.anomalyDetectorArn = anomalyDetectorArn;
    }

    /**
     * <p>
     * The ARN of the anomaly detector that identified this anomaly.
     * </p>
     * 
     * @return The ARN of the anomaly detector that identified this anomaly.
     */

    public String getAnomalyDetectorArn() {
        return this.anomalyDetectorArn;
    }

    /**
     * <p>
     * The ARN of the anomaly detector that identified this anomaly.
     * </p>
     * 
     * @param anomalyDetectorArn
     *        The ARN of the anomaly detector that identified this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withAnomalyDetectorArn(String anomalyDetectorArn) {
        setAnomalyDetectorArn(anomalyDetectorArn);
        return this;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in string format.
     * </p>
     * 
     * @param patternString
     *        The pattern used to help identify this anomaly, in string format.
     */

    public void setPatternString(String patternString) {
        this.patternString = patternString;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in string format.
     * </p>
     * 
     * @return The pattern used to help identify this anomaly, in string format.
     */

    public String getPatternString() {
        return this.patternString;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in string format.
     * </p>
     * 
     * @param patternString
     *        The pattern used to help identify this anomaly, in string format.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPatternString(String patternString) {
        setPatternString(patternString);
        return this;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in regular expression format.
     * </p>
     * 
     * @param patternRegex
     *        The pattern used to help identify this anomaly, in regular expression format.
     */

    public void setPatternRegex(String patternRegex) {
        this.patternRegex = patternRegex;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in regular expression format.
     * </p>
     * 
     * @return The pattern used to help identify this anomaly, in regular expression format.
     */

    public String getPatternRegex() {
        return this.patternRegex;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in regular expression format.
     * </p>
     * 
     * @param patternRegex
     *        The pattern used to help identify this anomaly, in regular expression format.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPatternRegex(String patternRegex) {
        setPatternRegex(patternRegex);
        return this;
    }

    /**
     * <p>
     * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log severity
     * labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the baseline. Possible
     * values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * </p>
     * 
     * @param priority
     *        The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log
     *        severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the
     *        baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     */

    public void setPriority(String priority) {
        this.priority = priority;
    }

    /**
     * <p>
     * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log severity
     * labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the baseline. Possible
     * values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * </p>
     * 
     * @return The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log
     *         severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the
     *         baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     */

    public String getPriority() {
        return this.priority;
    }

    /**
     * <p>
     * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log severity
     * labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the baseline. Possible
     * values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * </p>
     * 
     * @param priority
     *        The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log
     *        severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the
     *        baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPriority(String priority) {
        setPriority(priority);
        return this;
    }

    /**
     * <p>
     * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is the
     * number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param firstSeen
     *        The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which
     *        is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public void setFirstSeen(Long firstSeen) {
        this.firstSeen = firstSeen;
    }

    /**
     * <p>
     * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is the
     * number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @return The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which
     *         is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public Long getFirstSeen() {
        return this.firstSeen;
    }

    /**
     * <p>
     * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is the
     * number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param firstSeen
     *        The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which
     *        is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withFirstSeen(Long firstSeen) {
        setFirstSeen(firstSeen);
        return this;
    }

    /**
     * <p>
     * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time, which
     * is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param lastSeen
     *        The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time,
     *        which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public void setLastSeen(Long lastSeen) {
        this.lastSeen = lastSeen;
    }

    /**
     * <p>
     * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time, which
     * is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @return The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch
     *         time, which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public Long getLastSeen() {
        return this.lastSeen;
    }

    /**
     * <p>
     * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time, which
     * is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param lastSeen
     *        The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time,
     *        which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withLastSeen(Long lastSeen) {
        setLastSeen(lastSeen);
        return this;
    }

    /**
     * <p>
     * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * </p>
     * 
     * @param description
     *        A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     */

    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * <p>
     * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * </p>
     * 
     * @return A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     */

    public String getDescription() {
        return this.description;
    }

    /**
     * <p>
     * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * </p>
     * 
     * @param description
     *        A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withDescription(String description) {
        setDescription(description);
        return this;
    }

    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     * 
     * @param active
     *        Specifies whether this anomaly is still ongoing.
     */

    public void setActive(Boolean active) {
        this.active = active;
    }

    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     * 
     * @return Specifies whether this anomaly is still ongoing.
     */

    public Boolean getActive() {
        return this.active;
    }

    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     * 
     * @param active
     *        Specifies whether this anomaly is still ongoing.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withActive(Boolean active) {
        setActive(active);
        return this;
    }

    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     * 
     * @return Specifies whether this anomaly is still ongoing.
     */

    public Boolean isActive() {
        return this.active;
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * 
     * @param state
     *        Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *        <code>Active</code>. If you have suppressed this anomaly by using the <a
     *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *        >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *        be normal, the value is <code>Baseline</code>.
     * @see State
     */

    public void setState(String state) {
        this.state = state;
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * 
     * @return Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *         <code>Active</code>. If you have suppressed this anomaly by using the <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *         be normal, the value is <code>Baseline</code>.
     * @see State
     */

    public String getState() {
        return this.state;
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * 
     * @param state
     *        Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *        <code>Active</code>. If you have suppressed this anomaly by using the <a
     *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *        >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *        be normal, the value is <code>Baseline</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     * @see State
     */

    public Anomaly withState(String state) {
        setState(state);
        return this;
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * 
     * @param state
     *        Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *        <code>Active</code>. If you have suppressed this anomaly by using the <a
     *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *        >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *        be normal, the value is <code>Baseline</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     * @see State
     */

    public Anomaly withState(State state) {
        this.state = state.toString();
        return this;
    }

    /**
     * <p>
     * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
     * <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @return A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that
     *         were detected at each of those runs. The times are specified in epoch time, which is the number of
     *         seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public java.util.Map<String, Long> getHistogram() {
        if (histogram == null) {
            histogram = new com.amazonaws.internal.SdkInternalMap<String, Long>();
        }
        return histogram;
    }

    /**
     * <p>
     * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
     * <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param histogram
     *        A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     *        detected at each of those runs. The times are specified in epoch time, which is the number of seconds
     *        since <code>January 1, 1970, 00:00:00 UTC</code>.
     */

    public void setHistogram(java.util.Map<String, Long> histogram) {
        this.histogram = histogram == null ? null : new com.amazonaws.internal.SdkInternalMap<String, Long>(histogram);
    }

    /**
     * <p>
     * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
     * <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @param histogram
     *        A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     *        detected at each of those runs. The times are specified in epoch time, which is the number of seconds
     *        since <code>January 1, 1970, 00:00:00 UTC</code>.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withHistogram(java.util.Map<String, Long> histogram) {
        setHistogram(histogram);
        return this;
    }

    /**
     * Add a single Histogram entry
     *
     * @see Anomaly#withHistogram
     * @returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly addHistogramEntry(String key, Long value) {
        if (null == this.histogram) {
            this.histogram = new com.amazonaws.internal.SdkInternalMap<String, Long>();
        }
        if (this.histogram.containsKey(key))
            throw new IllegalArgumentException("Duplicated keys (" + key.toString() + ") are provided.");
        this.histogram.put(key, value);
        return this;
    }

    /**
     * Removes all the entries added into Histogram.
     *
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly clearHistogramEntries() {
        this.histogram = null;
        return this;
    }

    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     * 
     * @return An array of sample log event messages that are considered to be part of this anomaly.
     */

    public java.util.List<LogEvent> getLogSamples() {
        if (logSamples == null) {
            logSamples = new com.amazonaws.internal.SdkInternalList<LogEvent>();
        }
        return logSamples;
    }

    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     * 
     * @param logSamples
     *        An array of sample log event messages that are considered to be part of this anomaly.
     */

    public void setLogSamples(java.util.Collection<LogEvent> logSamples) {
        if (logSamples == null) {
            this.logSamples = null;
            return;
        }

        this.logSamples = new com.amazonaws.internal.SdkInternalList<LogEvent>(logSamples);
    }

    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
     * {@link #setLogSamples(java.util.Collection)} or {@link #withLogSamples(java.util.Collection)} if you want to
     * override the existing values.
     * </p>
     * 
     * @param logSamples
     *        An array of sample log event messages that are considered to be part of this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withLogSamples(LogEvent... logSamples) {
        if (this.logSamples == null) {
            setLogSamples(new com.amazonaws.internal.SdkInternalList<LogEvent>(logSamples.length));
        }
        for (LogEvent ele : logSamples) {
            this.logSamples.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     * 
     * @param logSamples
     *        An array of sample log event messages that are considered to be part of this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withLogSamples(java.util.Collection<LogEvent> logSamples) {
        setLogSamples(logSamples);
        return this;
    }

    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     * 
     * @return An array of structures where each structure contains information about one token that makes up the
     *         pattern.
     */

    public java.util.List<PatternToken> getPatternTokens() {
        if (patternTokens == null) {
            patternTokens = new com.amazonaws.internal.SdkInternalList<PatternToken>();
        }
        return patternTokens;
    }

    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     * 
     * @param patternTokens
     *        An array of structures where each structure contains information about one token that makes up the
     *        pattern.
     */

    public void setPatternTokens(java.util.Collection<PatternToken> patternTokens) {
        if (patternTokens == null) {
            this.patternTokens = null;
            return;
        }

        this.patternTokens = new com.amazonaws.internal.SdkInternalList<PatternToken>(patternTokens);
    }

    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
     * {@link #setPatternTokens(java.util.Collection)} or {@link #withPatternTokens(java.util.Collection)} if you want
     * to override the existing values.
     * </p>
     * 
     * @param patternTokens
     *        An array of structures where each structure contains information about one token that makes up the
     *        pattern.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPatternTokens(PatternToken... patternTokens) {
        if (this.patternTokens == null) {
            setPatternTokens(new com.amazonaws.internal.SdkInternalList<PatternToken>(patternTokens.length));
        }
        for (PatternToken ele : patternTokens) {
            this.patternTokens.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     * 
     * @param patternTokens
     *        An array of structures where each structure contains information about one token that makes up the
     *        pattern.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withPatternTokens(java.util.Collection<PatternToken> patternTokens) {
        setPatternTokens(patternTokens);
        return this;
    }

    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     * 
     * @return An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     */

    public java.util.List<String> getLogGroupArnList() {
        if (logGroupArnList == null) {
            logGroupArnList = new com.amazonaws.internal.SdkInternalList<String>();
        }
        return logGroupArnList;
    }

    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     * 
     * @param logGroupArnList
     *        An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     */

    public void setLogGroupArnList(java.util.Collection<String> logGroupArnList) {
        if (logGroupArnList == null) {
            this.logGroupArnList = null;
            return;
        }

        this.logGroupArnList = new com.amazonaws.internal.SdkInternalList<String>(logGroupArnList);
    }

    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     * <p>
     * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
     * {@link #setLogGroupArnList(java.util.Collection)} or {@link #withLogGroupArnList(java.util.Collection)} if you
     * want to override the existing values.
     * </p>
     * 
     * @param logGroupArnList
     *        An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withLogGroupArnList(String... logGroupArnList) {
        if (this.logGroupArnList == null) {
            setLogGroupArnList(new com.amazonaws.internal.SdkInternalList<String>(logGroupArnList.length));
        }
        for (String ele : logGroupArnList) {
            this.logGroupArnList.add(ele);
        }
        return this;
    }

    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     * 
     * @param logGroupArnList
     *        An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withLogGroupArnList(java.util.Collection<String> logGroupArnList) {
        setLogGroupArnList(logGroupArnList);
        return this;
    }

    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     * 
     * @param suppressed
     *        Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *        >UpdateAnomaly</a>.
     */

    public void setSuppressed(Boolean suppressed) {
        this.suppressed = suppressed;
    }

    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     * 
     * @return Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a>.
     */

    public Boolean getSuppressed() {
        return this.suppressed;
    }

    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     * 
     * @param suppressed
     *        Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *        >UpdateAnomaly</a>.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withSuppressed(Boolean suppressed) {
        setSuppressed(suppressed);
        return this;
    }

    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     * 
     * @return Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a>.
     */

    public Boolean isSuppressed() {
        return this.suppressed;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when it was suppressed.
     * </p>
     * 
     * @param suppressedDate
     *        If the anomaly is suppressed, this indicates when it was suppressed.
     */

    public void setSuppressedDate(Long suppressedDate) {
        this.suppressedDate = suppressedDate;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when it was suppressed.
     * </p>
     * 
     * @return If the anomaly is suppressed, this indicates when it was suppressed.
     */

    public Long getSuppressedDate() {
        return this.suppressedDate;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when it was suppressed.
     * </p>
     * 
     * @param suppressedDate
     *        If the anomaly is suppressed, this indicates when it was suppressed.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withSuppressedDate(Long suppressedDate) {
        setSuppressedDate(suppressedDate);
        return this;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>, the
     * anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * </p>
     * 
     * @param suppressedUntil
     *        If the anomaly is suppressed, this indicates when the suppression will end. If this value is
     *        <code>0</code>, the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     */

    public void setSuppressedUntil(Long suppressedUntil) {
        this.suppressedUntil = suppressedUntil;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>, the
     * anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * </p>
     * 
     * @return If the anomaly is suppressed, this indicates when the suppression will end. If this value is
     *         <code>0</code>, the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     */

    public Long getSuppressedUntil() {
        return this.suppressedUntil;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>, the
     * anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * </p>
     * 
     * @param suppressedUntil
     *        If the anomaly is suppressed, this indicates when the suppression will end. If this value is
     *        <code>0</code>, the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withSuppressedUntil(Long suppressedUntil) {
        setSuppressedUntil(suppressedUntil);
        return this;
    }

    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     * 
     * @param isPatternLevelSuppression
     *        If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern
     *        is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     */

    public void setIsPatternLevelSuppression(Boolean isPatternLevelSuppression) {
        this.isPatternLevelSuppression = isPatternLevelSuppression;
    }

    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     * 
     * @return If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern
     *         is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     */

    public Boolean getIsPatternLevelSuppression() {
        return this.isPatternLevelSuppression;
    }

    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     * 
     * @param isPatternLevelSuppression
     *        If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern
     *        is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * @return Returns a reference to this object so that method calls can be chained together.
     */

    public Anomaly withIsPatternLevelSuppression(Boolean isPatternLevelSuppression) {
        setIsPatternLevelSuppression(isPatternLevelSuppression);
        return this;
    }

    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     * 
     * @return If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern
     *         is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     */

    public Boolean isPatternLevelSuppression() {
        return this.isPatternLevelSuppression;
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getAnomalyId() != null)
            sb.append("AnomalyId: ").append(getAnomalyId()).append(",");
        if (getPatternId() != null)
            sb.append("PatternId: ").append(getPatternId()).append(",");
        if (getAnomalyDetectorArn() != null)
            sb.append("AnomalyDetectorArn: ").append(getAnomalyDetectorArn()).append(",");
        if (getPatternString() != null)
            sb.append("PatternString: ").append(getPatternString()).append(",");
        if (getPatternRegex() != null)
            sb.append("PatternRegex: ").append(getPatternRegex()).append(",");
        if (getPriority() != null)
            sb.append("Priority: ").append(getPriority()).append(",");
        if (getFirstSeen() != null)
            sb.append("FirstSeen: ").append(getFirstSeen()).append(",");
        if (getLastSeen() != null)
            sb.append("LastSeen: ").append(getLastSeen()).append(",");
        if (getDescription() != null)
            sb.append("Description: ").append(getDescription()).append(",");
        if (getActive() != null)
            sb.append("Active: ").append(getActive()).append(",");
        if (getState() != null)
            sb.append("State: ").append(getState()).append(",");
        if (getHistogram() != null)
            sb.append("Histogram: ").append(getHistogram()).append(",");
        if (getLogSamples() != null)
            sb.append("LogSamples: ").append(getLogSamples()).append(",");
        if (getPatternTokens() != null)
            sb.append("PatternTokens: ").append(getPatternTokens()).append(",");
        if (getLogGroupArnList() != null)
            sb.append("LogGroupArnList: ").append(getLogGroupArnList()).append(",");
        if (getSuppressed() != null)
            sb.append("Suppressed: ").append(getSuppressed()).append(",");
        if (getSuppressedDate() != null)
            sb.append("SuppressedDate: ").append(getSuppressedDate()).append(",");
        if (getSuppressedUntil() != null)
            sb.append("SuppressedUntil: ").append(getSuppressedUntil()).append(",");
        if (getIsPatternLevelSuppression() != null)
            sb.append("IsPatternLevelSuppression: ").append(getIsPatternLevelSuppression());
        sb.append("}");
        return sb.toString();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;

        if (obj instanceof Anomaly == false)
            return false;
        Anomaly other = (Anomaly) obj;
        if (other.getAnomalyId() == null ^ this.getAnomalyId() == null)
            return false;
        if (other.getAnomalyId() != null && other.getAnomalyId().equals(this.getAnomalyId()) == false)
            return false;
        if (other.getPatternId() == null ^ this.getPatternId() == null)
            return false;
        if (other.getPatternId() != null && other.getPatternId().equals(this.getPatternId()) == false)
            return false;
        if (other.getAnomalyDetectorArn() == null ^ this.getAnomalyDetectorArn() == null)
            return false;
        if (other.getAnomalyDetectorArn() != null && other.getAnomalyDetectorArn().equals(this.getAnomalyDetectorArn()) == false)
            return false;
        if (other.getPatternString() == null ^ this.getPatternString() == null)
            return false;
        if (other.getPatternString() != null && other.getPatternString().equals(this.getPatternString()) == false)
            return false;
        if (other.getPatternRegex() == null ^ this.getPatternRegex() == null)
            return false;
        if (other.getPatternRegex() != null && other.getPatternRegex().equals(this.getPatternRegex()) == false)
            return false;
        if (other.getPriority() == null ^ this.getPriority() == null)
            return false;
        if (other.getPriority() != null && other.getPriority().equals(this.getPriority()) == false)
            return false;
        if (other.getFirstSeen() == null ^ this.getFirstSeen() == null)
            return false;
        if (other.getFirstSeen() != null && other.getFirstSeen().equals(this.getFirstSeen()) == false)
            return false;
        if (other.getLastSeen() == null ^ this.getLastSeen() == null)
            return false;
        if (other.getLastSeen() != null && other.getLastSeen().equals(this.getLastSeen()) == false)
            return false;
        if (other.getDescription() == null ^ this.getDescription() == null)
            return false;
        if (other.getDescription() != null && other.getDescription().equals(this.getDescription()) == false)
            return false;
        if (other.getActive() == null ^ this.getActive() == null)
            return false;
        if (other.getActive() != null && other.getActive().equals(this.getActive()) == false)
            return false;
        if (other.getState() == null ^ this.getState() == null)
            return false;
        if (other.getState() != null && other.getState().equals(this.getState()) == false)
            return false;
        if (other.getHistogram() == null ^ this.getHistogram() == null)
            return false;
        if (other.getHistogram() != null && other.getHistogram().equals(this.getHistogram()) == false)
            return false;
        if (other.getLogSamples() == null ^ this.getLogSamples() == null)
            return false;
        if (other.getLogSamples() != null && other.getLogSamples().equals(this.getLogSamples()) == false)
            return false;
        if (other.getPatternTokens() == null ^ this.getPatternTokens() == null)
            return false;
        if (other.getPatternTokens() != null && other.getPatternTokens().equals(this.getPatternTokens()) == false)
            return false;
        if (other.getLogGroupArnList() == null ^ this.getLogGroupArnList() == null)
            return false;
        if (other.getLogGroupArnList() != null && other.getLogGroupArnList().equals(this.getLogGroupArnList()) == false)
            return false;
        if (other.getSuppressed() == null ^ this.getSuppressed() == null)
            return false;
        if (other.getSuppressed() != null && other.getSuppressed().equals(this.getSuppressed()) == false)
            return false;
        if (other.getSuppressedDate() == null ^ this.getSuppressedDate() == null)
            return false;
        if (other.getSuppressedDate() != null && other.getSuppressedDate().equals(this.getSuppressedDate()) == false)
            return false;
        if (other.getSuppressedUntil() == null ^ this.getSuppressedUntil() == null)
            return false;
        if (other.getSuppressedUntil() != null && other.getSuppressedUntil().equals(this.getSuppressedUntil()) == false)
            return false;
        if (other.getIsPatternLevelSuppression() == null ^ this.getIsPatternLevelSuppression() == null)
            return false;
        if (other.getIsPatternLevelSuppression() != null && other.getIsPatternLevelSuppression().equals(this.getIsPatternLevelSuppression()) == false)
            return false;
        return true;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = prime * hashCode + ((getAnomalyId() == null) ? 0 : getAnomalyId().hashCode());
        hashCode = prime * hashCode + ((getPatternId() == null) ? 0 : getPatternId().hashCode());
        hashCode = prime * hashCode + ((getAnomalyDetectorArn() == null) ? 0 : getAnomalyDetectorArn().hashCode());
        hashCode = prime * hashCode + ((getPatternString() == null) ? 0 : getPatternString().hashCode());
        hashCode = prime * hashCode + ((getPatternRegex() == null) ? 0 : getPatternRegex().hashCode());
        hashCode = prime * hashCode + ((getPriority() == null) ? 0 : getPriority().hashCode());
        hashCode = prime * hashCode + ((getFirstSeen() == null) ? 0 : getFirstSeen().hashCode());
        hashCode = prime * hashCode + ((getLastSeen() == null) ? 0 : getLastSeen().hashCode());
        hashCode = prime * hashCode + ((getDescription() == null) ? 0 : getDescription().hashCode());
        hashCode = prime * hashCode + ((getActive() == null) ? 0 : getActive().hashCode());
        hashCode = prime * hashCode + ((getState() == null) ? 0 : getState().hashCode());
        hashCode = prime * hashCode + ((getHistogram() == null) ? 0 : getHistogram().hashCode());
        hashCode = prime * hashCode + ((getLogSamples() == null) ? 0 : getLogSamples().hashCode());
        hashCode = prime * hashCode + ((getPatternTokens() == null) ? 0 : getPatternTokens().hashCode());
        hashCode = prime * hashCode + ((getLogGroupArnList() == null) ? 0 : getLogGroupArnList().hashCode());
        hashCode = prime * hashCode + ((getSuppressed() == null) ? 0 : getSuppressed().hashCode());
        hashCode = prime * hashCode + ((getSuppressedDate() == null) ? 0 : getSuppressedDate().hashCode());
        hashCode = prime * hashCode + ((getSuppressedUntil() == null) ? 0 : getSuppressedUntil().hashCode());
        hashCode = prime * hashCode + ((getIsPatternLevelSuppression() == null) ? 0 : getIsPatternLevelSuppression().hashCode());
        return hashCode;
    }

    @Override
    public Anomaly clone() {
        try {
            return (Anomaly) super.clone();
        } catch (CloneNotSupportedException e) {
            throw new IllegalStateException("Got a CloneNotSupportedException from Object.clone() " + "even though we're Cloneable!", e);
        }
    }

    @com.amazonaws.annotation.SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        com.amazonaws.services.logs.model.transform.AnomalyMarshaller.getInstance().marshall(this, protocolMarshaller);
    }
}
