/*
 * Copyright 2010 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 * 
 *  http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazonaws.services.s3.model;

import com.amazonaws.services.s3.AmazonS3;

/**
 * Multi-Factor Authentication (MFA) information to be included in S3 requests,
 * consisting of the serial number of the MFA device associated with your AWS
 * account and the current, unique MFA token generated by that device.
 * <p>
 * Each unique token generated by an MFA device can only be used in one request.
 * It is not valid to reuse the same token in additional requests.
 * <p>
 * For more information about uses of Multi-Factor Authentication in S3
 * operations, see {@link BucketVersioningConfiguration} and the explanation
 * of the MFA Delete functionality.
 * <p>
 * For more information on AWS Multi-Factor Authentication, including how to get
 * a device and associate it with your AWS account, see <a
 * href="http://aws.amazon.com/mfa"/>http://aws.amazon.com/mfa</a>
 */
public class MultiFactorAuthentication {

    /**
     * The serial number of the Multi-Factor Authentication device associated
     * with your AWS account.
     */
    private String deviceSerialNumber;

    /**
     * The current, unique Multi-Factor Authentication (MFA) token generated by
     * the MFA device associated with your AWS account.
     */
    private String token;

    /**
     * Constructs a new Multi-Factor Authentication object for use in any S3
     * operation that accepts requests with Multi-Factor Authentication (ex:
     * {@link AmazonS3#deleteVersion(DeleteVersionRequest)}).
     * 
     * @param deviceSerialNumber
     *            The serial number identifying the MFA device associated with
     *            your AWS account.
     * @param token
     *            The current, unique token generated by your MFA device.
     */
    public MultiFactorAuthentication(String deviceSerialNumber, String token) {
        this.deviceSerialNumber = deviceSerialNumber;
        this.token = token;
    }
    
    /**
     * Returns the Multi-Factor Authentication device serial number.
     * 
     * @return The Multi-Factor Authentication device serial number.
     */
    public String getDeviceSerialNumber() {
        return deviceSerialNumber;
    }

    /**
     * Sets the serial number of the Multi-Factor Authentication device
     * associated with your AWS account.
     * 
     * @param deviceSerialNumber
     *            The Multi-Factor Authentication device serial number for the
     *            device associated with your AWS account.
     */
    public void setDeviceSerialNumber(String deviceSerialNumber) {
        this.deviceSerialNumber = deviceSerialNumber;
    }

    /**
     * Sets the Multi-Factor Authentication device serial number to include with
     * this request, and returns this object so that additional method calls may
     * be chained together.
     * 
     * @param deviceSerialNumber
     *            The serial number of the Multi-Factor Authentication device
     *            associated with your AWS account.
     * 
     * @return This updated MultiFactorAuthentication object so that additional
     *         method calls may be chained together.
     */
    public MultiFactorAuthentication withDeviceSerialNumber(String deviceSerialNumber) {
        setDeviceSerialNumber(deviceSerialNumber);
        return this;
    }

    /**
     * Returns the Multi-Factor Authentication token.
     * 
     * @return The Multi-Factor Authentication token.
     */
    public String getToken() {
        return token;
    }

    /**
     * Sets the current, unique Multi-Factor Authentication token generated by
     * the device associated with your AWS account.
     * 
     * @param token
     *            The current, unique Multi-Factor Authentication token generated
     *            by the device associated with your AWS account.
     */
    public void setToken(String token) {
        this.token = token;
    }

    /**
     * Sets the current, unique Multi-Factor Authentication token generated by
     * the device associated with your AWS account, and returns this object so
     * that additional method calls may be chained together.
     * 
     * @param token
     *            The current, unique Multi-Factor Authentication token
     *            generated by the device associated with your AWS account.
     * 
     * @return This updated object so that additional method calls may be
     *         chained together.
     */
    public MultiFactorAuthentication withToken(String token) {
        setToken(token);
        return this;
    }

}
