/*
 * Copyright 2010-2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 * 
 *  http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazonaws.services.elasticache.model;
import com.amazonaws.AmazonWebServiceRequest;
import java.io.Serializable;

/**
 * Container for the parameters to the {@link com.amazonaws.services.elasticache.AmazonElastiCache#createCacheCluster(CreateCacheClusterRequest) CreateCacheCluster operation}.
 * <p>
 * Creates a new Cache Cluster.
 * </p>
 *
 * @see com.amazonaws.services.elasticache.AmazonElastiCache#createCacheCluster(CreateCacheClusterRequest)
 */
public class CreateCacheClusterRequest extends AmazonWebServiceRequest  implements Serializable  {

    /**
     * The Cache Cluster identifier. This parameter is stored as a lowercase
     * string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     * alphanumeric characters or hyphens.</li> <li>First character must be a
     * letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     * hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     */
    private String cacheClusterId;

    /**
     * The number of Cache Nodes the Cache Cluster should have.
     */
    private Integer numCacheNodes;

    /**
     * The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     * values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     * <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     * <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     * <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     * <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     * <code>cache.c1.xlarge</code>
     */
    private String cacheNodeType;

    /**
     * The name of the cache engine to be used for this Cache Cluster.
     * <note>Currently, <i>memcached</i> is the only cache engine supported
     * by the service.</note>
     */
    private String engine;

    /**
     * The version of the cache engine to be used for this cluster.
     */
    private String engineVersion;

    /**
     * The name of the cache parameter group to associate with this Cache
     * cluster. If this argument is omitted, the default CacheParameterGroup
     * for the specified engine will be used.
     */
    private String cacheParameterGroupName;

    /**
     * The name of the Cache Subnet Group to be used for the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     */
    private String cacheSubnetGroupName;

    /**
     * A list of Cache Security Group Names to associate with this Cache
     * Cluster. <p> Use this parameter only when you are creating a cluster
     * outside of an Amazon Virtual Private Cloud (VPC).
     */
    private com.amazonaws.internal.ListWithAutoConstructFlag<String> cacheSecurityGroupNames;

    /**
     * Specifies the VPC Security Groups associated with the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     */
    private com.amazonaws.internal.ListWithAutoConstructFlag<String> securityGroupIds;

    /**
     * The EC2 Availability Zone that the Cache Cluster will be created in.
     * <p> All cache nodes belonging to a cache cluster are placed in the
     * preferred availability zone. <p> Default: System chosen (random)
     * availability zone.
     */
    private String preferredAvailabilityZone;

    /**
     * The weekly time range (in UTC) during which system maintenance can
     * occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     */
    private String preferredMaintenanceWindow;

    /**
     * The port number on which each of the Cache Nodes will accept
     * connections.
     */
    private Integer port;

    /**
     * The Amazon Resource Name (ARN) of the Amazon Simple Notification
     * Service (SNS) topic to which notifications will be sent. <note> The
     * Amazon SNS topic owner must be the same as the Cache Cluster owner.
     * </note>
     */
    private String notificationTopicArn;

    /**
     * Indicates that minor engine upgrades will be applied automatically to
     * the Cache Cluster during the maintenance window. <p>Default:
     * <code>true</code>
     */
    private Boolean autoMinorVersionUpgrade;

    /**
     * Default constructor for a new CreateCacheClusterRequest object.  Callers should use the
     * setter or fluent setter (with...) methods to initialize this object after creating it.
     */
    public CreateCacheClusterRequest() {}
    


    /**
     * Constructs a new CreateCacheClusterRequest object.
     * Callers should use the setter or fluent setter (with...) methods to
     * initialize any additional object members.
     * 
     * @param cacheClusterId The Cache Cluster identifier. This parameter is
     * stored as a lowercase string. <p>Constraints: <ul> <li>Must contain
     * from 1 to 20 alphanumeric characters or hyphens.</li> <li>First
     * character must be a letter.</li> <li>Cannot end with a hyphen or
     * contain two consecutive hyphens.</li> </ul> <p>Example:
     * <code>mycachecluster</code>
     * @param numCacheNodes The number of Cache Nodes the Cache Cluster
     * should have.
     * @param cacheNodeType The compute and memory capacity of nodes in a
     * Cache Cluster. <p>Valid values: <code>cache.t1.micro</code> |
     * <code>cache.m1.small</code> | <code>cache.m1.medium</code> |
     * <code>cache.m1.large</code> | <code>cache.m1.xlarge</code> |
     * <code>cache.m3.xlarge</code> | <code>cache.m3.2xlarge</code> |
     * <code>cache.m2.xlarge</code> | <code>cache.m2.2xlarge</code> |
     * <code>cache.m2.4xlarge</code> | <code>cache.c1.xlarge</code>
     * @param engine The name of the cache engine to be used for this Cache
     * Cluster. <note>Currently, <i>memcached</i> is the only cache engine
     * supported by the service.</note>
     * @param cacheSecurityGroupNames A list of Cache Security Group Names to
     * associate with this Cache Cluster. <p> Use this parameter only when
     * you are creating a cluster outside of an Amazon Virtual Private Cloud
     * (VPC).
     */
    public CreateCacheClusterRequest(String cacheClusterId, Integer numCacheNodes, String cacheNodeType, String engine, java.util.List<String> cacheSecurityGroupNames) {
        setCacheClusterId(cacheClusterId);
        setNumCacheNodes(numCacheNodes);
        setCacheNodeType(cacheNodeType);
        setEngine(engine);
        setCacheSecurityGroupNames(cacheSecurityGroupNames);
    }

    
    
    /**
     * The Cache Cluster identifier. This parameter is stored as a lowercase
     * string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     * alphanumeric characters or hyphens.</li> <li>First character must be a
     * letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     * hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     *
     * @return The Cache Cluster identifier. This parameter is stored as a lowercase
     *         string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     *         alphanumeric characters or hyphens.</li> <li>First character must be a
     *         letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     *         hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     */
    public String getCacheClusterId() {
        return cacheClusterId;
    }
    
    /**
     * The Cache Cluster identifier. This parameter is stored as a lowercase
     * string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     * alphanumeric characters or hyphens.</li> <li>First character must be a
     * letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     * hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     *
     * @param cacheClusterId The Cache Cluster identifier. This parameter is stored as a lowercase
     *         string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     *         alphanumeric characters or hyphens.</li> <li>First character must be a
     *         letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     *         hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     */
    public void setCacheClusterId(String cacheClusterId) {
        this.cacheClusterId = cacheClusterId;
    }
    
    /**
     * The Cache Cluster identifier. This parameter is stored as a lowercase
     * string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     * alphanumeric characters or hyphens.</li> <li>First character must be a
     * letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     * hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheClusterId The Cache Cluster identifier. This parameter is stored as a lowercase
     *         string. <p>Constraints: <ul> <li>Must contain from 1 to 20
     *         alphanumeric characters or hyphens.</li> <li>First character must be a
     *         letter.</li> <li>Cannot end with a hyphen or contain two consecutive
     *         hyphens.</li> </ul> <p>Example: <code>mycachecluster</code>
     */
    public CreateCacheClusterRequest withCacheClusterId(String cacheClusterId) {
        this.cacheClusterId = cacheClusterId;
        return this;
    }
    
    
    /**
     * The number of Cache Nodes the Cache Cluster should have.
     *
     * @return The number of Cache Nodes the Cache Cluster should have.
     */
    public Integer getNumCacheNodes() {
        return numCacheNodes;
    }
    
    /**
     * The number of Cache Nodes the Cache Cluster should have.
     *
     * @param numCacheNodes The number of Cache Nodes the Cache Cluster should have.
     */
    public void setNumCacheNodes(Integer numCacheNodes) {
        this.numCacheNodes = numCacheNodes;
    }
    
    /**
     * The number of Cache Nodes the Cache Cluster should have.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param numCacheNodes The number of Cache Nodes the Cache Cluster should have.
     */
    public CreateCacheClusterRequest withNumCacheNodes(Integer numCacheNodes) {
        this.numCacheNodes = numCacheNodes;
        return this;
    }
    
    
    /**
     * The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     * values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     * <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     * <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     * <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     * <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     * <code>cache.c1.xlarge</code>
     *
     * @return The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     *         values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     *         <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     *         <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     *         <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     *         <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     *         <code>cache.c1.xlarge</code>
     */
    public String getCacheNodeType() {
        return cacheNodeType;
    }
    
    /**
     * The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     * values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     * <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     * <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     * <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     * <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     * <code>cache.c1.xlarge</code>
     *
     * @param cacheNodeType The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     *         values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     *         <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     *         <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     *         <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     *         <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     *         <code>cache.c1.xlarge</code>
     */
    public void setCacheNodeType(String cacheNodeType) {
        this.cacheNodeType = cacheNodeType;
    }
    
    /**
     * The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     * values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     * <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     * <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     * <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     * <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     * <code>cache.c1.xlarge</code>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheNodeType The compute and memory capacity of nodes in a Cache Cluster. <p>Valid
     *         values: <code>cache.t1.micro</code> | <code>cache.m1.small</code> |
     *         <code>cache.m1.medium</code> | <code>cache.m1.large</code> |
     *         <code>cache.m1.xlarge</code> | <code>cache.m3.xlarge</code> |
     *         <code>cache.m3.2xlarge</code> | <code>cache.m2.xlarge</code> |
     *         <code>cache.m2.2xlarge</code> | <code>cache.m2.4xlarge</code> |
     *         <code>cache.c1.xlarge</code>
     */
    public CreateCacheClusterRequest withCacheNodeType(String cacheNodeType) {
        this.cacheNodeType = cacheNodeType;
        return this;
    }
    
    
    /**
     * The name of the cache engine to be used for this Cache Cluster.
     * <note>Currently, <i>memcached</i> is the only cache engine supported
     * by the service.</note>
     *
     * @return The name of the cache engine to be used for this Cache Cluster.
     *         <note>Currently, <i>memcached</i> is the only cache engine supported
     *         by the service.</note>
     */
    public String getEngine() {
        return engine;
    }
    
    /**
     * The name of the cache engine to be used for this Cache Cluster.
     * <note>Currently, <i>memcached</i> is the only cache engine supported
     * by the service.</note>
     *
     * @param engine The name of the cache engine to be used for this Cache Cluster.
     *         <note>Currently, <i>memcached</i> is the only cache engine supported
     *         by the service.</note>
     */
    public void setEngine(String engine) {
        this.engine = engine;
    }
    
    /**
     * The name of the cache engine to be used for this Cache Cluster.
     * <note>Currently, <i>memcached</i> is the only cache engine supported
     * by the service.</note>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param engine The name of the cache engine to be used for this Cache Cluster.
     *         <note>Currently, <i>memcached</i> is the only cache engine supported
     *         by the service.</note>
     */
    public CreateCacheClusterRequest withEngine(String engine) {
        this.engine = engine;
        return this;
    }
    
    
    /**
     * The version of the cache engine to be used for this cluster.
     *
     * @return The version of the cache engine to be used for this cluster.
     */
    public String getEngineVersion() {
        return engineVersion;
    }
    
    /**
     * The version of the cache engine to be used for this cluster.
     *
     * @param engineVersion The version of the cache engine to be used for this cluster.
     */
    public void setEngineVersion(String engineVersion) {
        this.engineVersion = engineVersion;
    }
    
    /**
     * The version of the cache engine to be used for this cluster.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param engineVersion The version of the cache engine to be used for this cluster.
     */
    public CreateCacheClusterRequest withEngineVersion(String engineVersion) {
        this.engineVersion = engineVersion;
        return this;
    }
    
    
    /**
     * The name of the cache parameter group to associate with this Cache
     * cluster. If this argument is omitted, the default CacheParameterGroup
     * for the specified engine will be used.
     *
     * @return The name of the cache parameter group to associate with this Cache
     *         cluster. If this argument is omitted, the default CacheParameterGroup
     *         for the specified engine will be used.
     */
    public String getCacheParameterGroupName() {
        return cacheParameterGroupName;
    }
    
    /**
     * The name of the cache parameter group to associate with this Cache
     * cluster. If this argument is omitted, the default CacheParameterGroup
     * for the specified engine will be used.
     *
     * @param cacheParameterGroupName The name of the cache parameter group to associate with this Cache
     *         cluster. If this argument is omitted, the default CacheParameterGroup
     *         for the specified engine will be used.
     */
    public void setCacheParameterGroupName(String cacheParameterGroupName) {
        this.cacheParameterGroupName = cacheParameterGroupName;
    }
    
    /**
     * The name of the cache parameter group to associate with this Cache
     * cluster. If this argument is omitted, the default CacheParameterGroup
     * for the specified engine will be used.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheParameterGroupName The name of the cache parameter group to associate with this Cache
     *         cluster. If this argument is omitted, the default CacheParameterGroup
     *         for the specified engine will be used.
     */
    public CreateCacheClusterRequest withCacheParameterGroupName(String cacheParameterGroupName) {
        this.cacheParameterGroupName = cacheParameterGroupName;
        return this;
    }
    
    
    /**
     * The name of the Cache Subnet Group to be used for the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     *
     * @return The name of the Cache Subnet Group to be used for the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public String getCacheSubnetGroupName() {
        return cacheSubnetGroupName;
    }
    
    /**
     * The name of the Cache Subnet Group to be used for the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     *
     * @param cacheSubnetGroupName The name of the Cache Subnet Group to be used for the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public void setCacheSubnetGroupName(String cacheSubnetGroupName) {
        this.cacheSubnetGroupName = cacheSubnetGroupName;
    }
    
    /**
     * The name of the Cache Subnet Group to be used for the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheSubnetGroupName The name of the Cache Subnet Group to be used for the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public CreateCacheClusterRequest withCacheSubnetGroupName(String cacheSubnetGroupName) {
        this.cacheSubnetGroupName = cacheSubnetGroupName;
        return this;
    }
    
    
    /**
     * A list of Cache Security Group Names to associate with this Cache
     * Cluster. <p> Use this parameter only when you are creating a cluster
     * outside of an Amazon Virtual Private Cloud (VPC).
     *
     * @return A list of Cache Security Group Names to associate with this Cache
     *         Cluster. <p> Use this parameter only when you are creating a cluster
     *         outside of an Amazon Virtual Private Cloud (VPC).
     */
    public java.util.List<String> getCacheSecurityGroupNames() {
        
        if (cacheSecurityGroupNames == null) {
              cacheSecurityGroupNames = new com.amazonaws.internal.ListWithAutoConstructFlag<String>();
              cacheSecurityGroupNames.setAutoConstruct(true);
        }
        return cacheSecurityGroupNames;
    }
    
    /**
     * A list of Cache Security Group Names to associate with this Cache
     * Cluster. <p> Use this parameter only when you are creating a cluster
     * outside of an Amazon Virtual Private Cloud (VPC).
     *
     * @param cacheSecurityGroupNames A list of Cache Security Group Names to associate with this Cache
     *         Cluster. <p> Use this parameter only when you are creating a cluster
     *         outside of an Amazon Virtual Private Cloud (VPC).
     */
    public void setCacheSecurityGroupNames(java.util.Collection<String> cacheSecurityGroupNames) {
        if (cacheSecurityGroupNames == null) {
            this.cacheSecurityGroupNames = null;
            return;
        }
        com.amazonaws.internal.ListWithAutoConstructFlag<String> cacheSecurityGroupNamesCopy = new com.amazonaws.internal.ListWithAutoConstructFlag<String>(cacheSecurityGroupNames.size());
        cacheSecurityGroupNamesCopy.addAll(cacheSecurityGroupNames);
        this.cacheSecurityGroupNames = cacheSecurityGroupNamesCopy;
    }
    
    /**
     * A list of Cache Security Group Names to associate with this Cache
     * Cluster. <p> Use this parameter only when you are creating a cluster
     * outside of an Amazon Virtual Private Cloud (VPC).
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheSecurityGroupNames A list of Cache Security Group Names to associate with this Cache
     *         Cluster. <p> Use this parameter only when you are creating a cluster
     *         outside of an Amazon Virtual Private Cloud (VPC).
     */
    public CreateCacheClusterRequest withCacheSecurityGroupNames(String... cacheSecurityGroupNames) {
        if (getCacheSecurityGroupNames() == null) setCacheSecurityGroupNames(new java.util.ArrayList<String>(cacheSecurityGroupNames.length));
        for (String value : cacheSecurityGroupNames) {
            getCacheSecurityGroupNames().add(value);
        }
        return this;
    }
    
    /**
     * A list of Cache Security Group Names to associate with this Cache
     * Cluster. <p> Use this parameter only when you are creating a cluster
     * outside of an Amazon Virtual Private Cloud (VPC).
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param cacheSecurityGroupNames A list of Cache Security Group Names to associate with this Cache
     *         Cluster. <p> Use this parameter only when you are creating a cluster
     *         outside of an Amazon Virtual Private Cloud (VPC).
     */
    public CreateCacheClusterRequest withCacheSecurityGroupNames(java.util.Collection<String> cacheSecurityGroupNames) {
        if (cacheSecurityGroupNames == null) {
            this.cacheSecurityGroupNames = null;
        } else {
            com.amazonaws.internal.ListWithAutoConstructFlag<String> cacheSecurityGroupNamesCopy = new com.amazonaws.internal.ListWithAutoConstructFlag<String>(cacheSecurityGroupNames.size());
            cacheSecurityGroupNamesCopy.addAll(cacheSecurityGroupNames);
            this.cacheSecurityGroupNames = cacheSecurityGroupNamesCopy;
        }

        return this;
    }
    
    /**
     * Specifies the VPC Security Groups associated with the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     *
     * @return Specifies the VPC Security Groups associated with the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public java.util.List<String> getSecurityGroupIds() {
        
        if (securityGroupIds == null) {
              securityGroupIds = new com.amazonaws.internal.ListWithAutoConstructFlag<String>();
              securityGroupIds.setAutoConstruct(true);
        }
        return securityGroupIds;
    }
    
    /**
     * Specifies the VPC Security Groups associated with the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     *
     * @param securityGroupIds Specifies the VPC Security Groups associated with the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public void setSecurityGroupIds(java.util.Collection<String> securityGroupIds) {
        if (securityGroupIds == null) {
            this.securityGroupIds = null;
            return;
        }
        com.amazonaws.internal.ListWithAutoConstructFlag<String> securityGroupIdsCopy = new com.amazonaws.internal.ListWithAutoConstructFlag<String>(securityGroupIds.size());
        securityGroupIdsCopy.addAll(securityGroupIds);
        this.securityGroupIds = securityGroupIdsCopy;
    }
    
    /**
     * Specifies the VPC Security Groups associated with the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param securityGroupIds Specifies the VPC Security Groups associated with the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public CreateCacheClusterRequest withSecurityGroupIds(String... securityGroupIds) {
        if (getSecurityGroupIds() == null) setSecurityGroupIds(new java.util.ArrayList<String>(securityGroupIds.length));
        for (String value : securityGroupIds) {
            getSecurityGroupIds().add(value);
        }
        return this;
    }
    
    /**
     * Specifies the VPC Security Groups associated with the Cache Cluster.
     * <p>Use this parameter only when you are creating a cluster in an
     * Amazon Virtual Private Cloud (VPC).
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param securityGroupIds Specifies the VPC Security Groups associated with the Cache Cluster.
     *         <p>Use this parameter only when you are creating a cluster in an
     *         Amazon Virtual Private Cloud (VPC).
     */
    public CreateCacheClusterRequest withSecurityGroupIds(java.util.Collection<String> securityGroupIds) {
        if (securityGroupIds == null) {
            this.securityGroupIds = null;
        } else {
            com.amazonaws.internal.ListWithAutoConstructFlag<String> securityGroupIdsCopy = new com.amazonaws.internal.ListWithAutoConstructFlag<String>(securityGroupIds.size());
            securityGroupIdsCopy.addAll(securityGroupIds);
            this.securityGroupIds = securityGroupIdsCopy;
        }

        return this;
    }
    
    /**
     * The EC2 Availability Zone that the Cache Cluster will be created in.
     * <p> All cache nodes belonging to a cache cluster are placed in the
     * preferred availability zone. <p> Default: System chosen (random)
     * availability zone.
     *
     * @return The EC2 Availability Zone that the Cache Cluster will be created in.
     *         <p> All cache nodes belonging to a cache cluster are placed in the
     *         preferred availability zone. <p> Default: System chosen (random)
     *         availability zone.
     */
    public String getPreferredAvailabilityZone() {
        return preferredAvailabilityZone;
    }
    
    /**
     * The EC2 Availability Zone that the Cache Cluster will be created in.
     * <p> All cache nodes belonging to a cache cluster are placed in the
     * preferred availability zone. <p> Default: System chosen (random)
     * availability zone.
     *
     * @param preferredAvailabilityZone The EC2 Availability Zone that the Cache Cluster will be created in.
     *         <p> All cache nodes belonging to a cache cluster are placed in the
     *         preferred availability zone. <p> Default: System chosen (random)
     *         availability zone.
     */
    public void setPreferredAvailabilityZone(String preferredAvailabilityZone) {
        this.preferredAvailabilityZone = preferredAvailabilityZone;
    }
    
    /**
     * The EC2 Availability Zone that the Cache Cluster will be created in.
     * <p> All cache nodes belonging to a cache cluster are placed in the
     * preferred availability zone. <p> Default: System chosen (random)
     * availability zone.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param preferredAvailabilityZone The EC2 Availability Zone that the Cache Cluster will be created in.
     *         <p> All cache nodes belonging to a cache cluster are placed in the
     *         preferred availability zone. <p> Default: System chosen (random)
     *         availability zone.
     */
    public CreateCacheClusterRequest withPreferredAvailabilityZone(String preferredAvailabilityZone) {
        this.preferredAvailabilityZone = preferredAvailabilityZone;
        return this;
    }
    
    
    /**
     * The weekly time range (in UTC) during which system maintenance can
     * occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     *
     * @return The weekly time range (in UTC) during which system maintenance can
     *         occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     */
    public String getPreferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }
    
    /**
     * The weekly time range (in UTC) during which system maintenance can
     * occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     *
     * @param preferredMaintenanceWindow The weekly time range (in UTC) during which system maintenance can
     *         occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     */
    public void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
        this.preferredMaintenanceWindow = preferredMaintenanceWindow;
    }
    
    /**
     * The weekly time range (in UTC) during which system maintenance can
     * occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param preferredMaintenanceWindow The weekly time range (in UTC) during which system maintenance can
     *         occur. <p> Example: <code>sun:05:00-sun:09:00</code>
     */
    public CreateCacheClusterRequest withPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
        this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        return this;
    }
    
    
    /**
     * The port number on which each of the Cache Nodes will accept
     * connections.
     *
     * @return The port number on which each of the Cache Nodes will accept
     *         connections.
     */
    public Integer getPort() {
        return port;
    }
    
    /**
     * The port number on which each of the Cache Nodes will accept
     * connections.
     *
     * @param port The port number on which each of the Cache Nodes will accept
     *         connections.
     */
    public void setPort(Integer port) {
        this.port = port;
    }
    
    /**
     * The port number on which each of the Cache Nodes will accept
     * connections.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param port The port number on which each of the Cache Nodes will accept
     *         connections.
     */
    public CreateCacheClusterRequest withPort(Integer port) {
        this.port = port;
        return this;
    }
    
    
    /**
     * The Amazon Resource Name (ARN) of the Amazon Simple Notification
     * Service (SNS) topic to which notifications will be sent. <note> The
     * Amazon SNS topic owner must be the same as the Cache Cluster owner.
     * </note>
     *
     * @return The Amazon Resource Name (ARN) of the Amazon Simple Notification
     *         Service (SNS) topic to which notifications will be sent. <note> The
     *         Amazon SNS topic owner must be the same as the Cache Cluster owner.
     *         </note>
     */
    public String getNotificationTopicArn() {
        return notificationTopicArn;
    }
    
    /**
     * The Amazon Resource Name (ARN) of the Amazon Simple Notification
     * Service (SNS) topic to which notifications will be sent. <note> The
     * Amazon SNS topic owner must be the same as the Cache Cluster owner.
     * </note>
     *
     * @param notificationTopicArn The Amazon Resource Name (ARN) of the Amazon Simple Notification
     *         Service (SNS) topic to which notifications will be sent. <note> The
     *         Amazon SNS topic owner must be the same as the Cache Cluster owner.
     *         </note>
     */
    public void setNotificationTopicArn(String notificationTopicArn) {
        this.notificationTopicArn = notificationTopicArn;
    }
    
    /**
     * The Amazon Resource Name (ARN) of the Amazon Simple Notification
     * Service (SNS) topic to which notifications will be sent. <note> The
     * Amazon SNS topic owner must be the same as the Cache Cluster owner.
     * </note>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param notificationTopicArn The Amazon Resource Name (ARN) of the Amazon Simple Notification
     *         Service (SNS) topic to which notifications will be sent. <note> The
     *         Amazon SNS topic owner must be the same as the Cache Cluster owner.
     *         </note>
     */
    public CreateCacheClusterRequest withNotificationTopicArn(String notificationTopicArn) {
        this.notificationTopicArn = notificationTopicArn;
        return this;
    }
    
    
    /**
     * Indicates that minor engine upgrades will be applied automatically to
     * the Cache Cluster during the maintenance window. <p>Default:
     * <code>true</code>
     *
     * @return Indicates that minor engine upgrades will be applied automatically to
     *         the Cache Cluster during the maintenance window. <p>Default:
     *         <code>true</code>
     */
    public Boolean isAutoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }
    
    /**
     * Indicates that minor engine upgrades will be applied automatically to
     * the Cache Cluster during the maintenance window. <p>Default:
     * <code>true</code>
     *
     * @param autoMinorVersionUpgrade Indicates that minor engine upgrades will be applied automatically to
     *         the Cache Cluster during the maintenance window. <p>Default:
     *         <code>true</code>
     */
    public void setAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
        this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
    }
    
    /**
     * Indicates that minor engine upgrades will be applied automatically to
     * the Cache Cluster during the maintenance window. <p>Default:
     * <code>true</code>
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param autoMinorVersionUpgrade Indicates that minor engine upgrades will be applied automatically to
     *         the Cache Cluster during the maintenance window. <p>Default:
     *         <code>true</code>
     */
    public CreateCacheClusterRequest withAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
        this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
        return this;
    }
    
    
    /**
     * Indicates that minor engine upgrades will be applied automatically to
     * the Cache Cluster during the maintenance window. <p>Default:
     * <code>true</code>
     *
     * @return Indicates that minor engine upgrades will be applied automatically to
     *         the Cache Cluster during the maintenance window. <p>Default:
     *         <code>true</code>
     */
    public Boolean getAutoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }
    
    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getCacheClusterId() != null) sb.append("CacheClusterId: " + getCacheClusterId() + ",");
        if (getNumCacheNodes() != null) sb.append("NumCacheNodes: " + getNumCacheNodes() + ",");
        if (getCacheNodeType() != null) sb.append("CacheNodeType: " + getCacheNodeType() + ",");
        if (getEngine() != null) sb.append("Engine: " + getEngine() + ",");
        if (getEngineVersion() != null) sb.append("EngineVersion: " + getEngineVersion() + ",");
        if (getCacheParameterGroupName() != null) sb.append("CacheParameterGroupName: " + getCacheParameterGroupName() + ",");
        if (getCacheSubnetGroupName() != null) sb.append("CacheSubnetGroupName: " + getCacheSubnetGroupName() + ",");
        if (getCacheSecurityGroupNames() != null) sb.append("CacheSecurityGroupNames: " + getCacheSecurityGroupNames() + ",");
        if (getSecurityGroupIds() != null) sb.append("SecurityGroupIds: " + getSecurityGroupIds() + ",");
        if (getPreferredAvailabilityZone() != null) sb.append("PreferredAvailabilityZone: " + getPreferredAvailabilityZone() + ",");
        if (getPreferredMaintenanceWindow() != null) sb.append("PreferredMaintenanceWindow: " + getPreferredMaintenanceWindow() + ",");
        if (getPort() != null) sb.append("Port: " + getPort() + ",");
        if (getNotificationTopicArn() != null) sb.append("NotificationTopicArn: " + getNotificationTopicArn() + ",");
        if (isAutoMinorVersionUpgrade() != null) sb.append("AutoMinorVersionUpgrade: " + isAutoMinorVersionUpgrade() );
        sb.append("}");
        return sb.toString();
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;
        
        hashCode = prime * hashCode + ((getCacheClusterId() == null) ? 0 : getCacheClusterId().hashCode()); 
        hashCode = prime * hashCode + ((getNumCacheNodes() == null) ? 0 : getNumCacheNodes().hashCode()); 
        hashCode = prime * hashCode + ((getCacheNodeType() == null) ? 0 : getCacheNodeType().hashCode()); 
        hashCode = prime * hashCode + ((getEngine() == null) ? 0 : getEngine().hashCode()); 
        hashCode = prime * hashCode + ((getEngineVersion() == null) ? 0 : getEngineVersion().hashCode()); 
        hashCode = prime * hashCode + ((getCacheParameterGroupName() == null) ? 0 : getCacheParameterGroupName().hashCode()); 
        hashCode = prime * hashCode + ((getCacheSubnetGroupName() == null) ? 0 : getCacheSubnetGroupName().hashCode()); 
        hashCode = prime * hashCode + ((getCacheSecurityGroupNames() == null) ? 0 : getCacheSecurityGroupNames().hashCode()); 
        hashCode = prime * hashCode + ((getSecurityGroupIds() == null) ? 0 : getSecurityGroupIds().hashCode()); 
        hashCode = prime * hashCode + ((getPreferredAvailabilityZone() == null) ? 0 : getPreferredAvailabilityZone().hashCode()); 
        hashCode = prime * hashCode + ((getPreferredMaintenanceWindow() == null) ? 0 : getPreferredMaintenanceWindow().hashCode()); 
        hashCode = prime * hashCode + ((getPort() == null) ? 0 : getPort().hashCode()); 
        hashCode = prime * hashCode + ((getNotificationTopicArn() == null) ? 0 : getNotificationTopicArn().hashCode()); 
        hashCode = prime * hashCode + ((isAutoMinorVersionUpgrade() == null) ? 0 : isAutoMinorVersionUpgrade().hashCode()); 
        return hashCode;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj == null) return false;

        if (obj instanceof CreateCacheClusterRequest == false) return false;
        CreateCacheClusterRequest other = (CreateCacheClusterRequest)obj;
        
        if (other.getCacheClusterId() == null ^ this.getCacheClusterId() == null) return false;
        if (other.getCacheClusterId() != null && other.getCacheClusterId().equals(this.getCacheClusterId()) == false) return false; 
        if (other.getNumCacheNodes() == null ^ this.getNumCacheNodes() == null) return false;
        if (other.getNumCacheNodes() != null && other.getNumCacheNodes().equals(this.getNumCacheNodes()) == false) return false; 
        if (other.getCacheNodeType() == null ^ this.getCacheNodeType() == null) return false;
        if (other.getCacheNodeType() != null && other.getCacheNodeType().equals(this.getCacheNodeType()) == false) return false; 
        if (other.getEngine() == null ^ this.getEngine() == null) return false;
        if (other.getEngine() != null && other.getEngine().equals(this.getEngine()) == false) return false; 
        if (other.getEngineVersion() == null ^ this.getEngineVersion() == null) return false;
        if (other.getEngineVersion() != null && other.getEngineVersion().equals(this.getEngineVersion()) == false) return false; 
        if (other.getCacheParameterGroupName() == null ^ this.getCacheParameterGroupName() == null) return false;
        if (other.getCacheParameterGroupName() != null && other.getCacheParameterGroupName().equals(this.getCacheParameterGroupName()) == false) return false; 
        if (other.getCacheSubnetGroupName() == null ^ this.getCacheSubnetGroupName() == null) return false;
        if (other.getCacheSubnetGroupName() != null && other.getCacheSubnetGroupName().equals(this.getCacheSubnetGroupName()) == false) return false; 
        if (other.getCacheSecurityGroupNames() == null ^ this.getCacheSecurityGroupNames() == null) return false;
        if (other.getCacheSecurityGroupNames() != null && other.getCacheSecurityGroupNames().equals(this.getCacheSecurityGroupNames()) == false) return false; 
        if (other.getSecurityGroupIds() == null ^ this.getSecurityGroupIds() == null) return false;
        if (other.getSecurityGroupIds() != null && other.getSecurityGroupIds().equals(this.getSecurityGroupIds()) == false) return false; 
        if (other.getPreferredAvailabilityZone() == null ^ this.getPreferredAvailabilityZone() == null) return false;
        if (other.getPreferredAvailabilityZone() != null && other.getPreferredAvailabilityZone().equals(this.getPreferredAvailabilityZone()) == false) return false; 
        if (other.getPreferredMaintenanceWindow() == null ^ this.getPreferredMaintenanceWindow() == null) return false;
        if (other.getPreferredMaintenanceWindow() != null && other.getPreferredMaintenanceWindow().equals(this.getPreferredMaintenanceWindow()) == false) return false; 
        if (other.getPort() == null ^ this.getPort() == null) return false;
        if (other.getPort() != null && other.getPort().equals(this.getPort()) == false) return false; 
        if (other.getNotificationTopicArn() == null ^ this.getNotificationTopicArn() == null) return false;
        if (other.getNotificationTopicArn() != null && other.getNotificationTopicArn().equals(this.getNotificationTopicArn()) == false) return false; 
        if (other.isAutoMinorVersionUpgrade() == null ^ this.isAutoMinorVersionUpgrade() == null) return false;
        if (other.isAutoMinorVersionUpgrade() != null && other.isAutoMinorVersionUpgrade().equals(this.isAutoMinorVersionUpgrade()) == false) return false; 
        return true;
    }
    
}
    