/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.ivs.player;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.media.MediaFormat;
import android.os.BatteryManager;
import android.os.Build;
import android.os.PowerManager;
import android.os.Process;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.WindowManager;
import androidx.annotation.NonNull;
import com.amazonaws.ivs.net.HttpClient;
import com.amazonaws.ivs.net.HttpClientFactory;
import com.amazonaws.ivs.net.NetworkLinkInfo;
import com.amazonaws.ivs.player.AudioTrackRenderer;
import com.amazonaws.ivs.player.ErrorType;
import com.amazonaws.ivs.player.Logging;
import com.amazonaws.ivs.player.MediaCodecDecoder;
import com.amazonaws.ivs.player.MediaDecoder;
import com.amazonaws.ivs.player.MediaRenderer;
import com.amazonaws.ivs.player.MediaType;
import com.amazonaws.ivs.player.PlayerException;
import com.amazonaws.ivs.player.ProtectionSystem;
import com.amazonaws.ivs.player.Quality;
import com.amazonaws.ivs.player.Size;
import com.amazonaws.ivs.player.SurfaceRenderer;
import com.amazonaws.ivs.player.VideoCapabilities;
import com.amazonaws.ivs.player.VideoRenderer;
import java.lang.ref.WeakReference;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;

public class Platform {
    public static final String LIBRARY_NAME = "playercore";
    private static ErrorListener errorListener;
    private final Context context;
    private AudioTrackRenderer audioRenderer;
    private SurfaceRenderer surfaceRenderer;
    private ConcurrentLinkedQueue<Throwable> pendingExceptions;
    private WeakReference<VideoRenderer.VideoSizeListener> videoSizeListener;
    private Map<String, String> fixedDeviceProperties;
    private Size maxDisplayModeSize;
    private String videoMediaType;
    private PowerManager powerManager;
    private BatteryManager batteryManager;
    private Intent batteryStatus;

    Platform(Context context) {
        this(context, null);
    }

    Platform(Context context, VideoRenderer.VideoSizeListener videoSizeListener) {
        this.context = context;
        this.pendingExceptions = new ConcurrentLinkedQueue();
        this.fixedDeviceProperties = new HashMap<String, String>();
        this.maxDisplayModeSize = new Size(0, 0);
        if (videoSizeListener != null) {
            this.videoSizeListener = new WeakReference<VideoRenderer.VideoSizeListener>(videoSizeListener);
        }
        this.fixedDeviceProperties.put("device_software", "android-" + Build.VERSION.SDK_INT);
        this.fixedDeviceProperties.put("device_os_version", Build.VERSION.RELEASE);
        this.fixedDeviceProperties.put("device_model", Build.MODEL);
        this.fixedDeviceProperties.put("device_manufacturer", Build.MANUFACTURER);
        this.fixedDeviceProperties.put("device_diagonal", this.getDeviceDiagonalSize());
        if (context != null && context.getPackageManager() != null) {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), 128);
                if (appInfo != null && appInfo.metaData != null) {
                    String framework = appInfo.metaData.getString("com.amazonaws.ivs.player.FRAMEWORK_NAME");
                    String version = appInfo.metaData.getString("com.amazonaws.ivs.player.FRAMEWORK_VERSION");
                    if (framework != null && !framework.isEmpty()) {
                        this.fixedDeviceProperties.put("player_framework", framework);
                    }
                    if (version != null && !version.isEmpty()) {
                        this.fixedDeviceProperties.put("player_framework_version", version);
                    }
                }
            }
            catch (PackageManager.NameNotFoundException nnfe) {
                Logging.d("Unable to find package", nnfe);
            }
            IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
            this.batteryStatus = context.registerReceiver(null, filter);
            this.batteryManager = (BatteryManager)context.getSystemService("batterymanager");
            this.powerManager = (PowerManager)context.getSystemService("power");
        }
    }

    private String getDeviceDiagonalSize() {
        WindowManager wm = (WindowManager)this.context.getSystemService("window");
        if (wm != null) {
            DisplayMetrics dm = new DisplayMetrics();
            wm.getDefaultDisplay().getRealMetrics(dm);
            double widthInches = (float)dm.widthPixels / dm.xdpi;
            double heightInches = (float)dm.heightPixels / dm.ydpi;
            double sizeInches = Math.sqrt(Math.pow(widthInches, 2.0) + Math.pow(heightInches, 2.0));
            BigDecimal bd = new BigDecimal(sizeInches).setScale(1, 4);
            return bd.toString();
        }
        return "0.0";
    }

    public static void setErrorListener(ErrorListener listener) {
        errorListener = listener;
    }

    MediaDecoder createDecoder(MediaFormat format) {
        MediaCodecDecoder decoder = null;
        String mime = format.getString("mime");
        if (mime.startsWith("video")) {
            if (this.surfaceRenderer == null) {
                VideoRenderer.VideoSizeListener listener = this.videoSizeListener != null ? (VideoRenderer.VideoSizeListener)this.videoSizeListener.get() : null;
                this.surfaceRenderer = new SurfaceRenderer(this.context, listener);
            }
            this.videoMediaType = mime;
            return new MediaCodecDecoder(format, this.surfaceRenderer);
        }
        if (mime.startsWith("audio")) {
            if (this.audioRenderer != null) {
                this.audioRenderer.release();
            }
            this.audioRenderer = new AudioTrackRenderer();
            decoder = new MediaCodecDecoder(format, this.audioRenderer);
        }
        return decoder;
    }

    synchronized MediaRenderer createRenderer(MediaFormat format) {
        String mime = format.getString("mime");
        if (mime.startsWith("audio")) {
            return this.audioRenderer;
        }
        if (mime.startsWith("video")) {
            return this.surfaceRenderer;
        }
        return null;
    }

    PlayerException createException(String source, int result, int code, String message) {
        PlayerException exception;
        ErrorType mediaResult = ErrorType.fromInt(result);
        Throwable cause = this.pendingExceptions.poll();
        if (cause != null) {
            exception = new PlayerException(cause, source, mediaResult, code, message);
            if (this.pendingExceptions.size() > 1) {
                Logging.e("Errored with multiple exceptions");
                for (Throwable t : this.pendingExceptions) {
                    Logging.e(source, t);
                }
            }
            this.pendingExceptions.clear();
        } else {
            exception = new PlayerException(source, mediaResult, code, message);
        }
        return exception;
    }

    int getAudioSessionId() {
        return this.audioRenderer == null ? -1 : this.audioRenderer.getAudioSessionId();
    }

    HttpClient createHttpClient() {
        return HttpClientFactory.create(this.context);
    }

    NetworkLinkInfo createNetworkLinkInfo() {
        return new NetworkLinkInfo(this.context);
    }

    static ByteBuffer[] getSupportedProtectionSystemUUIDs() {
        ProtectionSystem[] schemes = ProtectionSystem.getSupported().toArray(new ProtectionSystem[0]);
        ByteBuffer[] result = new ByteBuffer[schemes.length];
        for (int i = 0; i < schemes.length; ++i) {
            ByteBuffer buffer = ByteBuffer.allocateDirect(16);
            UUID id = schemes[i].getUUID();
            buffer.putLong(id.getMostSignificantBits());
            buffer.putLong(id.getLeastSignificantBits());
            result[i] = buffer;
        }
        return result;
    }

    VideoCapabilities getVideoDecoderCapabilities(String mediaType) {
        return VideoCapabilities.getCapabilities(MediaType.removeParameters(mediaType));
    }

    public void handleDecoderException(Throwable t) {
        if (t != null) {
            Logging.e("Decoder Error", t);
            this.pendingExceptions.add(t);
            if (errorListener != null) {
                errorListener.onError(t);
            }
        }
    }

    public void handleRendererException(Throwable t) {
        if (t != null) {
            Logging.e("Renderer Error", t);
            this.pendingExceptions.add(t);
            if (errorListener != null) {
                errorListener.onError(t);
            }
        }
    }

    public void onThreadCreated(String name) {
        try {
            Process.setThreadPriority((int)-2);
        }
        catch (Exception e) {
            Logging.e("Error while setting thread priority", e);
        }
        if (name != null && !name.isEmpty()) {
            Thread.currentThread().setName(name);
        }
    }

    private void updateDeviceDisplayMode() {
        Size newSize;
        WindowManager wm = (WindowManager)this.context.getSystemService("window");
        int width = 0;
        int height = 0;
        if (wm != null) {
            if (Build.VERSION.SDK_INT >= 23) {
                Display.Mode[] modes;
                for (Display.Mode mode : modes = wm.getDefaultDisplay().getSupportedModes()) {
                    int modeHeight;
                    int modeWidth = mode.getPhysicalWidth();
                    if (modeWidth * (modeHeight = mode.getPhysicalHeight()) <= width * height) continue;
                    width = modeWidth;
                    height = modeHeight;
                }
            } else {
                DisplayMetrics displayMetrics = new DisplayMetrics();
                wm.getDefaultDisplay().getRealMetrics(displayMetrics);
                width = displayMetrics.widthPixels;
                height = displayMetrics.heightPixels;
            }
        }
        if (width > 0 && height > 0 && !(newSize = new Size(width, height)).equals(this.maxDisplayModeSize)) {
            this.maxDisplayModeSize = newSize;
        }
    }

    Size getDisplaySize() {
        this.updateDeviceDisplayMode();
        return this.maxDisplayModeSize;
    }

    float getMaxPlaybackRateForQuality(Quality quality) {
        if (this.videoMediaType == null) {
            Logging.d("getMaxPlaybackRateForQuality: decoder has not been created yet");
            return 1.0f;
        }
        return VideoCapabilities.getMaxPlaybackRateForQuality(MediaType.removeParameters(this.videoMediaType), quality);
    }

    String getThermalState() {
        int status = -1;
        if (Build.VERSION.SDK_INT >= 29 && this.powerManager != null) {
            status = this.powerManager.getCurrentThermalStatus();
        }
        return ThermalState.valueOf(status).toString();
    }

    Integer getBatteryCharge() {
        return this.batteryManager != null ? Integer.valueOf(this.batteryManager.getIntProperty(4)) : null;
    }

    Boolean isBatteryCharging() {
        int status = 1;
        status = Build.VERSION.SDK_INT >= 26 && this.batteryManager != null ? this.batteryManager.getIntProperty(6) : this.batteryStatus.getIntExtra("status", 1);
        if (status == 1) {
            return null;
        }
        return status == 2 || status == 5;
    }

    Boolean isLowPowerModeEnabled() {
        return this.powerManager != null ? Boolean.valueOf(this.powerManager.isPowerSaveMode()) : null;
    }

    Map<String, String> getMutableProperties() {
        Integer batteryCharge;
        Boolean isCharging;
        HashMap<String, String> mutableDevicePropertiesMap = new HashMap<String, String>();
        Boolean isLowPowerModeEnabled = this.isLowPowerModeEnabled();
        if (isLowPowerModeEnabled != null) {
            mutableDevicePropertiesMap.put("low_power_mode", String.valueOf(isLowPowerModeEnabled));
        }
        if ((isCharging = this.isBatteryCharging()) != null) {
            mutableDevicePropertiesMap.put("charging", String.valueOf(isCharging));
        }
        if ((batteryCharge = this.getBatteryCharge()) != null) {
            float batteryPercent = (float)batteryCharge.intValue() / 100.0f;
            mutableDevicePropertiesMap.put("battery_percent", String.valueOf(batteryPercent));
        }
        mutableDevicePropertiesMap.put("thermal_state", this.getThermalState());
        return mutableDevicePropertiesMap;
    }

    Map<String, String> getDeviceProperties() {
        HashMap<String, String> allProperties = new HashMap<String, String>(this.fixedDeviceProperties);
        allProperties.putAll(this.getMutableProperties());
        return allProperties;
    }

    String getStringOrientation() {
        switch (this.context.getResources().getConfiguration().orientation) {
            case 2: {
                return "landscape";
            }
            case 1: {
                return "portrait";
            }
        }
        return "unknown";
    }

    public static interface ErrorListener {
        public void onError(Throwable var1);
    }

    private static enum ThermalState {
        NONE(0),
        LIGHT(1),
        MODERATE(2),
        SEVERE(3),
        CRITICAL(4),
        EMERGENCY(5),
        SHUTDOWN(6),
        UNKNOWN(-1);

        private final int id;

        private ThermalState(int id) {
            this.id = id;
        }

        static ThermalState valueOf(int id) {
            for (ThermalState state : ThermalState.values()) {
                if (state.id != id) continue;
                return state;
            }
            return UNKNOWN;
        }

        @NonNull
        public String toString() {
            return this.name().toLowerCase();
        }
    }
}

