package common.base;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.util.Log;

import java.util.Locale;

import static android.util.Log.VERBOSE;
import static android.util.Log.DEBUG;
import static android.util.Log.INFO;
import static android.util.Log.WARN;
import static android.util.Log.ERROR;
import static android.util.Log.ASSERT;

/**
 * Created by apolunor on 16/2/19.
 */
public final class LogHelper {

    public interface Printer {

        Printer tag(String tag);

        void v(String message, Object... args);

        void d(String message, Object... args);

        void i(String message, Object... args);

        void e(String message, Object... args);

        void wtf(String message, Object... args);

        void w(String message, Object... args);

    }

    private static Printer sPrinter;
    private static String sTag;
    private static int sLevel = 6;

    private LogHelper() {
    }

    static void initialize(@NonNull Printer printer,
                           @NonNull String tag,
                           @IntRange(from = VERBOSE, to = ASSERT) int level) {
        sPrinter = printer;
        sTag = tag;
        sLevel = level;
    }

    public static Printer tag(String tag) {
        return sPrinter.tag(tag);
    }

    public static void d(String message, Object... args) {
        sPrinter.d(message, args);
    }

    public static void e(String message, Object... args) {
        sPrinter.e(null, message, args);
    }

    public static void wtf(String message, Object... args) {
        sPrinter.wtf(message, args);
    }

    public static void i(String message, Object... args) {
        sPrinter.i(message, args);
    }

    public static void v(String message, Object... args) {
        sPrinter.v(message, args);
    }

    public static void w(String message, Object... args) {
        sPrinter.w(message, args);
    }

    static class Default implements Printer {

        private final ThreadLocal<String> localTag = new ThreadLocal<>();

        @Override
        public Printer tag(String tag) {
            if (tag != null) {
                localTag.set(tag);
            }
            return this;
        }

        @Override
        public void v(String message, Object... args) {
            impl(VERBOSE, message, args);
        }

        @Override
        public void d(String message, Object... args) {
            impl(DEBUG, message, args);
        }

        @Override
        public void i(String message, Object... args) {
            impl(INFO, message, args);
        }

        @Override
        public void w(String message, Object... args) {
            impl(WARN, message, args);
        }

        @Override
        public void e(String message, Object... args) {
            impl(ERROR, message, args);
        }

        public void wtf(String message, Object... args) {
            impl(ASSERT, message, args);
        }

        private void impl(int level, String message, Object... args) {
            if (level < sLevel) {
                return;
            }

            switch (level) {
                case ERROR:
                    Log.e(getTag(), buildMessage(message, args));
                    break;
                case ASSERT:
                    Log.wtf(getTag(), buildMessage(message, args));
                    break;
                case INFO:
                    Log.i(getTag(), buildMessage(message, args));
                    break;
                case VERBOSE:
                    Log.v(getTag(), buildMessage(message, args));
                    break;
                case WARN:
                    Log.w(getTag(), buildMessage(message, args));
                    break;
                case DEBUG:
                    // Fall through, log debug by default
                default:
                    Log.d(getTag(), buildMessage(message, args));
                    break;
            }
        }

        private String getTag() {
            String tag = localTag.get();
            if (tag != null) {
                localTag.remove();
                return tag;
            }
            return sTag;
        }

        private static String buildMessage(String message, Object... args) {
            String msg = (args == null) ? message : String.format(Locale.US, message, args);
            StackTraceElement[] trace = new Throwable().fillInStackTrace().getStackTrace();

            String caller = "<unknown>";
            // Walk up the stack looking for the first caller outside of VolleyLog.
            // It will be at least two frames up, so start there.
            for (int i = 2; i < trace.length; i++) {
                Class<?> clazz = trace[i].getClass();
                if (!clazz.equals(Default.class)) {
                    String callingClass = trace[i].getClassName();
                    callingClass = callingClass.substring(callingClass.lastIndexOf('.') + 1);
                    callingClass = callingClass.substring(callingClass.lastIndexOf('$') + 1);

                    caller = callingClass + "." + trace[i].getMethodName();
                    break;
                }
            }
            return String.format(Locale.US, "[%d] %s: %s",
                    Thread.currentThread().getId(), caller, msg);
        }

    }
}
