package com.applitools.eyes;

import com.applitools.utils.ArgumentGuard;
import com.applitools.utils.ImageUtils;
import io.appium.java_client.AppiumDriver;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.ios.IOSDriver;
import io.appium.java_client.remote.MobileCapabilityType;
import org.openqa.selenium.*;
import org.openqa.selenium.interactions.*;
import org.openqa.selenium.internal.*;
import org.openqa.selenium.remote.*;

import java.awt.image.BufferedImage;
import java.util.*;

/**
 * An Eyes implementation of the interfaces implemented by
 * {@link org.openqa.selenium.remote.RemoteWebDriver}.
 * Used so we'll be able to return the users an object with the same
 * functionality as {@link org.openqa.selenium.remote.RemoteWebDriver}.
 */
public class EyesWebDriver implements HasCapabilities, HasInputDevices,
        FindsByClassName, FindsByCssSelector, FindsById, FindsByLinkText,
        FindsByName, FindsByTagName, FindsByXPath, JavascriptExecutor,
        SearchContext, TakesScreenshot, WebDriver, HasTouchScreen {

    // See Applitools WiKi for explanation.
    private static final String JS_GET_VIEWPORT_WIDTH =
            "var width = undefined;" +
            " if (window.innerWidth) {width = window.innerWidth;}" +
            " else if (document.documentElement " +
                    "&& document.documentElement.clientWidth) " +
                        "{width = document.documentElement.clientWidth;}" +
            " else { var b = document.getElementsByTagName('body')[0]; " +
                    "if (b.clientWidth) {" +
                        "width = b.clientWidth;}" +
                    "};" +
            "return width;";

    private static final String JS_GET_VIEWPORT_HEIGHT =
            "var height = undefined;" +
            "  if (window.innerHeight) {height = window.innerHeight;}" +
            "  else if (document.documentElement " +
                    "&& document.documentElement.clientHeight) " +
                        "{height = document.documentElement.clientHeight;}" +
            "  else { var b = document.getElementsByTagName('body')[0]; " +
                        "if (b.clientHeight) {height = b.clientHeight;}" +
                    "};" +
            "return height;";

    private final Logger logger;
    private final Eyes eyes;
    private final RemoteWebDriver driver;
    private final TouchScreen touch;
    private final Map<String, WebElement> elementsIds;
    private final FrameChain frameChain;
    private ImageRotation rotation;

    /**
     * Rotates the image as necessary. The rotation is either manually forced
     * by passing a non-null ImageRotation, or automatically inferred.
     *
     * @param driver The driver which produced the screenshot.
     * @param image The image to normalize.
     * @param rotation The degrees by which to rotate the image:
     *                 positive values = clockwise rotation,
     *                 negative values = counter-clockwise,
     *                 0 = force no rotation, null = rotate automatically
     *                 when needed.
     * @return A normalized image.
     */
    public static BufferedImage normalizeRotation(EyesWebDriver driver,
                                                  BufferedImage image,
                                                  ImageRotation rotation) {
        ArgumentGuard.notNull(driver, "driver");
        ArgumentGuard.notNull(image, "image");
        BufferedImage normalizedImage = image;
        if (rotation != null) {
            if (rotation.getRotation() != 0) {
                normalizedImage = ImageUtils.rotateImage(image,
                        rotation.getRotation());
            }
        } else { // Do automatic rotation if necessary
            if (driver.isMobileDevice() && driver.isLandscapeOrientation() &&
                    image.getHeight() > image.getWidth()) {
                // For Android, we need to rotate images to the right, and for
                // iOS to the left.
                int degrees = driver.isAndroid() ? 90 : -90;
                normalizedImage = ImageUtils.rotateImage(image, degrees);
            }
        }

        return normalizedImage;
    }

    public EyesWebDriver(Logger logger, Eyes eyes, RemoteWebDriver driver)
            throws EyesException {
        ArgumentGuard.notNull(logger, "logger");
        ArgumentGuard.notNull(eyes, "eyes");
        ArgumentGuard.notNull(driver, "driver");

        this.logger = logger;
        this.eyes = eyes;
        this.driver = driver;
        elementsIds = new HashMap<String, WebElement>();
        this.frameChain = new FrameChain(logger);

        // initializing "touch" if possible
        ExecuteMethod executeMethod = null;
        try {
            executeMethod = new RemoteExecuteMethod(driver);
        } catch (Exception e) {
            // If an exception occurred, we simply won't instantiate "touch".
        }
        if (null != executeMethod) {
            touch = new EyesTouchScreen(logger, this,
                    new RemoteTouchScreen(executeMethod));
        } else {
            touch = null;
        }

        logger.verbose("EyesWebDriver(): Driver session is " + getSessionId());
    }

    public Eyes getEyes() {
        return eyes;
    }

    @SuppressWarnings("UnusedDeclaration")
    public RemoteWebDriver getRemoteWebDriver() {
        return driver;
    }

    public TouchScreen getTouch() {
        return touch;
    }

    /**
     *
     * @return The image rotation data.
     */
    public ImageRotation getRotation() {
        return rotation;
    }

    /**
     *
     * @param rotation The image rotation data.
     */
    public void setRotation(ImageRotation rotation) {
        this.rotation = rotation;
    }

    /**
     *
     * @return {@code true} if the platform running the test is a mobile
     * platform. {@code false} otherwise.
     */
    public boolean isMobileDevice() {
        return driver instanceof AppiumDriver;
    }

    /**
     *
     * @return {@code true} if the driver is an Android driver.
     * {@code false} otherwise.
     */
    public boolean isAndroid() {
        return driver instanceof AndroidDriver;
    }

    /**
     *
     * @return {@code true} if the driver is an iOS driver.
     * {@code false} otherwise.
     */
    public boolean isIOS() {
        return driver instanceof IOSDriver;
    }

    /**
     *
     * @return {@code true} if the
     */
    public boolean isLandscapeOrientation() {
        if (driver instanceof AppiumDriver) {
            try {
                return ((AppiumDriver) driver).getOrientation() ==
                        ScreenOrientation.LANDSCAPE;
            } catch (WebDriverException e) {
                // Ignored. Some drivers have no 'orientation' attribute, and
                // that's fine.
            }
        }
        logger.verbose(
                "driver has no 'orientation' attribute. Assuming Portrait.");
        return false;
    }

    /**
     *
     * @return The plaform version or {@code null} if it is undefined.
     */
    public String getPlatformVersion() {
        Capabilities capabilities = getCapabilities();
        Object platformVersionObj =
                capabilities.getCapability
                        (MobileCapabilityType.PLATFORM_VERSION);

        return platformVersionObj == null ?
                 null : String.valueOf(platformVersionObj);
    }

    public void get(String s) {
        frameChain.clear();
        driver.get(s);
    }

    public String getCurrentUrl() {
        return driver.getCurrentUrl();
    }

    public String getTitle() {
        return driver.getTitle();
    }

    public List<WebElement> findElements(By by) {
        List<WebElement> foundWebElementsList = driver.findElements(by);

        // This list will contain the found elements wrapped with our class.
        List<WebElement> resultElementsList =
                new ArrayList<WebElement>(foundWebElementsList.size());

        for (WebElement currentElement : foundWebElementsList) {
            if (currentElement instanceof RemoteWebElement) {
                resultElementsList.add(new EyesRemoteWebElement(logger, this,
                        (RemoteWebElement) currentElement));

                // For Remote web elements, we can keep the IDs
                elementsIds.put(((RemoteWebElement) currentElement).getId(),
                        currentElement);

            } else {
                throw new EyesException(String.format(
                        "findElements: element is not a RemoteWebElement: %s",
                        by));
            }
        }

        return resultElementsList;
    }

    public WebElement findElement(By by) {
        WebElement webElement = driver.findElement(by);
        if (webElement instanceof RemoteWebElement) {
            webElement = new EyesRemoteWebElement(logger, this,
                    (RemoteWebElement) webElement);

            // For Remote web elements, we can keep the IDs,
            // for Id based lookup (mainly used for Javascript related
            // activities).
            elementsIds.put(((RemoteWebElement) webElement).getId(),
                    webElement);
        } else {
            throw new EyesException(String.format(
                    "findElement: Element is not a RemoteWebElement: %s", by));
        }

        return webElement;
    }

    /**
     * Found elements are sometimes accessed by their IDs (e.g. tapping an
     * element in Appium).
     * @return Maps of IDs for found elements.
     */
    @SuppressWarnings("UnusedDeclaration")
    public Map<String, WebElement> getElementIds () {
        return elementsIds;
    }

    public String getPageSource() {
        return driver.getPageSource();
    }

    public void close() {
        driver.close();
    }

    public void quit() {
        driver.quit();
    }

    public Set<String> getWindowHandles() {
        return driver.getWindowHandles();
    }

    public String getWindowHandle() {
        return driver.getWindowHandle();
    }

    public TargetLocator switchTo() {
        logger.verbose("switchTo()");
        return new EyesTargetLocator(logger, this, driver.switchTo(),
                new EyesTargetLocator.OnWillSwitch() {
                    public void willSwitchToFrame(
                            EyesTargetLocator.TargetType targetType,
                            WebElement targetFrame) {
                        logger.verbose("willSwitchToFrame()");
                        switch(targetType) {
                            case DEFAULT_CONTENT:
                                logger.verbose("Default content.");
                                frameChain.clear();
                                break;
                            case PARENT_FRAME:
                                logger.verbose("Parent frame.");
                                frameChain.pop();
                                break;
                            default: // Switching into a frame
                                logger.verbose("Frame");

                                String frameId = ((EyesRemoteWebElement)
                                        targetFrame).getId();
                                Point pl = targetFrame.getLocation();
                                Dimension ds = targetFrame.getSize();
                                frameChain.push(new Frame(logger, targetFrame,
                                        frameId,
                                        new Location(pl.getX(), pl.getY()),
                                        new RectangleSize(ds.getWidth(),
                                                ds.getHeight()),
                                        new ScrollPositionProvider(logger,
                                                driver).getCurrentPosition()));
                        }
                        logger.verbose("Done!");
                    }

                    public void willSwitchToWindow(String nameOrHandle) {
                        logger.verbose("willSwitchToWindow()");
                        frameChain.clear();
                        logger.verbose("Done!");
                    }
                });
    }

    public Navigation navigate() {
        return driver.navigate();
    }

    public Options manage() {
        return driver.manage();
    }

    public Mouse getMouse() {
        return new EyesMouse(logger, this,
                driver.getMouse());
    }

    public Keyboard getKeyboard() {
        return new EyesKeyboard(logger, this, driver.getKeyboard());
    }

    public WebElement findElementByClassName(String className) {
        return findElement(By.className(className));
    }

    public List<WebElement> findElementsByClassName(String className) {
        return findElements(By.className(className));
    }

    public WebElement findElementByCssSelector(String cssSelector) {
        return findElement(By.cssSelector(cssSelector));
    }

    public List<WebElement> findElementsByCssSelector(String cssSelector) {
        return findElements(By.cssSelector(cssSelector));
    }

    public WebElement findElementById(String id) {
        return findElement(By.id(id));
    }

    public List<WebElement> findElementsById(String id) {
        return findElements(By.id(id));
    }

    public WebElement findElementByLinkText(String linkText) {
        return findElement(By.linkText(linkText));
    }

    public List<WebElement> findElementsByLinkText(String linkText) {
        return findElements(By.linkText(linkText));
    }

    public WebElement findElementByPartialLinkText(String partialLinkText) {
        return findElement(By.partialLinkText(partialLinkText));
    }

    public List<WebElement> findElementsByPartialLinkText(String
                                                                  partialLinkText) {
        return findElements(By.partialLinkText(partialLinkText));
    }

    public WebElement findElementByName(String name) {
        return findElement(By.name(name));
    }

    public List<WebElement> findElementsByName(String name) {
        return findElements(By.name(name));
    }

    public WebElement findElementByTagName(String tagName) {
        return findElement(By.tagName(tagName));
    }

    public List<WebElement> findElementsByTagName(String tagName) {
        return findElements(By.tagName(tagName));
    }

    public WebElement findElementByXPath(String path) {
        return findElement(By.xpath(path));
    }

    public List<WebElement> findElementsByXPath(String path) {
        return findElements(By.xpath(path));
    }

    public Capabilities getCapabilities() {
        return driver.getCapabilities();
    }

    public Object executeScript(String script, Object... args) {

        // Appium commands are sometimes sent as Javascript
        if (AppiumJsCommandExtractor.isAppiumJsCommand(script)) {
            Trigger trigger =
                    AppiumJsCommandExtractor.extractTrigger(elementsIds,
                            driver.manage().window().getSize(), script, args);

            if (trigger != null) {
                // TODO - Daniel, additional type of triggers
                if (trigger instanceof MouseTrigger) {
                    MouseTrigger mt = (MouseTrigger) trigger;
                    eyes.addMouseTrigger(mt.getMouseAction(),
                            mt.getControl(), mt.getLocation());
                }
            }
        }
        logger.verbose("Execute script...");
        Object result = driver.executeScript(script, args);
        logger.verbose("Done!");
        return result;
    }

    public Object executeAsyncScript(String script, Object... args) {

        // Appium commands are sometimes sent as Javascript
        if (AppiumJsCommandExtractor.isAppiumJsCommand(script)) {
            Trigger trigger =
                    AppiumJsCommandExtractor.extractTrigger(elementsIds,
                            driver.manage().window().getSize(), script, args);

            if (trigger != null) {
                // TODO - Daniel, additional type of triggers
                if (trigger instanceof MouseTrigger) {
                    MouseTrigger mt = (MouseTrigger) trigger;
                    eyes.addMouseTrigger(mt.getMouseAction(),
                            mt.getControl(), mt.getLocation());
                }
            }
        }

        return driver.executeAsyncScript(script, args);
    }

    /**
     * Sets the overflow of the current context's document element
     * @param value The overflow value to set.
     * @return The previous overflow value (could be {@code null} if undefined).
     */
    public String setOverflow(String value) {
        logger.verbose("setOverflow()");
        String script;
        if (value == null) {
            script =
                "var origOverflow = document.documentElement.style.overflow; " +
                "document.documentElement.style.overflow = undefined; " +
                "return origOverflow";
        } else {
            script = String.format(
                "var origOverflow = document.documentElement.style.overflow; " +
                        "document.documentElement.style.overflow = \"%s\"; " +
                        "return origOverflow",
                value);
        }
        String originalOverflow = (String) executeScript(script);

        logger.verbose("Done!");
        return originalOverflow;
    }

    /**
     * Hides the scrollbars of the current context's document element.
     * @return The previous value of the overflow property (could be
     *          {@code null}).
     */
    public String hideScrollbars() {
        return setOverflow("hidden");
    }

    protected int extractViewportWidth() {
        logger.verbose("extractViewportWidth()");
        int viewportWidth = Integer.parseInt(
                executeScript(JS_GET_VIEWPORT_WIDTH).toString()
        );
        logger.verbose("Done!");
        return viewportWidth;
    }

    protected int extractViewportHeight() {
        logger.verbose("extractViewportHeight()");
        int result = Integer.parseInt(
                executeScript(JS_GET_VIEWPORT_HEIGHT).toString()
        );
        logger.verbose("Done!");
        return result;
    }

    /**
     *
     * @return The viewport size of the default content (outer most frame).
     */
    public RectangleSize getDefaultContentViewportSize() {
        logger.verbose("getDefaultContentViewportSize()");
        RectangleSize viewportSize;
        FrameChain currentFrames = new FrameChain(logger, frameChain);
        switchTo().defaultContent();
        try {
            logger.verbose("Getting viewport size...");
            viewportSize = new RectangleSize(extractViewportWidth(),
                    extractViewportHeight());
            logger.verbose("Done!");
        } catch (Exception e) {
            // There are platforms for which we can't extract the viewport size
            // (e.g. Appium)
            logger.verbose(
                    "Can't get viewport size, using window size instead..");
            Dimension windowSize = manage().window().getSize();
            viewportSize = new RectangleSize(windowSize.getWidth(),
                    windowSize.getHeight());
        }
        ((EyesTargetLocator) switchTo()).frames(currentFrames);
        return viewportSize;
    }

    /**
     *
     * @return A copy of the current frame chain.
     */
    public FrameChain getFrameChain() {
        return new FrameChain(logger, frameChain);
    }



    /**
     * Creates a full page image by scrolling the viewport and "stitching"
     * the screenshots to each other.
     *
     * @return The image of the entire page.
     */
    public BufferedImage getFullPageScreenshot() {
        logger.verbose("Getting full page screenshot..");

        // Save the current frame path.
        FrameChain originalFrame = getFrameChain();

        switchTo().defaultContent();

        FullPageCaptureAlgorithm algo = new FullPageCaptureAlgorithm(logger,
                this);
        BufferedImage fullPageImage = algo.getStitchedRegion(
                new RegionProvider() {
                    public Region getRegion() {
                        return Region.EMPTY;
                    }

                    public CoordinatesType getCoordinatesType() {
                        return null;
                    }
                },
                new ScrollPositionProvider(logger, this));

        ((EyesTargetLocator)switchTo()).frames(originalFrame);

        return fullPageImage;
    }

    public <X> X getScreenshotAs(OutputType<X> xOutputType)
            throws WebDriverException {
        // Get the image as base64.
        String screenshot64 = driver.getScreenshotAs(OutputType.BASE64);
        BufferedImage screenshot = ImageUtils.imageFromBase64(screenshot64);
        screenshot = normalizeRotation(this, screenshot, rotation);

        // Return the image in the requested format.
        screenshot64 = ImageUtils.base64FromImage(screenshot);
        return xOutputType.convertFromBase64Png(screenshot64);
    }

    public String getUserAgent() {
        String userAgent;
        try {
            userAgent = (String) this.driver.executeScript(
                    "return navigator.userAgent");
            logger.verbose("getUserAgent(): '" + userAgent + "'");
        } catch (Exception e) {
            logger.verbose("getUserAgent(): Failed to obtain user-agent string");
            userAgent = null;
        }

        return userAgent;
    }

    private String getSessionId() {
        // extract remote web driver information
        return driver.getSessionId().toString();
    }
}
