package com.applitools.eyes;

import com.applitools.utils.ArgumentGuard;
import org.openqa.selenium.remote.RemoteWebDriver;

class ScrollPositionProvider implements PositionProvider {
    protected final Logger logger;
    protected final RemoteWebDriver driver;

    public ScrollPositionProvider(Logger logger, RemoteWebDriver driver) {
        ArgumentGuard.notNull(logger, "logger");
        ArgumentGuard.notNull(driver, "driver");

        this.logger = logger;
        this.driver = driver;
    }

    public ScrollPositionProvider(Logger logger, EyesWebDriver driver) {
        this(logger, driver.getRemoteWebDriver());
    }

    /**
     * @return The scroll position of the current frame.
     */
    public Location getCurrentPosition() {
        logger.verbose("getCurrentScrollPosition()");
        int x,y;
        Object xo, yo;

        xo = driver.executeScript("return window.scrollX");
        if (xo == null) {
            // IE
            xo = driver.executeScript("var doc = document.documentElement; var left = (window.pageXOffset || doc.scrollLeft) - (doc.clientLeft || 0); return left");
            if (xo == null) {
                throw new EyesException(
                        "Could not get left scroll position!");
            }
        }

        yo = driver.executeScript("return window.scrollY");
        if (yo == null) {
            // For IE
            yo = driver.executeScript("var doc = document.documentElement; var top = (window.pageYOffset || doc.scrollTop)  - (doc.clientTop || 0); return top");
            if (yo == null) {
                throw new EyesException(
                        "Could not get top scroll position");
            }
        }

        x = Integer.parseInt(xo.toString());
        y = Integer.parseInt(yo.toString());

        Location result = new Location(x, y);
        logger.verbose(String.format("Current position: %s", result));
        return result;
    }

    /**
     * Go to the specified location.
     * @param location The position to scroll to.
     */
    public void setPosition(Location location) {
        logger.verbose(String.format("Scrolling to %s", location));

        driver.executeScript(String.format("window.scrollTo(%d,%d)",
                location.getX(), location.getY()));
        logger.verbose("Done scrolling!");
    }

    /**
     *
     * @return The entire size of the container which the position is relative
     * to.
     */
    public RectangleSize getEntireSize() {
        logger.verbose("getEntireSize()");

        int scrollWidth =
                Integer.parseInt(driver.executeScript
                        ("return document.documentElement.scrollWidth")
                        .toString());

        int bodyScrollWidth =
                Integer.parseInt(driver.executeScript
                        ("return document.body.scrollWidth")
                        .toString());

        int totalWidth = Math.max(scrollWidth, bodyScrollWidth);

        // IMPORTANT: Notice there's a major difference between scrollWidth
        // and scrollHeight. While scrollWidth is the maximum between an
        // element's width and its content width, scrollHeight might be
        // smaller (!) than the clientHeight, which is why we take the
        // maximum between them.
        int clientHeight =
                Integer.parseInt(driver.executeScript
                        ("return document.documentElement.clientHeight")
                        .toString());
        int bodyClientHeight =
                Integer.parseInt(driver.executeScript
                        ("return document.body.clientHeight")
                        .toString());
        int scrollHeight =
                Integer.parseInt(driver.executeScript
                        ("return document.documentElement.scrollHeight")
                        .toString());
        int bodyScrollHeight =
                Integer.parseInt(driver.executeScript
                        ("return document.body.scrollHeight")
                        .toString());
        int maxDocumentElementHeight = Math.max(clientHeight, scrollHeight);
        int maxBodyHeight = Math.max(bodyClientHeight, bodyScrollHeight);
        int totalHeight = Math.max(maxDocumentElementHeight, maxBodyHeight);


        RectangleSize result = new RectangleSize(totalWidth, totalHeight);
        logger.verbose(String.format("Entire size: %s", result));
        return result;
    }
}
