package com.applovin.mediation;

import com.applovin.impl.sdk.Logger;

import java.util.List;

/**
 * This class encapsulates a key-value pair, where the key is an {@code int} and the value is a {@code List<Integer>}.
 */
public class MaxSegment
{
    private static final String TAG = "MaxSegment";

    private static final int MIN_NUMERIC_VALUE = 0;
    private static final int MAX_NUMERIC_VALUE = Integer.MAX_VALUE;

    private final int           key;
    private final List<Integer> values;

    /**
     * Initializes a new {@link MaxSegment} with the specified key and value(s).
     *
     * @param key    The key of the segment. Must be a non-negative number in the range of [0, 32000].
     * @param values The values(s) associated with the key. Each value must be a non-negative number in the range of [0, 32000].
     */
    public MaxSegment(final int key, final List<Integer> values)
    {
        this.key = key;
        this.values = values;

        checkValueWithinRange( key );

        for ( int value : values )
        {
            checkValueWithinRange( value );
        }
    }

    /**
     * @return The key of the segment. Must be a non-negative number in the range of [0, 32000].
     */
    public int getKey()
    {
        return key;
    }

    /**
     * @return The value(s) associated with the key. Each value must be a non-negative number in the range of [0, 32000].
     */
    public List<Integer> getValues()
    {
        return values;
    }

    private void checkValueWithinRange(final int value)
    {
        // Check valid case (no need to check for max value, since it can't be greater than Integer.MAX_VALUE)
        if ( MIN_NUMERIC_VALUE <= value ) return;

        // Handle invalid case
        Logger.userError( TAG, "Please ensure that the segment value entered is a non-negative number in the range of [0, " + MAX_NUMERIC_VALUE + "]: " + value );
    }

    @Override
    public String toString()
    {
        return "MaxSegment{" +
                "key=" + key +
                ", values=" + values +
                '}';
    }
}
