package com.applovin.sdk;

import android.content.Context;
import android.text.TextUtils;

import com.applovin.impl.mediation.MaxMediatedNetworkInfoImpl;
import com.applovin.impl.mediation.utils.MediationUtils;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.utils.AndroidManifest;
import com.applovin.impl.sdk.utils.CollectionUtils;
import com.applovin.impl.sdk.utils.JsonUtils;
import com.applovin.mediation.MaxMediatedNetworkInfo;
import com.applovin.mediation.MaxSegmentCollection;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import androidx.annotation.Nullable;
import lombok.val;

/**
 * This is the base class for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public final class AppLovinSdk
{
    /**
     * Current SDK version. The returned value will be in the format of "<Major>.<Minor>.<Revision>".
     */
    public static final String VERSION = getVersion();

    public static final int VERSION_CODE = getVersionCode();

    private static final String TAG = "AppLovinSdk";

    private static       AppLovinSdk instance; // default shared instance
    private static final Object      instanceLock = new Object();

    private static final Map<String, AppLovinSdk> sdkInstances     = new HashMap<>( 1 ); // deprecated
    private static final Object                   sdkInstancesLock = new Object(); // deprecated

    /**
     * Use {@link AppLovinSdk#a()} to access.
     * NOTE: This field used to be private and retrieved via reflection, but was causing crashes.
     *
     * @hide
     */
    private final CoreSdk coreSdk;

    /**
     * Listener interface to be used with {@link AppLovinSdk#initializeSdk(SdkInitializationListener)}
     */
    public interface SdkInitializationListener
    {
        void onSdkInitialized(AppLovinSdkConfiguration config);
    }

    //region High Level SDK Properties

    /**
     * Get client SDK key.
     *
     * @return Client SDK key.
     * @hide
     */
    public String getSdkKey()
    {
        return coreSdk.getSdkKey();
    }

    /**
     * Get SDK settings provided on initialization.
     *
     * @return Provided settings.
     */
    public AppLovinSdkSettings getSettings()
    {
        return coreSdk.getSettings();
    }

    /**
     * Get the SDK configuration object provided upon initialization.
     *
     * @return An instance of the SDK configuration.
     */
    public AppLovinSdkConfiguration getConfiguration()
    {
        return coreSdk.getConfiguration();
    }

    /**
     * Get the {@link com.applovin.mediation.MaxSegmentCollection} object that the SDK had been initialized with.
     */
    public MaxSegmentCollection getSegmentCollection()
    {
        return coreSdk.getSegmentCollection();
    }

    //endregion

    //region SDK Services

    /**
     * Get an instance of the AppLovin Ad service. This service is used to fetch ads from AppLovin servers.
     *
     * @return Ad service. Guaranteed not to be null.
     */
    public AppLovinAdService getAdService()
    {
        return coreSdk.getAdService();
    }

    /**
     * Get an instance of the AppLovin event service. This service is used to track post-install user events.
     *
     * @return Event service. Guaranteed not to be null.
     */
    public AppLovinEventService getEventService()
    {
        return coreSdk.getEventService();
    }

    /**
     * The CMP service, which provides direct APIs for interfacing with the Google-certified CMP installed, if any.
     *
     * @return CMP service. Guaranteed not to be {@code null}.
     */
    public AppLovinCmpService getCmpService()
    {
        return coreSdk.getCmpService();
    }

    //endregion

    //region MAX

    /**
     * Get the mediation provider that was last set using {@link AppLovinSdk#setMediationProvider(String)}, or null if none was set.
     *
     * @return The mediation provider that was last set, or null if none was set.
     */
    public String getMediationProvider()
    {
        return coreSdk.getMediationProvider();
    }

    /**
     * Returns the list of available mediation networks.
     *
     * @return List of {@link MaxMediatedNetworkInfo} objects.
     */
    public List<MaxMediatedNetworkInfo> getAvailableMediatedNetworks()
    {
        JSONArray availableMediationAdapters = MediationUtils.getAvailableMediationAdapters( coreSdk );
        List<MaxMediatedNetworkInfo> availableMediatedNetworks = new ArrayList<>( availableMediationAdapters.length() );

        for ( int i = 0; i < availableMediationAdapters.length(); i++ )
        {
            JSONObject adapter = JsonUtils.getJSONObject( availableMediationAdapters, i, null );
            availableMediatedNetworks.add( new MaxMediatedNetworkInfoImpl( adapter ) );
        }

        return availableMediatedNetworks;
    }

    /**
     * Present the mediation debugger UI.
     * This debugger tool provides the status of your integration for each third-party ad network.
     * <p>
     * Please call this method after the SDK has initialized, e.g. {@link AppLovinSdk#initializeSdk(SdkInitializationListener)}.
     */
    public void showMediationDebugger()
    {
        coreSdk.showMediationDebugger();
    }

    /**
     * Present the mediation debugger UI.
     * This debugger tool provides the status of your integration for each third-party ad network.
     * <p>
     * Please call this method after the SDK has initialized, e.g. {@link AppLovinSdk#initializeSdk(SdkInitializationListener)}.
     *
     * @param amazonAdSizes A map of the MAX Ad Unit Id to Amazon Publisher Services' {@code com.amazon.device.ads.DTBAdSize}.
     */
    public void showMediationDebugger(@Nullable final Map<String, List<?>> amazonAdSizes)
    {
        coreSdk.showMediationDebugger( amazonAdSizes );
    }

    /**
     * Present the Creative Debugger UI.
     * This debugger tool provides information for recently displayed ads.
     */
    public void showCreativeDebugger()
    {
        coreSdk.showCreativeDebugger();
    }

    //endregion

    //region SDK Initialization

    /**
     * Check whether the SDK has fully been initialized without errors and the completion callback called.
     */
    public boolean isInitialized()
    {
        return coreSdk.isEnabled();
    }

    /**
     * Get instance of AppLovin SDK that is configured in <code>AndroidManifest.xml</code>. Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:value="YOUR_SDK_KEY_HERE" android:name="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context Android application context.
     *
     * @return An instance of AppLovinSDK
     */
    public static AppLovinSdk getInstance(final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        final String sdkKey = AndroidManifest.getManifest( context ).getMetaDataString( "applovin.sdk.key", null );
        if ( sdkKey != null )
        {
            return getInstance( new AppLovinInternalSdkSettings( context ), context );
        }
        else
        {
            synchronized ( instanceLock )
            {
                if ( instance == null )
                {
                    val sdk = new CoreSdk( context );
                    val wrapper = new AppLovinSdk( sdk );
                    sdk.setWrappingSdk( wrapper );  // The public-facing interface used by the pub

                    instance = wrapper;
                }

                return instance;
            }
        }
    }

    /**
     * Initializes the SDK with the given initialization configuration and listener.
     * <p>
     * The callback will be invoked on the main thread.
     *
     * @param initializationConfiguration The configuration to initialize the SDK with.
     * @param listener                    The callback that will be run on the main thread when the SDK finishes initializing.
     */
    public void initialize(final AppLovinSdkInitializationConfiguration initializationConfiguration, @Nullable final SdkInitializationListener listener)
    {
        coreSdk.initialize( initializationConfiguration, listener );
    }

    //endregion

    //region Private Functions

    /**
     * Get the SDK version string from the Build.gradle file.
     * <p>
     * Abstracted into method to prevent compiler optimization in adapters which hardcodes references to `VERSION` and `VERSION_CODE`
     * when it should dynamically retrieve versioning info depending on current SDK installed.
     */
    private static String getVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    /**
     * Get the SDK version code from the Build.gradle file.
     * <p>
     * Abstracted into method to prevent compiler optimization in adapters which hardcodes references to `VERSION` and `VERSION_CODE`
     * when it should dynamically retrieve versioning info depending on current SDK installed.
     */
    private static int getVersionCode()
    {
        return BuildConfig.VERSION_CODE;
    }

    /**
     * Force re-initialization of all known SDKs for when user consent or "do not sell" status state has explicitly changed.
     * <p>
     * <b>Please note:</b> This method is for internal use only.
     * </p>
     *
     * @hide
     */
    static void reinitializeAll(final Boolean hasUserConsent, final Boolean doNotSell)
    {
        synchronized ( sdkInstancesLock )
        {
            for ( AppLovinSdk sdk : getInstances() )
            {
                sdk.reinitialize( hasUserConsent, doNotSell );
            }
        }
    }

    private void reinitialize(final Boolean hasUserConsent, final Boolean doNotSell)
    {
        // Re-init if we are in the old init world OR we have already initialized before
        if ( coreSdk.isLegacyInitialization().get() || coreSdk.isInitializationConfigured() )
        {
            coreSdk.reinitialize();
        }

        coreSdk.notifyPrivacySettingUpdated();

        // Fire an event if the user consent state has explicitly changed
        if ( hasUserConsent != null )
        {
            coreSdk.getLogger().i( TAG, "Toggled 'huc' to " + hasUserConsent );

            val parameters = CollectionUtils.map( "value", hasUserConsent.toString() );
            getEventService().trackEvent( "huc", parameters );
        }

        // Fire an event if the "do not sell" state has explicitly changed
        if ( doNotSell != null )
        {
            coreSdk.getLogger().i( TAG, "Toggled 'dns' to " + doNotSell );

            val parameters = CollectionUtils.map( "value", doNotSell.toString() );
            getEventService().trackEvent( "dns", parameters );
        }
    }

    private AppLovinSdk(CoreSdk sdk)
    {
        this.coreSdk = sdk;
    }

    private static Collection<AppLovinSdk> getInstances()
    {
        if ( instance != null ) return Arrays.asList( instance );

        synchronized ( sdkInstancesLock )
        {
            return sdkInstances.values();
        }
    }

    /**
     * Internal SDK settings class.
     * <p>
     * <b>Please note:</b> This class is for internal use only.
     * </p>
     *
     * @hide
     */
    private static class AppLovinInternalSdkSettings
            extends AppLovinSdkSettings
    {
        AppLovinInternalSdkSettings(final Context context) { super( context ); }
    }

    //endregion

    //region Miscellaneous Public Functions

    @Override
    public String toString()
    {
        return "AppLovinSdk{" +
                "sdkKey='" + getSdkKey() + "'" +
                ", isInitialized=" + isInitialized() +
                ", isFirstSession=" + coreSdk.isFirstSession() +
                '}';
    }

    /**
     * This returns the {@link CoreSdk} instance and should not be used by publishers.
     *
     * @return The {@link CoreSdk} instance.
     * @hide
     */
    public CoreSdk a()
    {
        return coreSdk;
    }

    //endregion

    //region Deprecated APIs

    /**
     * Set Plugin version.
     *
     * @param version Plugin version to set.
     *
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdkInitializationConfiguration.Builder#setPluginVersion(String)}
     */
    @Deprecated
    public void setPluginVersion(String version)
    {
        coreSdk.setPluginVersion( version );
    }

    /**
     * @deprecated This API has been moved to {@link AppLovinSdkSettings#getUserIdentifier()} and will be removed in a future SDK version.
     */
    @Deprecated
    public String getUserIdentifier()
    {
        return coreSdk.getUserId();
    }

    /**
     * @deprecated This API has been moved to {@link AppLovinSdkSettings#setUserIdentifier(String)} and will be removed in a future SDK version.
     */
    @Deprecated
    public void setUserIdentifier(String userIdentifier)
    {
        coreSdk.setUserId( userIdentifier );
    }

    /**
     * Set mediation provider using one of the provided strings above specified by {@link AppLovinMediationProvider}, or your own if not defined.
     *
     * @param mediationProvider The name of the mediation provider.
     */
    @Deprecated
    public void setMediationProvider(String mediationProvider)
    {
        coreSdk.setMediationProvider( mediationProvider );
    }

    /**
     * Initialize the SDK
     *
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}
     */
    @Deprecated
    public void initializeSdk()
    {
        // Syntactic sugar
    }

    /**
     * Initialize the SDK with a given listener.
     * <p>
     * The callback will be invoked on the main thread.
     *
     * @param listener The callback that will be run on the main thread when the SDK finishes initializing. May be null.
     *
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}
     */
    @Deprecated
    public void initializeSdk(SdkInitializationListener listener)
    {
        coreSdk.initializeSdk( listener );
    }

    /**
     * Initialize the default version of the SDK.
     * <p>
     * Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:name="applovin.sdk.key" android:value="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context Android application context.
     *
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}
     */
    @Deprecated
    public static void initializeSdk(Context context)
    {
        initializeSdk( context, null );
    }

    /**
     * Initialize the default version of the SDK.
     * <p>
     * Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:name="applovin.sdk.key" android:value="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context  Android application context.
     * @param listener The callback that will be run on the main thread when the SDK finishes initializing. May be null.
     *
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}
     */
    @Deprecated
    public static void initializeSdk(Context context, SdkInitializationListener listener)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        final AppLovinSdk sdk = getInstance( context );
        if ( sdk != null )
        {
            sdk.initializeSdk( listener );
        }
        else
        {
            Logger.userError( TAG, "Unable to initialize AppLovin SDK: SDK object not created" );
        }
    }

    /**
     * Get instance of AppLovin SDK that is configured in <code>AndroidManifest.xml</code>. Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:value="YOUR_SDK_KEY_HERE" android:name="applovin.sdk.key" />
     *     &lt;/application>
     * </pre>
     *
     * @param settings Settings to use with an SDK. Must not be null.
     * @param context  Android application context.
     *
     * @return An instance of AppLovinSDK
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#getInstance(Context)}.
     */
    @Deprecated
    public static AppLovinSdk getInstance(final AppLovinSdkSettings settings, final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        final String sdkKey = AndroidManifest.getManifest( context ).getMetaDataString( "applovin.sdk.key", "" );
        return getInstance( sdkKey, settings, context );
    }

    /**
     * Get an instance of AppLovin SDK.
     *
     * @param sdkKey       Client SDK's key.
     * @param userSettings User-provided settings. May be null.
     * @param context      Android application context.
     *
     * @return An instance of AppLovinSDK
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link AppLovinSdk#getInstance(Context)}.
     */
    @Deprecated
    public static AppLovinSdk getInstance(String sdkKey, AppLovinSdkSettings userSettings, Context context)
    {
        // Check input
        if ( userSettings == null ) throw new IllegalArgumentException( "No userSettings specified" );
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        synchronized ( instanceLock )
        {
            if ( instance != null && sdkKey.equals( instance.getSdkKey() ) ) return instance;
        }

        synchronized ( sdkInstancesLock )
        {
            AppLovinSdk result;
            // Check if SDK exists already
            if ( sdkInstances.containsKey( sdkKey ) )
            {
                result = sdkInstances.get( sdkKey );
            }
            else
            {
                // Hack inspired by Fanatee who entered an invalid SDK key in one of their 50+ AdMob mediation group
                // with a path separator which crashed SharedPreferences during the SDK initialization
                //
                // The exception thrown, for the curious: https://github.com/aosp-mirror/platform_frameworks_base/blob/master/core/java/android/app/ContextImpl.java#L2555
                if ( !TextUtils.isEmpty( sdkKey ) && sdkKey.contains( File.separator ) )
                {
                    final String log =
                            "\n**************************************************\n" +
                                    "INVALID SDK KEY: " + sdkKey + "\n" +
                                    "**************************************************\n";
                    Logger.userError( TAG, log );

                    // Attempt to use an SDK that already exists (with the correct SDK key)
                    if ( !sdkInstances.isEmpty() )
                    {
                        return sdkInstances.values().iterator().next();
                    }

                    // No existing SDKs... Explicitly remove the path separator to prevent crash...
                    sdkKey = sdkKey.replace( File.separator, "" );
                }

                val newImpl = new CoreSdk( context );
                newImpl.initialize( sdkKey, userSettings );

                val newSdk = new AppLovinSdk( newImpl );
                newImpl.setWrappingSdk( newSdk );

                // Attach the SDK instance to the settings object.
                userSettings.attachAppLovinSdk( newImpl );

                sdkInstances.put( sdkKey, newSdk );

                result = newSdk;
            }

            return result;
        }
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release.
     */
    @Deprecated
    public AppLovinPostbackService getPostbackService()
    {
        return coreSdk.getPostbackService();
    }

    //endregion
}
