package com.applovin.adview;

import android.content.Context;
import android.view.ViewGroup;

import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.reward.IncentivizedAdController;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdClickListener;
import com.applovin.sdk.AppLovinAdDisplayListener;
import com.applovin.sdk.AppLovinAdLoadListener;
import com.applovin.sdk.AppLovinAdRewardListener;
import com.applovin.sdk.AppLovinAdVideoPlaybackListener;
import com.applovin.sdk.AppLovinSdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.Lifecycle;

/**
 * This class shows rewarded videos to the user. These differ from regular interstitials in that they allow you to provide your user virtual currency in exchange for watching a video.
 */
public class AppLovinIncentivizedInterstitial
{
    private final static String TAG = "AppLovinIncentivizedInterstitial";

    private final IncentivizedAdController controller;

    /**
     * Create a new instance of an {@link AppLovinIncentivizedInterstitial}.
     */
    public AppLovinIncentivizedInterstitial()
    {
        this( null, AppLovinSdk.getInstance( CoreSdk.getApplicationContext() ) );
    }

    /**
     * Create a new instance of an {@link AppLovinIncentivizedInterstitial} for a given zone identifier.
     *
     * @param zoneId Zone identifier to load ads for. May be null.
     */
    public AppLovinIncentivizedInterstitial(@Nullable final String zoneId)
    {
        this( zoneId, AppLovinSdk.getInstance( CoreSdk.getApplicationContext() ) );
    }

    /**
     * Set extra info to pass to the SDK.
     *
     * @param key   Parameter key. Must not be {@code null}.
     * @param value Parameter value. May be {@code null}.
     */
    public void setExtraInfo(@NonNull final String key, @Nullable final Object value)
    {
        // Check input
        if ( key == null ) throw new IllegalArgumentException( "No key specified" );

        controller.setExtraInfo( key, value );
    }

    /**
     * The zone identifier this incentivized ad was initialized with and is loading ads for, if any.
     */
    public String getZoneId()
    {
        return controller.getZoneId();
    }

    /**
     * Pre-load an incentivized interstitial.
     * <p>
     * Calling this method will overwrite any existing cached ad. This is not a queue. Invoke once to preload, then do not invoke again until the ad has has been closed (e.g., AppLovinAdDisplayListener's adHidden callback). You may pass a null argument to preload if you intend to use the synchronous
     * ( isAdReadyToDisplay ) flow. Note that this is NOT recommended; we HIGHLY RECOMMEND you use an ad load delegate. Note that we internally try to pull down the next ad's resources before you need it. Therefore, this method may complete immediately in many circumstances.
     *
     * @param adLoadListener Ad load listener to notify. May be null..
     */
    public void preload(final AppLovinAdLoadListener adLoadListener)
    {
        if ( adLoadListener == null )
        {
            Logger.userInfo( TAG, "AppLovinAdLoadListener was null when preloading incentivized interstitials; using a listener is highly recommended." );
        }
        controller.preload( adLoadListener );
    }

    /**
     * Check if an ad is currently ready on this object. You must call preloadAndNotify in order to reach this state.
     * <p>
     * It is highly recommended that you implement an asynchronous flow (using an AppLovinAdDisplayListener with preload) rather than checking this method. This class does not contain a queue and can hold only one preloaded ad at a time. Further calls to preload will overwrite any existing ad.
     * Therefore, you should NOT simply call preload any time this method returns false; it is important to invoke only one ad load - then not invoke any further loads until the ad has been closed (e.g., AppLovinAdDisplayListener's adHidden callback).
     *
     * @return True if an ad has been loaded into this incentivized interstitial and is ready to display. False otherwise.
     */
    public boolean isAdReadyToDisplay()
    {
        return controller.isAdReady();
    }

    /**
     * Show a rewarded ad for the provided {@link AppLovinAd}.
     *
     * @param ad                    Ad to show. Must not be null.
     * @param adRewardListener      Reward listener to notify. Must not be null.
     * @param videoPlaybackListener Video playback listener to notify. Must not be null.
     * @param adDisplayListener     Ad display listener to notify, if desired. May be null.
     * @param adClickListener       Ad click listener to notify, if desired. May be null.
     */
    public void show(final AppLovinAd ad,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, CoreSdk.getApplicationContext(), adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * Show a rewarded ad for the provided {@link AppLovinAd} in a container view.
     *
     * @param ad                    The ad to render into this incentivized ad. Must not be null.
     * @param containerView         ViewGroup used to show the ad in. Must not be null.
     * @param lifecycle             Lifecycle object to manage ad lifecycle events in container views. Must not be null.
     * @param adRewardListener      Reward listener to notify. Must not be null.
     * @param videoPlaybackListener Video playback listener to notify. Must not be null.
     * @param adDisplayListener     Ad display listener to notify, if desired. May be null.
     * @param adClickListener       Ad click listener to notify, if desired. May be null.
     */
    public void show(final AppLovinAd ad,
                     final ViewGroup containerView,
                     final Lifecycle lifecycle,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, containerView, lifecycle, CoreSdk.getApplicationContext(), adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final Context context)
    {
        this( AppLovinSdk.getInstance( context ) );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final AppLovinSdk sdk)
    {
        this( null, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial(String)} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final String zoneId, final AppLovinSdk sdk)
    {
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );

        controller = new IncentivizedAdController( zoneId, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final Context context)
    {
        return create( AppLovinSdk.getInstance( context ) );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final AppLovinSdk sdk)
    {
        return create( null, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial(String)} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final String zoneId, final AppLovinSdk sdk)
    {
        return new AppLovinIncentivizedInterstitial( zoneId, sdk );
    }


    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener)
    {
        show( context, adRewardListener, videoPlaybackListener, adDisplayListener, null );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        show( null, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final AppLovinAd ad,
                     final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, ViewGroup, Lifecycle, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    public void show(final AppLovinAd ad,
                     final ViewGroup containerView,
                     final Lifecycle lifecycle,
                     final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, containerView, lifecycle, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }


    @Override
    public String toString()
    {
        return "AppLovinIncentivizedInterstitial{" +
                "zoneId='" + getZoneId() + "'" +
                ", isAdReadyToDisplay=" + isAdReadyToDisplay() +
                '}';
    }
}
