package com.applovin.sdk;

import android.content.Context;

import com.applovin.impl.sdk.AppLovinSdkInitializationConfigurationImpl;
import com.applovin.mediation.MaxSegmentCollection;

import java.util.List;

import androidx.annotation.Nullable;

/**
 * This class contains configurable fields for the initialization of the AppLovin SDK.
 * To be used in @{@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}.
 * <p>
 * NOTE: This is an abstract class rather than an interface to fix the static method not found error when minimum Android SDK is <24.
 * TODO: Return to interface when our minimum Android SDK is 24.
 */
public abstract class AppLovinSdkInitializationConfiguration
{
    public static Builder builder(final String sdkKey)
    {
        return builder( sdkKey, "" );
    }

    public static Builder builder(final String sdkKey, final String axonEventKey)
    {
        return new AppLovinSdkInitializationConfigurationImpl.BuilderImpl( sdkKey, axonEventKey );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #builder(String)} instead.
     */
    @Deprecated
    public static Builder builder(final String sdkKey, final Context context)
    {
        return builder( sdkKey, "" );
    }

    /**
     * @return the SDK key for the AppLovin SDK.
     */
    @Nullable
    public abstract String getSdkKey();

    /**
     * @return
     */
    @Nullable
    public abstract String getAxonEventKey();

    /**
     * @return mediation provider.
     */
    @Nullable
    public abstract String getMediationProvider();

    /**
     * @return the plugin version for the mediation adapter or plugin.
     */
    @Nullable
    public abstract String getPluginVersion();

    /**
     * @return A readonly property of a {@link com.applovin.mediation.MaxSegmentCollection}, representing a collection of segments.
     */
    @Nullable
    public abstract MaxSegmentCollection getSegmentCollection();

    /**
     * @return the list of advertising identifiers that will receive test ads.
     */
    public abstract List<String> getTestDeviceAdvertisingIds();

    /**
     * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
     */
    public abstract List<String> getAdUnitIds();

    /**
     * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    public abstract boolean isExceptionHandlerEnabled();

    /**
     * Builder class used to create an {@link AppLovinSdkInitializationConfiguration} object.
     * This class contains configurable fields for the initialization of the AppLovin SDK.
     */
    public interface Builder
    {
        /**
         * Set mediation provider using one of the provided strings above specified by {@link AppLovinMediationProvider}, or your own if not defined.
         */
        Builder setMediationProvider(@Nullable final String mediationProvider);

        /**
         * Sets the plugin version for the mediation adapter or plugin.
         */
        Builder setPluginVersion(@Nullable final String version);

        /**
         * Set the {@link MaxSegmentCollection}, a collection of segments for this instance of the SDK.
         */
        Builder setSegmentCollection(final MaxSegmentCollection segmentCollection);

        /**
         * Enable devices to receive test ads, by passing in the advertising identifier (GAID or App Set ID) of each test device.
         * Refer to AppLovin logs for the GAID or App Set ID of your current device.
         */
        Builder setTestDeviceAdvertisingIds(final List<String> testDeviceAdvertisingIds);

        /**
         * Set the MAX ad unit ids that will be used for this instance of the SDK. 3rd-party SDKs will be initialized with the credentials configured for these ad unit ids.
         */
        Builder setAdUnitIds(final List<String> adUnitIds);

        /**
         * Set whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        Builder setExceptionHandlerEnabled(final boolean exceptionHandlerEnabled);

        /**
         * Build method to create an instance of {@link AppLovinSdkInitializationConfiguration}.
         *
         * @return an instance of {@link AppLovinSdkInitializationConfiguration}.
         */
        AppLovinSdkInitializationConfiguration build();

        /**
         * @return the SDK key for the AppLovin SDK.
         */
        String getSdkKey();

        /**
         * @return mediation provider.
         */
        @Nullable
        String getMediationProvider();

        /**
         * @return the plugin version for the mediation adapter or plugin.
         */
        @Nullable
        String getPluginVersion();

        /**
         * @return the {@link MaxSegmentCollection} object.
         */
        @Nullable
        MaxSegmentCollection getSegmentCollection();

        /**
         * @return the list of advertising identifiers that will receive test ads.
         */
        List<String> getTestDeviceAdvertisingIds();

        /**
         * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
         */
        List<String> getAdUnitIds();

        /**
         * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        boolean isExceptionHandlerEnabled();
    }
}
