package com.applovin.mediation;

import lombok.val;

/**
 * Represents information for a mediated network.
 */
public interface MaxMediatedNetworkInfo
{
    /**
     * An enum describing the adapter's initialization status.
     */
    enum InitializationStatus
    {
        /**
         * The adapter is not initialized. Note: networks need to be enabled for an ad unit id to be initialized.
         */
        NOT_INITIALIZED( -4 ),

        /**
         * The 3rd-party SDK does not have an initialization callback with status.
         */
        DOES_NOT_APPLY( -3 ),

        /**
         * The 3rd-party SDK is currently initializing.
         */
        INITIALIZING( -2 ),

        /**
         * The 3rd-party SDK explicitly initialized, but without a status.
         */
        INITIALIZED_UNKNOWN( -1 ),

        /**
         * The 3rd-party SDK initialization failed.
         */
        INITIALIZED_FAILURE( 0 ),

        /**
         * The 3rd-party SDK initialization was successful.
         */
        INITIALIZED_SUCCESS( 1 );

        private final int code;

        InitializationStatus(final int code)
        {
            this.code = code;
        }

        public int getCode()
        {
            return code;
        }

        public static InitializationStatus fromCode(int code)
        {
            for ( val status : InitializationStatus.values() )
            {
                if ( status.getCode() == code )
                {
                    return status;
                }
            }

            return NOT_INITIALIZED;
        }
    }

    /**
     * @return The name of the mediated network.
     */
    String getName();

    /**
     * @return The class name of the adapter for the mediated network.
     */
    String getAdapterClassName();

    /**
     * @return The version of the adapter for the mediated network.
     */
    String getAdapterVersion();

    /**
     * @return The version of the mediated network's SDK.
     */
    String getSdkVersion();

    /**
     * @return The initialization status of the mediated network's SDK.
     */
    InitializationStatus getInitializationStatus();
}
