package com.appsflyer.migration.internal

import com.appsflyer.AFLogger
import org.json.JSONObject
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit

/**
 * This class provides mechanisms to supply attribution and deep linking data
 * to the AppsFlyer SDK. The SDK triggers the corresponding `waitFor*` methods to retrieve
 * the provided data. Additionally, the `clear` method is invoked to clean up
 * collected data when the session is about to end (e.g., when the app is sent to the background).
 *
 * This class ensures that the data is provided asynchronously and supports a timeout mechanism
 * to handle scenarios where the data is not available in a timely manner.
 *
 * Created by Taras Lozovyi on 08.01.2025
 */
object MigrationDataProvider {
    private var attributionAsyncData = AsyncData<JSONObject?>()
    private var deepLinkingAsyncData = AsyncData<JSONObject?>()

    internal fun setAttributionData(attributionData: JSONObject?) {
        attributionAsyncData.setData(attributionData)
    }

    internal fun setDeepLinkingData(deepLinkingData: JSONObject?) {
        deepLinkingAsyncData.setData(deepLinkingData)
    }

    /**
     * Waits for the attribution data to be provided.
     *
     * @param timeoutMillis The maximum time in milliseconds to wait for attribution data.
     * @return The attribution data if available, or `null` if the timeout occurs or no data is provided.
     */
    @JvmStatic
    fun waitForAttributionData(timeoutMillis: Long): JSONObject? {
        return attributionAsyncData.waitForData(timeoutMillis)
    }

    /**
     * @return The attribution data if available, or `null` if no data is provided.
     */
    @JvmStatic
    fun getAttributionData(): JSONObject? {
        return attributionAsyncData.waitForData(0)
    }

    /**
     * Waits for the deep linking data to be provided.
     *
     * @param timeoutMillis The maximum time in milliseconds to wait for deep linking data.
     * @return The deep linking data if available, or `null` if the timeout occurs or no data is provided.
     */
    @JvmStatic
    fun waitForDeepLinkingData(timeoutMillis: Long): JSONObject? {
        return deepLinkingAsyncData.waitForData(timeoutMillis)
    }

    /**
     * Clears all collected data.
     * This method is typically called when the session is about to end (e.g., when the app goes to the background).
     */
    @JvmStatic
    fun clear() {
        attributionAsyncData.clear()
        deepLinkingAsyncData.clear()
    }
}

/**
 * A helper class to manage asynchronous data provisioning with support for a timeout mechanism.
 *
 * This class is used internally by the `MigrationDataProvider` to handle the storage,
 * retrieval, and clearing of migration data in a thread-safe manner.
 *
 * @param <T> The type of data managed by this class.
 */
internal class AsyncData<T> {
    private var value: T? = null
    private var latch: CountDownLatch? = null

    /**
     * Sets the provided data.
     *
     * @param data The data to store. If `null`, it indicates that no data is available.
     */
    internal fun setData(data: T) {
        synchronized(this) {
            value = data
            latch?.countDown()
        }
    }

    /**
     * Waits for the data to be provided within the specified timeout period if timeout > 0.
     * Otherwise just returns the value
     *
     * @param timeoutMillis The maximum time in milliseconds to wait for the data.
     * @return The data if available, or `null` if the timeout occurs or no data is provided.
     */
    internal fun waitForData(timeoutMillis: Long): T? {
        if (timeoutMillis > 0) {
            synchronized(this) {
                if (value != null) {
                    return value
                }
                latch = CountDownLatch(1)
            }

            val isSuccess = latch?.await(timeoutMillis, TimeUnit.MILLISECONDS) ?: false
            latch = null

            if (!isSuccess) {
                AFLogger.afInfoLog("Timeout occurred while waiting for migration data")
            }
        }
        return value
    }

    /**
     * Clears the stored data and resets the state.
     */
    internal fun clear() {
        value = null
        latch = null
    }
}
