/*
 * DISCLAIMER
 *
 * Copyright 2016 ArangoDB GmbH, Cologne, Germany
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright holder is ArangoDB GmbH, Cologne, Germany
 */

package com.arangodb;

import com.arangodb.entity.ErrorEntity;

import java.util.Objects;
import java.util.concurrent.CompletionException;

/**
 * @author Mark Vollmary
 */
public class ArangoDBException extends RuntimeException {

    private static final long serialVersionUID = 6165638002614173801L;
    private final ErrorEntity entity;
    private final Integer responseCode;
    private final Long requestId;

    public ArangoDBException(final ErrorEntity errorEntity) {
        super(String.format("Response: %s, Error: %s - %s", errorEntity.getCode(), errorEntity.getErrorNum(),
                errorEntity.getErrorMessage()));
        this.entity = errorEntity;
        this.responseCode = errorEntity.getCode();
        this.requestId = null;
    }

    public ArangoDBException(final String message) {
        super(message);
        this.entity = null;
        this.responseCode = null;
        this.requestId = null;
    }

    public ArangoDBException(final String message, final Integer responseCode) {
        super(message);
        this.entity = null;
        this.responseCode = responseCode;
        this.requestId = null;
    }

    /**
     * @deprecated use {@link com.arangodb.ArangoDBException#of(java.lang.Throwable)} instead
     */
    @Deprecated
    public ArangoDBException(final Throwable cause) {
        super(cause);
        this.entity = null;
        this.responseCode = null;
        this.requestId = null;
    }

    /**
     * @deprecated use {@link com.arangodb.ArangoDBException#of(String, Throwable)} instead
     */
    @Deprecated
    public ArangoDBException(final String message, final Throwable cause) {
        super(message, cause);
        this.entity = null;
        this.responseCode = null;
        this.requestId = null;
    }

    /**
     * @deprecated use {@link com.arangodb.ArangoDBException#of(Throwable, Long)} instead
     */
    @Deprecated
    public ArangoDBException(Throwable cause, long requestId) {
        super(cause);
        this.entity = null;
        this.responseCode = null;
        this.requestId = requestId;
    }

    private ArangoDBException(
            String message,
            Throwable cause,
            ErrorEntity entity,
            Integer responseCode,
            Long requestId
    ) {
        super(message, cause);
        this.entity = entity;
        this.responseCode = responseCode;
        this.requestId = requestId;
    }

    public static ArangoDBException of(Throwable t) {
        return of(null, t);
    }

    public static ArangoDBException of(String message, Throwable t) {
        return of(message, t, null);
    }

    public static ArangoDBException of(Throwable t, Long requestId) {
        return of(null, t, requestId);
    }

    private static ArangoDBException of(String message, Throwable t, Long requestId) {
        Objects.requireNonNull(t);
        if (t instanceof CompletionException) {
            return of(message, t.getCause(), requestId);
        }
        Throwable cause = unwrapCause(t);
        String msg = message != null ? message
                : t.getMessage() != null ? t.getMessage()
                : cause.getMessage();
        ErrorEntity entity = null;
        Integer responseCode = null;
        Long reqId = requestId;

        if (t instanceof ArangoDBException) {
            entity = ((ArangoDBException) t).entity;
            responseCode = ((ArangoDBException) t).responseCode;
            reqId = reqId != null ? reqId : ((ArangoDBException) t).requestId;
        }

        return new ArangoDBException(
                msg,
                cause,
                entity,
                responseCode,
                reqId
        );
    }

    private static Throwable unwrapCause(Throwable t) {
        if (t instanceof ArangoDBException && t.getCause() != null) {
            return unwrapCause(t.getCause());
        }
        return t;
    }

    /**
     * @return ArangoDB error message
     */
    public String getErrorMessage() {
        return entity != null ? entity.getErrorMessage() : null;
    }

    /**
     * @return ArangoDB exception
     */
    public String getException() {
        return entity != null ? entity.getException() : null;
    }

    /**
     * @return HTTP response code
     */
    public Integer getResponseCode() {
        Integer entityResponseCode = entity != null ? entity.getCode() : null;
        return responseCode != null ? responseCode : entityResponseCode;
    }

    /**
     * @return ArangoDB error number
     */
    public Integer getErrorNum() {
        return entity != null ? entity.getErrorNum() : null;
    }

    /**
     * @return request id
     */
    public Long getRequestId() {
        return requestId;
    }
}
