/*
 * Decompiled with CFR 0.152.
 */
package com.orientechnologies.orient.core.storage.fs;

import com.kenai.jffi.MemoryIO;
import com.kenai.jffi.Platform;
import com.orientechnologies.common.concur.lock.ScalableRWLock;
import com.orientechnologies.common.exception.OException;
import com.orientechnologies.common.jnr.LastErrorException;
import com.orientechnologies.common.jnr.ONative;
import com.orientechnologies.common.log.OLogManager;
import com.orientechnologies.common.util.ORawPair;
import com.orientechnologies.orient.core.exception.OStorageException;
import com.orientechnologies.orient.core.storage.fs.IOResult;
import com.orientechnologies.orient.core.storage.fs.OFile;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.CompletionHandler;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;

public final class AsyncFile
implements OFile {
    private static final int ALLOCATION_THRESHOLD = 0x100000;
    private final ScalableRWLock lock = new ScalableRWLock();
    private volatile Path osFile;
    private final AtomicLong dirtyCounter = new AtomicLong();
    private final Object flushSemaphore = new Object();
    private final AtomicLong size = new AtomicLong();
    private final AtomicLong committedSize = new AtomicLong();
    private final boolean useNativeOsAPI;
    private AsynchronousFileChannel fileChannel;
    private int fd = -1;
    private final int pageSize;

    public AsyncFile(Path osFile, int pageSize, boolean useNativeOsAPI) {
        this.osFile = osFile;
        this.useNativeOsAPI = useNativeOsAPI;
        this.pageSize = pageSize;
    }

    @Override
    public void create() throws IOException {
        this.lock.exclusiveLock();
        try {
            if (this.fileChannel != null) {
                throw new OStorageException("File " + this.osFile + " is already opened.");
            }
            Files.createFile(this.osFile, new FileAttribute[0]);
            this.doOpen();
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    private void initSize() throws IOException {
        long currentSize;
        if (this.fileChannel.size() < 1024L) {
            Future<Integer> writeFuture;
            ByteBuffer buffer = ByteBuffer.allocate(1024);
            int written = 0;
            do {
                buffer.position(written);
                writeFuture = this.fileChannel.write(buffer, written);
                try {
                }
                catch (InterruptedException | ExecutionException e) {
                    throw OException.wrapException(new OStorageException("Error during write operation to the file " + this.osFile), e);
                }
            } while ((written += writeFuture.get().intValue()) < 1024);
            this.dirtyCounter.incrementAndGet();
        }
        if ((currentSize = this.fileChannel.size() - 1024L) % (long)this.pageSize != 0L) {
            long initialSize = currentSize;
            currentSize = currentSize / (long)this.pageSize * (long)this.pageSize;
            this.fileChannel.truncate(currentSize + 1024L);
            OLogManager.instance().warnNoDb(this, "Data page in file {} was partially written and will be truncated, initial size {}, truncated size {}", this.osFile, initialSize, currentSize);
        }
        this.size.set(currentSize);
        this.committedSize.set(currentSize);
    }

    @Override
    public void open() {
        this.lock.exclusiveLock();
        try {
            this.doOpen();
        }
        catch (IOException e) {
            throw OException.wrapException(new OStorageException("Can not open file " + this.osFile), e);
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    private void doOpen() throws IOException {
        if (this.fileChannel != null) {
            throw new OStorageException("File " + this.osFile + " is already opened.");
        }
        this.fileChannel = AsynchronousFileChannel.open(this.osFile, StandardOpenOption.READ, StandardOpenOption.WRITE);
        if (this.useNativeOsAPI && Platform.getPlatform().getOS() == Platform.OS.LINUX) {
            try {
                this.fd = ONative.instance().open(this.osFile.toAbsolutePath().toString(), 65);
            }
            catch (LastErrorException e) {
                this.fd = -1;
            }
        }
        this.initSize();
    }

    @Override
    public long getFileSize() {
        return this.size.get();
    }

    @Override
    public String getName() {
        return this.osFile.getFileName().toString();
    }

    @Override
    public boolean isOpen() {
        this.lock.sharedLock();
        try {
            boolean bl = this.fileChannel != null;
            return bl;
        }
        finally {
            this.lock.sharedUnlock();
        }
    }

    @Override
    public boolean exists() {
        return Files.exists(this.osFile, new LinkOption[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void write(long offset, ByteBuffer buffer) {
        this.lock.sharedLock();
        try {
            Future<Integer> writeFuture;
            buffer.rewind();
            this.checkForClose();
            this.checkPosition(offset);
            this.checkPosition(offset + (long)buffer.limit() - 1L);
            int written = 0;
            do {
                buffer.position(written);
                writeFuture = this.fileChannel.write(buffer, offset + 1024L + (long)written);
                try {
                }
                catch (InterruptedException | ExecutionException e) {
                    throw OException.wrapException(new OStorageException("Error during write operation to the file " + this.osFile), e);
                }
            } while ((written += writeFuture.get().intValue()) < buffer.limit());
            this.dirtyCounter.incrementAndGet();
            assert (written == buffer.limit());
        }
        finally {
            this.lock.sharedUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public IOResult write(List<ORawPair<Long, ByteBuffer>> buffers) {
        CountDownLatch latch = new CountDownLatch(buffers.size());
        AsyncIOResult asyncIOResult = new AsyncIOResult(latch);
        for (ORawPair<Long, ByteBuffer> pair : buffers) {
            ByteBuffer byteBuffer = (ByteBuffer)pair.second;
            byteBuffer.rewind();
            this.lock.sharedLock();
            try {
                this.checkForClose();
                this.checkPosition((Long)pair.first);
                this.checkPosition((Long)pair.first + (long)((ByteBuffer)pair.second).limit() - 1L);
                long position = (Long)pair.first + 1024L;
                this.fileChannel.write(byteBuffer, position, latch, new WriteHandler(byteBuffer, asyncIOResult, position));
            }
            finally {
                this.lock.sharedUnlock();
            }
        }
        return asyncIOResult;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void read(long offset, ByteBuffer buffer, boolean throwOnEof) throws IOException {
        this.lock.sharedLock();
        try {
            int bytesRead;
            this.checkForClose();
            this.checkPosition(offset);
            int read = 0;
            do {
                buffer.position(read);
                Future<Integer> readFuture = this.fileChannel.read(buffer, offset + 1024L + (long)read);
                try {
                    bytesRead = readFuture.get();
                }
                catch (InterruptedException | ExecutionException e) {
                    throw OException.wrapException(new OStorageException("Error during read operation from the file " + this.osFile), e);
                }
                if (bytesRead != -1) continue;
                if (throwOnEof) {
                    throw new EOFException("End of file " + this.osFile + " is reached.");
                }
                break;
            } while ((read += bytesRead) < buffer.limit());
        }
        finally {
            this.lock.sharedUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long allocateSpace(int size) throws IOException {
        long allocatedPosition;
        this.lock.sharedLock();
        try {
            long currentSize;
            block16: {
                currentSize = this.size.addAndGet(size);
                allocatedPosition = currentSize - (long)size;
                long currentCommittedSize = this.committedSize.get();
                long sizeDifference = currentSize - currentCommittedSize;
                if (this.fd >= 0 && sizeDifference <= 0x100000L) {
                    long l = allocatedPosition;
                    return l;
                }
                while (currentCommittedSize < currentSize && !this.committedSize.compareAndSet(currentCommittedSize, currentSize)) {
                    currentCommittedSize = this.committedSize.get();
                }
                if (this.fd < 0) {
                    MemoryIO memoryIO = MemoryIO.getInstance();
                    long ptr = memoryIO.allocateMemory(size, true);
                    try {
                        Future<Integer> writeFuture;
                        ByteBuffer buffer = memoryIO.newDirectByteBuffer(ptr, size).order(ByteOrder.nativeOrder());
                        int written = 0;
                        do {
                            buffer.position(written);
                            writeFuture = this.fileChannel.write(buffer, allocatedPosition + (long)written + 1024L);
                            try {
                            }
                            catch (InterruptedException | ExecutionException e) {
                                throw OException.wrapException(new OStorageException("Error during write operation to the file " + this.osFile), e);
                            }
                        } while ((written += writeFuture.get().intValue()) < size);
                        assert (written == size);
                        break block16;
                    }
                    finally {
                        memoryIO.freeMemory(ptr);
                    }
                }
                long sizeDiff = currentSize - currentCommittedSize;
                if (sizeDiff > 0L) {
                    ONative.instance().fallocate(this.fd, currentCommittedSize + 1024L, sizeDiff);
                }
            }
            assert (this.fileChannel.size() >= currentSize + 1024L);
        }
        finally {
            this.lock.sharedUnlock();
        }
        return allocatedPosition;
    }

    @Override
    public void shrink(long size) throws IOException {
        this.lock.exclusiveLock();
        try {
            this.checkForClose();
            this.size.set(0L);
            this.committedSize.set(size);
            this.fileChannel.truncate(size + 1024L);
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    @Override
    public void synch() {
        this.lock.sharedLock();
        try {
            this.doSynch();
        }
        finally {
            this.lock.sharedUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doSynch() {
        Object object = this.flushSemaphore;
        synchronized (object) {
            long dirtyCounterValue = this.dirtyCounter.get();
            if (dirtyCounterValue > 0L) {
                try {
                    this.fileChannel.force(false);
                }
                catch (IOException e) {
                    OLogManager.instance().warn((Object)this, "Error during flush of file %s. Data may be lost in case of power failure", e, this.getName());
                }
                this.dirtyCounter.addAndGet(-dirtyCounterValue);
            }
        }
    }

    @Override
    public void close() {
        this.lock.exclusiveLock();
        try {
            this.doSynch();
            this.doClose();
        }
        catch (IOException e) {
            throw OException.wrapException(new OStorageException("Error during closing the file " + this.osFile), e);
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    private void doClose() throws IOException {
        if (this.fileChannel != null) {
            this.fileChannel.close();
            this.fileChannel = null;
            if (this.fd >= 0) {
                long sizeDiff = this.size.get() - this.committedSize.get();
                if (sizeDiff > 0L) {
                    ONative.instance().fallocate(this.fd, this.committedSize.get() + 1024L, sizeDiff);
                }
                ONative.instance().close(this.fd);
                this.fd = -1;
            }
        }
    }

    @Override
    public void delete() throws IOException {
        this.lock.exclusiveLock();
        try {
            this.doClose();
            Files.delete(this.osFile);
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    @Override
    public void renameTo(Path newFile) throws IOException {
        this.lock.exclusiveLock();
        try {
            this.doClose();
            this.osFile = Files.move(this.osFile, newFile, new CopyOption[0]);
            this.doOpen();
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    @Override
    public void replaceContentWith(Path newContentFile) throws IOException {
        this.lock.exclusiveLock();
        try {
            this.doClose();
            Files.copy(newContentFile, this.osFile, StandardCopyOption.REPLACE_EXISTING);
            this.doOpen();
        }
        finally {
            this.lock.exclusiveUnlock();
        }
    }

    private void checkPosition(long offset) {
        long fileSize = this.size.get();
        if (offset < 0L || offset >= fileSize) {
            throw new OStorageException("You are going to access region outside of allocated file position. File size = " + fileSize + ", requested position " + offset);
        }
    }

    private void checkForClose() {
        if (this.fileChannel == null) {
            throw new OStorageException("File " + this.osFile + " is closed");
        }
    }

    private static final class AsyncIOResult
    implements IOResult {
        private final CountDownLatch latch;
        private Throwable exc;

        private AsyncIOResult(CountDownLatch latch) {
            this.latch = latch;
        }

        @Override
        public void await() {
            try {
                this.latch.await();
            }
            catch (InterruptedException e) {
                throw OException.wrapException(new OStorageException("IO operation was interrupted"), e);
            }
            if (this.exc != null) {
                throw OException.wrapException(new OStorageException("Error during IO operation"), this.exc);
            }
        }
    }

    private final class WriteHandler
    implements CompletionHandler<Integer, CountDownLatch> {
        private final ByteBuffer byteBuffer;
        private final AsyncIOResult ioResult;
        private final long position;

        private WriteHandler(ByteBuffer byteBuffer, AsyncIOResult ioResult, long position) {
            this.byteBuffer = byteBuffer;
            this.ioResult = ioResult;
            this.position = position;
        }

        @Override
        public void completed(Integer result, CountDownLatch attachment) {
            if (this.byteBuffer.remaining() > 0) {
                AsyncFile.this.lock.sharedLock();
                try {
                    AsyncFile.this.checkForClose();
                    AsyncFile.this.fileChannel.write(this.byteBuffer, this.position + (long)this.byteBuffer.position(), attachment, this);
                }
                finally {
                    AsyncFile.this.lock.sharedUnlock();
                }
            } else {
                AsyncFile.this.dirtyCounter.incrementAndGet();
                attachment.countDown();
            }
        }

        @Override
        public void failed(Throwable exc, CountDownLatch attachment) {
            this.ioResult.exc = exc;
            OLogManager.instance().error(this, "Error during write operation to the file " + AsyncFile.this.osFile, exc, new Object[0]);
            AsyncFile.this.dirtyCounter.incrementAndGet();
            attachment.countDown();
        }
    }
}

