/*
 * MIT License
 *
 * Copyright (c) 2020 Artipie
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.artipie.docker;

import com.artipie.asto.Content;
import com.artipie.docker.manifest.Manifest;
import com.artipie.docker.ref.ManifestRef;
import java.util.Optional;
import java.util.concurrent.CompletionStage;

/**
 * Docker repository manifests.
 *
 * @since 0.3
 */
public interface Manifests {

    /**
     * Put manifest.
     *
     * @param ref Manifest reference.
     * @param content Manifest content.
     * @return Added manifest.
     */
    CompletionStage<Manifest> put(ManifestRef ref, Content content);

    /**
     * Get manifest by reference.
     *
     * @param ref Manifest reference
     * @return Manifest instance if it is found, empty if manifest is absent.
     */
    CompletionStage<Optional<Manifest>> get(ManifestRef ref);

    /**
     * Abstract decorator for Manifests.
     *
     * @since 0.3
     */
    abstract class Wrap implements Manifests {

        /**
         * Origin manifests.
         */
        private final Manifests manifests;

        /**
         * Ctor.
         *
         * @param manifests Manifests.
         */
        protected Wrap(final Manifests manifests) {
            this.manifests = manifests;
        }

        @Override
        public final CompletionStage<Manifest> put(final ManifestRef ref, final Content content) {
            return this.manifests.put(ref, content);
        }

        @Override
        public final CompletionStage<Optional<Manifest>> get(final ManifestRef ref) {
            return this.manifests.get(ref);
        }
    }
}
