/*
 * MIT License
 *
 * Copyright (c) 2020 Artipie
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.artipie.docker.http;

import com.artipie.asto.Content;
import com.artipie.docker.Docker;
import com.artipie.docker.RepoName;
import com.artipie.docker.error.ManifestError;
import com.artipie.docker.manifest.Manifest;
import com.artipie.docker.misc.RqByRegex;
import com.artipie.docker.ref.ManifestRef;
import com.artipie.http.Response;
import com.artipie.http.Slice;
import com.artipie.http.async.AsyncResponse;
import com.artipie.http.headers.ContentLength;
import com.artipie.http.headers.ContentType;
import com.artipie.http.headers.Location;
import com.artipie.http.rq.RqHeaders;
import com.artipie.http.rs.RsStatus;
import com.artipie.http.rs.RsWithBody;
import com.artipie.http.rs.RsWithHeaders;
import com.artipie.http.rs.RsWithStatus;
import com.artipie.http.rs.StandardRs;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.regex.Pattern;
import org.reactivestreams.Publisher;

/**
 * Manifest entity in Docker HTTP API..
 * See <a href="https://docs.docker.com/registry/spec/api/#manifest">Manifest</a>.
 *
 * @since 0.2
 * @checkstyle ClassDataAbstractionCouplingCheck (500 lines)
 */
final class ManifestEntity {

    /**
     * RegEx pattern for path.
     */
    public static final Pattern PATH = Pattern.compile(
        "^/v2/(?<name>.*)/manifests/(?<reference>.*)$"
    );

    /**
     * Ctor.
     */
    private ManifestEntity() {
    }

    /**
     * Slice for HEAD method, checking manifest existence.
     *
     * @since 0.2
     */
    public static class Head implements Slice {

        /**
         * Docker repository.
         */
        private final Docker docker;

        /**
         * Ctor.
         *
         * @param docker Docker repository.
         */
        Head(final Docker docker) {
            this.docker = docker;
        }

        @Override
        public Response response(
            final String line,
            final Iterable<Map.Entry<String, String>> headers,
            final Publisher<ByteBuffer> body) {
            final Request request = new Request(line);
            final ManifestRef ref = request.reference();
            return new AsyncResponse(
                this.docker.repo(request.name()).manifests().get(ref).thenApply(
                    manifest -> manifest.<Response>map(
                        found -> new BaseResponse(found.convert(Head.acceptHeader(headers)))
                    ).orElseGet(
                        () -> new ErrorsResponse(RsStatus.NOT_FOUND, new ManifestError(ref))
                    )
                )
            );
        }

        /**
         * Gets accept header.
         * @param headers Request headers
         * @return Accept headers
         */
        private static RqHeaders acceptHeader(final Iterable<Map.Entry<String, String>> headers) {
            return new RqHeaders(headers, "Accept");
        }
    }

    /**
     * Slice for GET method, getting manifest content.
     *
     * @since 0.2
     */
    public static class Get implements Slice {

        /**
         * Docker repository.
         */
        private final Docker docker;

        /**
         * Ctor.
         *
         * @param docker Docker repository.
         */
        Get(final Docker docker) {
            this.docker = docker;
        }

        @Override
        public Response response(
            final String line,
            final Iterable<Map.Entry<String, String>> headers,
            final Publisher<ByteBuffer> body
        ) {
            final Request request = new Request(line);
            final RepoName name = request.name();
            final ManifestRef ref = request.reference();
            return new AsyncResponse(
                this.docker.repo(name).manifests().get(ref).thenApply(
                    manifest -> manifest.<Response>map(
                        found -> {
                            final Manifest mnf = found.convert(Head.acceptHeader(headers));
                            return new RsWithBody(new BaseResponse(mnf), mnf.content());
                        }
                    ).orElseGet(
                        () -> new ErrorsResponse(RsStatus.NOT_FOUND, new ManifestError(ref))
                    )
                )
            );
        }

    }

    /**
     * Slice for PUT method, uploading manifest content.
     *
     * @since 0.2
     */
    public static class Put implements Slice {

        /**
         * Docker repository.
         */
        private final Docker docker;

        /**
         * Ctor.
         *
         * @param docker Docker repository.
         */
        Put(final Docker docker) {
            this.docker = docker;
        }

        @Override
        public Response response(
            final String line,
            final Iterable<Map.Entry<String, String>> headers,
            final Publisher<ByteBuffer> body
        ) {
            final Request request = new Request(line);
            final RepoName name = request.name();
            final ManifestRef ref = request.reference();
            return new AsyncResponse(
                this.docker.repo(name).manifests().put(ref, new Content.From(body)).thenApply(
                    manifest -> new RsWithHeaders(
                        new RsWithStatus(RsStatus.CREATED),
                        new Location(
                            String.format("/v2/%s/manifests/%s", name.value(), ref.string())
                        ),
                        new ContentLength("0"),
                        new DigestHeader(manifest.digest())
                    )
                )
            );
        }
    }

    /**
     * HTTP request to manifest entity.
     *
     * @since 0.2
     */
    static final class Request {

        /**
         * HTTP request by RegEx.
         */
        private final RqByRegex rqregex;

        /**
         * Ctor.
         *
         * @param line HTTP request line.
         */
        Request(final String line) {
            this.rqregex = new RqByRegex(line, ManifestEntity.PATH);
        }

        /**
         * Get repository name.
         *
         * @return Repository name.
         */
        RepoName name() {
            return new RepoName.Valid(this.rqregex.path().group("name"));
        }

        /**
         * Get manifest reference.
         *
         * @return Manifest reference.
         */
        ManifestRef reference() {
            return new ManifestRef.FromString(this.rqregex.path().group("reference"));
        }

    }

    /**
     * Manifest base response.
     * @since 0.2
     */
    static final class BaseResponse extends Response.Wrap {

        /**
         * Ctor.
         *
         * @param mnf Manifest
         */
        BaseResponse(final Manifest mnf) {
            super(
                new RsWithHeaders(
                    StandardRs.EMPTY,
                    new ContentType(mnf.mediaType()),
                    new DigestHeader(mnf.digest())
                )
            );
        }

    }
}
