/*
 * MIT License
 *
 * Copyright (c) 2020 Artipie
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.artipie.nuget.http;

import com.artipie.http.Headers;
import com.artipie.http.Response;
import com.artipie.http.Slice;
import com.artipie.http.rq.RequestLine;
import com.artipie.http.rq.RqMethod;
import io.reactivex.Flowable;
import java.nio.ByteBuffer;
import org.reactivestreams.Publisher;

/**
 * Resource created from {@link Slice}.
 *
 * @since 0.2
 */
final class ResourceFromSlice implements Resource {

    /**
     * Path to resource.
     */
    private final String path;

    /**
     * Origin slice.
     */
    private final Slice origin;

    /**
     * Ctor.
     *
     * @param path Path to resource.
     * @param origin Origin slice.
     */
    ResourceFromSlice(final String path, final Slice origin) {
        this.path = path;
        this.origin = origin;
    }

    @Override
    public Response get(final Headers headers) {
        return this.delegate(RqMethod.GET, headers, Flowable.empty());
    }

    @Override
    public Response put(final Headers headers, final Publisher<ByteBuffer> body) {
        return this.delegate(RqMethod.PUT, headers, body);
    }

    /**
     * Delegates request handling to origin slice.
     *
     * @param method Request method.
     * @param headers Request headers.
     * @param body Request body.
     * @return Response generated by origin slice.
     */
    private Response delegate(
        final RqMethod method,
        final Headers headers,
        final Publisher<ByteBuffer> body
    ) {
        return this.origin.response(
            new RequestLine(method.value(), this.path).toString(),
            headers,
            body
        );
    }
}
