/*
 * MIT License
 *
 * Copyright (c) 2020 Artipie
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.artipie.nuget.http.metadata;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Collection of completion stages that all can be combined into single one.
 *
 * @param <T> Completion stages type.
 * @since 0.4
 */
final class CompletionStages<T> {

    /**
     * Completion stages.
     */
    private final Collection<CompletionStage<T>> stages;

    /**
     * Ctor.
     *
     * @param stages Completion stages.
     */
    CompletionStages(final Stream<CompletionStage<T>> stages) {
        this(stages.collect(Collectors.toList()));
    }

    /**
     * Ctor.
     *
     * @param stages Completion stages.
     */
    CompletionStages(final Collection<CompletionStage<T>> stages) {
        this.stages = stages;
    }

    /**
     * Combine original stages into single one that completes when all stages are complete.
     *
     * @return Combined completion stages.
     */
    public CompletionStage<Collection<T>> all() {
        final List<CompletableFuture<T>> futures = this.stages.stream()
            .map(CompletionStage::toCompletableFuture)
            .collect(Collectors.toList());
        return CompletableFuture.allOf(
            futures.stream().toArray(CompletableFuture[]::new)
        ).thenApply(
            nothing -> futures.stream().map(CompletableFuture::join).collect(Collectors.toList())
        );
    }
}
