/*
 * MIT License
 *
 * Copyright (c) 2020 Artipie
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.artipie.nuget.http.metadata;

import com.artipie.nuget.PackageId;
import com.artipie.nuget.PackageIdentity;
import com.artipie.nuget.Repository;
import com.artipie.nuget.Version;
import java.util.List;
import java.util.concurrent.CompletionStage;
import javax.json.Json;
import javax.json.JsonArrayBuilder;
import javax.json.JsonObject;

/**
 * Registration page.
 * See <a href="https://docs.microsoft.com/en-us/nuget/api/registration-base-url-resource#registration-page-object">Registration page</a>
 *
 * @since 0.1
 */
final class RegistrationPage {

    /**
     * Repository.
     */
    private final Repository repository;

    /**
     * Package content location.
     */
    private final ContentLocation content;

    /**
     * Package identifier.
     */
    private final PackageId id;

    /**
     * Ordered list of versions on this page from lowest to highest.
     */
    private final List<Version> versions;

    /**
     * Ctor.
     *
     * @param repository Repository.
     * @param content Package content location.
     * @param id Package identifier.
     * @param versions Ordered list of versions on this page from lowest to highest.
     * @todo #87:60min Refactor RegistrationPage class, reduce number of fields.
     *  Probably it is needed to extract some abstraction for creating leaf objects,
     *  that will join `repository` and `content` fields and produce leaf JSON for package identity.
     * @checkstyle ParameterNumberCheck (2 line)
     */
    RegistrationPage(
        final Repository repository,
        final ContentLocation content,
        final PackageId id,
        final List<Version> versions
    ) {
        this.repository = repository;
        this.content = content;
        this.id = id;
        this.versions = versions;
    }

    /**
     * Generates page in JSON.
     *
     * @return Page JSON.
     */
    public CompletionStage<JsonObject> json() {
        if (this.versions.isEmpty()) {
            throw new IllegalStateException(
                String.format("Registration page contains no versions: '%s'", this.id)
            );
        }
        final Version lower = this.versions.get(0);
        final Version upper = this.versions.get(this.versions.size() - 1);
        return new CompletionStages<>(
            this.versions.stream().map(
                version -> this.leaf(new PackageIdentity(this.id, version))
            )
        ).all().thenApply(
            leafs -> {
                final JsonArrayBuilder items = Json.createArrayBuilder();
                for (final JsonObject leaf : leafs) {
                    items.add(leaf);
                }
                return Json.createObjectBuilder()
                    .add("lower", lower.normalized())
                    .add("upper", upper.normalized())
                    .add("count", this.versions.size())
                    .add("items", items)
                    .build();
            }
        );
    }

    /**
     * Builds registration leaf.
     * See <a href="https://docs.microsoft.com/en-us/nuget/api/registration-base-url-resource#registration-leaf-object-in-a-page"></a>
     *
     * @param identity Package identity.
     * @return JSON representing registration leaf.
     */
    private CompletionStage<JsonObject> leaf(final PackageIdentity identity) {
        return this.repository.nuspec(identity).thenApply(
            nuspec -> Json.createObjectBuilder()
                .add(
                    "catalogEntry",
                    Json.createObjectBuilder()
                        .add("id", nuspec.packageId().original())
                        .add("version", nuspec.version().normalized())
                )
                .add("packageContent", this.content.url(identity).toString())
                .build()
        );
    }
}
