package com.atlassian.bamboo.specs.builders.repository.bitbucket.cloud;

import com.atlassian.bamboo.specs.api.builders.credentials.SharedCredentials;
import com.atlassian.bamboo.specs.api.builders.credentials.SharedCredentialsIdentifier;
import com.atlassian.bamboo.specs.api.builders.repository.VcsChangeDetection;
import com.atlassian.bamboo.specs.api.builders.repository.VcsRepository;
import com.atlassian.bamboo.specs.api.exceptions.PropertiesValidationException;
import com.atlassian.bamboo.specs.api.model.repository.VcsChangeDetectionProperties;
import com.atlassian.bamboo.specs.api.util.EntityPropertiesBuilders;
import com.atlassian.bamboo.specs.builders.repository.git.SshPrivateKeyAuthentication;
import com.atlassian.bamboo.specs.builders.repository.git.UserPasswordAuthentication;
import com.atlassian.bamboo.specs.builders.repository.viewer.BitbucketCloudRepositoryViewer;
import com.atlassian.bamboo.specs.model.repository.bitbucket.cloud.BitbucketCloudRepositoryProperties;
import com.atlassian.bamboo.specs.model.repository.git.AuthenticationProperties;
import com.atlassian.bamboo.specs.model.repository.git.SharedCredentialsAuthenticationProperties;
import com.atlassian.bamboo.specs.model.repository.git.UserPasswordAuthenticationProperties;
import org.jetbrains.annotations.NotNull;

import java.time.Duration;

import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNotBlank;
import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNotNull;

/**
 * Represents Bitbucket Cloud repository in Bamboo.
 */
public class BitbucketCloudRepository extends VcsRepository<BitbucketCloudRepository, BitbucketCloudRepositoryProperties> {

    private String repositorySlug;

    private String branch;

    private boolean useShallowClones = BitbucketCloudRepositoryProperties.DEFAULT_SHALLOW_CLONES;
    private boolean useRemoteAgentCache = BitbucketCloudRepositoryProperties.DEFAULT_REMOTE_AGENT_CACHE;
    private boolean useSubmodules = BitbucketCloudRepositoryProperties.DEFAULT_USE_SUBMODULES;
    private boolean useSubmodulesWithShallowClones = BitbucketCloudRepositoryProperties.DEFAULT_USE_SUBMODULES_WITH_SHALLOW_CLONES;
    private boolean sshKeyAppliesToSubmodules;
    private Duration commandTimeout = BitbucketCloudRepositoryProperties.DEFAULT_COMMAND_TIMEOUT;
    private boolean verboseLogs = BitbucketCloudRepositoryProperties.DEFAULT_VERBOSE_LOGS;
    private boolean fetchWholeRepository = BitbucketCloudRepositoryProperties.DEFAULT_FETCH_WHOLE_REPOSITORY;
    private boolean useLfs = BitbucketCloudRepositoryProperties.DEFAULT_LFS;
    private boolean webhookEnabled = BitbucketCloudRepositoryProperties.DEFAULT_WEBHOOK_ENABLED;

    private AuthenticationProperties accountAuthentication;
    private AuthenticationProperties checkoutAuthentication;
    private VcsChangeDetectionProperties vcsChangeDetection;

    public BitbucketCloudRepository() throws PropertiesValidationException {
        repositoryViewer(new BitbucketCloudRepositoryViewer());
    }

    /**
     * <p>
     * Sets the Bitbucket repository slug to checkout. A repository slug is a URL-friendly version of a repository name,
     * automatically generated by Bitbucket Cloud for use in the URL.
     * </p>
     * <p>
     * This field is required and it must represents existing Bitbucket Cloud repository.
     * </p>
     */
    public BitbucketCloudRepository repositorySlug(@NotNull String owner, @NotNull String repositorySlug) {
        checkNotBlank("owner", owner);
        checkNotBlank("repositorySlug", repositorySlug);
        this.repositorySlug = String.format("%s/%s", owner, repositorySlug);
        return this;
    }

    /**
     * Sets a branch to check out. This field is required.
     */
    public BitbucketCloudRepository branch(@NotNull String branch) {
        checkNotNull("branch", branch);
        this.branch = branch;
        return this;
    }

    /**
     * <p>
     * Enables/disables shallow clones when checking out from the repository.
     * With shallow clones on, Bamboo fetches the shallowest commit history possible. Do not use if your build depends on full repository history.
     * </p>
     * <p>
     * Shallow clones are switched off by default.
     * </p>
     * <p>
     *    Option does nothing with Mercurial type of repository.
     * </p>
     */
    public BitbucketCloudRepository shallowClonesEnabled(boolean useShallowClones) {
        this.useShallowClones = useShallowClones;
        return this;
    }

    /**
     * Enables/disabled authenticating to submodules with SSH key configured for this repository.
     */
    public BitbucketCloudRepository sshKeyAppliesToSubmodules(boolean sshKeyAppliesToSubmodules) {
        this.sshKeyAppliesToSubmodules = sshKeyAppliesToSubmodules;
        return this;
    }

    /**
     * Enables/disables submodule support. Turned off by default.
     */
    public BitbucketCloudRepository submodulesEnabled(boolean useSubmodules) {
        this.useSubmodules = useSubmodules;
        return this;
    }

    /**
     * Enables/disables shallow clone support for submodules. Turned off by default.
     */
    public BitbucketCloudRepository submodulesUseShallowClonesEnabled(boolean useSubmodulesWithShallowClones) {
        this.useSubmodulesWithShallowClones = useSubmodulesWithShallowClones;
        return this;
    }

    /**
     * <p>
     * Enables/disables caching repository content on the remote and elastic agents. Bamboo uses caching to reduce bandwidth needed when retrieving
     * source code from the repository.
     * </p>
     * <p>
     * The feature is turned on by default in Git repositories
     * </p>
     * <p>
     *    Option does nothing with Mercurial type of repository.
     * </p>
     */
    public BitbucketCloudRepository remoteAgentCacheEnabled(boolean useRemoteAgentCache) {
        this.useRemoteAgentCache = useRemoteAgentCache;
        return this;
    }

    /**
     * Specifies how much time is given for git commands to finish. Default is 180 minutes.
     */
    public BitbucketCloudRepository commandTimeout(Duration commandTimeout) {
        this.commandTimeout = commandTimeout;
        return this;
    }

    /**
     * Specifies how much time in minutes is given for git commands to finish. Default is 180 minutes.
     */
    public BitbucketCloudRepository commandTimeoutInMinutes(int commandTimeoutMinutes) {
        return commandTimeout(Duration.ofMinutes(commandTimeoutMinutes));
    }

    /**
     * Enables/disables verbose logs from git commands. Off by default.
     */
    public BitbucketCloudRepository verboseLogs(boolean verboseLogs) {
        this.verboseLogs = verboseLogs;
        return this;
    }

    /**
     * Enforces (or not) fetching all remote refs from the repository rather than single branch. Off by default.
     */
    public BitbucketCloudRepository fetchWholeRepository(boolean fetchWholeRepository) {
        this.fetchWholeRepository = fetchWholeRepository;
        return this;
    }

    /**
     * Enables/disables git lfs support. Off by default.
     * <p>
     *    Option does nothing with Mercurial type of repository.
     * </p>
     */
    public BitbucketCloudRepository lfsEnabled(boolean useLfs) {
        this.useLfs = useLfs;
        return this;
    }

    /**
     * Enables webhooks support for the repository. Off by default.
     */
    public BitbucketCloudRepository webhookEnabled(boolean webhookEnabled) {
        this.webhookEnabled = webhookEnabled;
        return this;
    }

    /**
     * Resets all change detection options to defaults.
     */
    public BitbucketCloudRepository defaultChangeDetection() {
        this.vcsChangeDetection = null;
        return this;
    }

    /**
     * Sets change detection options for this repository.
     *
     * @see VcsChangeDetection
     */
    public BitbucketCloudRepository changeDetection(VcsChangeDetection vcsChangeDetection) {
        this.vcsChangeDetection = EntityPropertiesBuilders.build(vcsChangeDetection);
        return this;
    }

    /**
     * <p>
     * Selects a previously defined {@link SharedCredentials} to configure account authentication with bitbucket.org
     * </p>
     * <p>
     * Currently, only username/password credentials are supported.
     * </p>
     */
    public BitbucketCloudRepository accountAuthentication(@NotNull SharedCredentialsIdentifier sharedCredentialsIdentifier) {
        checkNotNull("sharedCredentialsIdentifier", sharedCredentialsIdentifier);
        this.accountAuthentication = new SharedCredentialsAuthenticationProperties(EntityPropertiesBuilders.build(sharedCredentialsIdentifier));
        return this;
    }

    /**
     * <p>
     * Specifies username/password Bitbucket Cloud account authentication.
     * </p>
     * <p>
     * The password can be defined in both
     * Bamboo-encrypted and plain format.
     * </p>
     *
     * @see <a href="https://confluence.atlassian.com/bamboo/bamboo-specs-encryption-970268127.html">Encryption in Bamboo Specs</a>
     */
    public BitbucketCloudRepository accountAuthentication(@NotNull UserPasswordAuthentication userPasswordAuthentication) {
        checkNotNull("userPasswordAuthentication", userPasswordAuthentication);
        final UserPasswordAuthenticationProperties userAuthProperties = EntityPropertiesBuilders.build(userPasswordAuthentication);
        checkNotBlank("password", userAuthProperties.getPassword());
        this.accountAuthentication = userAuthProperties;
        return this;
    }

    /**
     * <p>
     * Selects a previously defined {@link SharedCredentials} which will be used to checkout repository from Bitbucket Cloud.
     * Currently SSH credentials are supported.
     * </p>
     * <p>
     * If checkout authentication is not configured, the <em>accountAuthentication</em>
     * will be used.
     * </p>
     */
    public BitbucketCloudRepository checkoutAuthentication(@NotNull SharedCredentialsIdentifier sharedCredentialsIdentifier) {
        checkNotNull("sharedCredentialsIdentifier", sharedCredentialsIdentifier);
        this.checkoutAuthentication = new SharedCredentialsAuthenticationProperties(EntityPropertiesBuilders.build(sharedCredentialsIdentifier));
        return this;
    }

    /**
     * <p>
     * Specifies SSH private key which will be used to checkout repository from Bitbucket Cloud. If checkout authentication
     * is not configured, the <em>accountAuthentication</em> will be used.
     * </p>
     * <p>
     * The key and passphrase can be defined in both
     * Bamboo-encrypted and plain format.
     * </p>
     *
     * @see <a href="https://confluence.atlassian.com/bamboo/bamboo-specs-encryption-970268127.html">Encryption in Bamboo Specs</a>
     */
    public BitbucketCloudRepository checkoutAuthentication(@NotNull SshPrivateKeyAuthentication sshPrivateKeyAuthentication) {
        checkNotNull("sshPrivateKeyAuthentication", sshPrivateKeyAuthentication);
        this.checkoutAuthentication = EntityPropertiesBuilders.build(sshPrivateKeyAuthentication);
        return this;
    }

    @Override
    protected BitbucketCloudRepositoryProperties build() throws PropertiesValidationException {
        return new BitbucketCloudRepositoryProperties(name,
                oid,
                description,
                parent,
                repositoryViewer,
                repositorySlug,
                accountAuthentication,
                checkoutAuthentication,
                branch,
                project,
                useShallowClones,
                useRemoteAgentCache,
                useSubmodules,
                useSubmodulesWithShallowClones,
                sshKeyAppliesToSubmodules,
                commandTimeout,
                verboseLogs,
                fetchWholeRepository,
                useLfs,
                webhookEnabled,
                vcsChangeDetection);
    }
}
