package com.atlassian.bamboo.specs.builders.repository.bitbucket.server;

import com.atlassian.bamboo.specs.api.builders.applink.ApplicationLink;
import com.atlassian.bamboo.specs.api.builders.plan.Plan;
import com.atlassian.bamboo.specs.api.builders.repository.VcsChangeDetection;
import com.atlassian.bamboo.specs.api.builders.repository.VcsRepository;
import com.atlassian.bamboo.specs.api.model.applink.ApplicationLinkProperties;
import com.atlassian.bamboo.specs.api.model.repository.VcsChangeDetectionProperties;
import com.atlassian.bamboo.specs.api.util.EntityPropertiesBuilders;
import com.atlassian.bamboo.specs.model.repository.bitbucket.server.BitbucketServerMirrorProperties;
import com.atlassian.bamboo.specs.model.repository.bitbucket.server.BitbucketServerRepositoryProperties;
import org.jetbrains.annotations.NotNull;

import java.time.Duration;

import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNotBlank;
import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNotNull;

/**
 * Represents Bitbucket Server repository in Bamboo.
 */
public class BitbucketServerRepository extends VcsRepository<BitbucketServerRepository, BitbucketServerRepositoryProperties> {
    private ApplicationLinkProperties server;

    private String projectKey;
    private String repositorySlug;
    private String sshPublicKey;
    private String sshPrivateKey;
    private String sshCloneUrl;

    private String branch;

    private boolean useShallowClones;
    private boolean useRemoteAgentCache = true;
    private boolean useSubmodules;
    private boolean useSubmodulesWithShallowClones;
    private boolean sshKeyAppliesToSubmodules;
    private Duration commandTimeout = Duration.ofMinutes(180);
    private boolean verboseLogs;
    private boolean fetchWholeRepository;
    private boolean useLfs;
    private BitbucketServerMirrorProperties mirror;

    private VcsChangeDetectionProperties vcsChangeDetection;

    /**
     * Specifies a Bitbucket Server repository
     * <p>
     * A repository can be global, in which case any changes made to a repository are automatically applied to all plans or jobs that use that repository.
     * Repository can also be defined in scope of a {@link Plan}, in which case it is only accessible in context of plan it is defined for.
     */
    public BitbucketServerRepository() {
    }

    /**
     * Selects Bitbucket Server instance which hosts this repository. The Bitbucket Server must be accessible from Bamboo via an application link.
     *
     * @param server Bitbucket Server instance
     * @see ApplicationLink
     */
    public BitbucketServerRepository server(@NotNull ApplicationLink server) {
        checkNotNull("server", server);
        this.server = EntityPropertiesBuilders.build(server);
        return this;
    }

    /**
     * Sets key of Bitbucket Server's project which repository belongs to.
     */
    public BitbucketServerRepository projectKey(@NotNull String projectKey) {
        checkNotBlank("projectKey", projectKey);
        this.projectKey = projectKey;
        return this;
    }

    /**
     * Sets the repository slug. A repository slug is a URL-friendly version of a repository name, automatically generated by Bitbucket for use in the URL.
     * <p>
     * To find out what is the repository slug of your repository, open it in Bitbucket Server UI and check the URL.
     */
    public BitbucketServerRepository repositorySlug(@NotNull String repositorySlug) {
        checkNotBlank("repositorySlug", repositorySlug);
        this.repositorySlug = repositorySlug;
        return this;
    }

    /**
     * Sets public SSH key of the repository.
     * <p>
     * If specified, a private key should be specified as well.
     * <p>
     * If not provided, Bamboo will attempt to generate and install new SSH keys into Bamboo Server instance.
     * This property is required when using repository stored Specs feature.
     */
    public BitbucketServerRepository sshPublicKey(@NotNull String sshPublicKey) {
        checkNotBlank("sshPublicKey", sshPublicKey);
        this.sshPublicKey = sshPublicKey;
        return this;
    }

    /**
     * Sets private SSH key of the repository. The key can be defined in both Bamboo-encrypted and plain format.
     * <p>
     * If specified, a public key should be specified as well.
     * <p>
     * If not provided, Bamboo will attempt to generate and install new SSH keys into Bamboo Server instance.
     * This property is required when using repository stored Specs feature.
     *
     * @see <a href="https://confluence.atlassian.com/bamboo/bamboo-specs-encryption-970268127.html">Encryption in Bamboo Specs</a>
     */
    public BitbucketServerRepository sshPrivateKey(@NotNull String sshPrivateKey) {
        checkNotBlank("sshPrivateKey", sshPrivateKey);
        this.sshPrivateKey = sshPrivateKey;
        return this;
    }

    /**
     * Sets repository url that is used for cloning the repository. Must be an ssh url.
     * If not defined, the importer tries to fetch this value from Bitbucket.
     * This property is required when using repository stored Specs feature.
     */
    public BitbucketServerRepository sshCloneUrl(@NotNull String sshCloneUrl) {
        checkNotBlank("sshCloneUrl", sshCloneUrl);
        this.sshCloneUrl = sshCloneUrl;
        return this;
    }

    /**
     * Sets a branch to check out.
     */
    public BitbucketServerRepository branch(@NotNull String branch) {
        checkNotNull("branch", branch);
        this.branch = branch;
        return this;
    }

    /**
     * Enables/disables shallow clones when checking out from the repository.
     * With shallow clones on, Bamboo fetches the shallowest commit history possible. Do not use if your build depends on full repository history.
     * Shallow clones are switched off by default.
     */
    public BitbucketServerRepository shallowClonesEnabled(boolean useShallowClones) {
        this.useShallowClones = useShallowClones;
        return this;
    }

    /**
     * Enables/disables submodule support. Turned off by default.
     */
    public BitbucketServerRepository submodulesEnabled(boolean useSubmodules) {
        this.useSubmodules = useSubmodules;
        return this;
    }

    /**
     * Enables/disables shallow clone support for submodules. Turned off by default.
     */
    public BitbucketServerRepository submodulesUseShallowClonesEnabled(boolean useSubmodulesWithShallowClones) {
        this.useSubmodulesWithShallowClones = useSubmodulesWithShallowClones;
        return this;
    }

    /**
     * Enables/disabled authenticating to submodules with SSH key configured for this repository.
     */
    public BitbucketServerRepository sshKeyAppliesToSubmodules(boolean sshKeyAppliesToSubmodules) {
        this.sshKeyAppliesToSubmodules = sshKeyAppliesToSubmodules;
        return this;
    }

    /**
     * Enables/disables caching repository content on the remote and elastic agents. Bamboo uses caching to reduce bandwidth needed when retrieving
     * source code from the repository. The feature is turned on by default.
     */
    public BitbucketServerRepository remoteAgentCacheEnabled(boolean useRemoteAgentCache) {
        this.useRemoteAgentCache = useRemoteAgentCache;
        return this;
    }

    /**
     * Specifies how much time is given for git commands to finish. Default is 180 minutes.
     */
    public BitbucketServerRepository commandTimeout(Duration commandTimeout) {
        this.commandTimeout = commandTimeout;
        return this;
    }

    /**
     * Specifies how much time in minutes is given for git commands to finish. Default is 180 minutes.
     */
    public BitbucketServerRepository commandTimeoutInMinutes(int commandTimeoutMinutes) {
        return commandTimeout(Duration.ofMinutes(commandTimeoutMinutes));
    }

    /**
     * Enables/disables verbose logs from git commands. Off by default.
     */
    public BitbucketServerRepository verboseLogs(boolean verboseLogs) {
        this.verboseLogs = verboseLogs;
        return this;
    }

    /**
     * Enforces (or not) fetching all remote refs from the repository rather than single branch. Off by default.
     */
    public BitbucketServerRepository fetchWholeRepository(boolean fetchWholeRepository) {
        this.fetchWholeRepository = fetchWholeRepository;
        return this;
    }

    /**
     * Enables/disables git lfs support. Off by default.
     */
    public BitbucketServerRepository lfsEnabled(boolean useLfs) {
        this.useLfs = useLfs;
        return this;
    }

    /**
     * Resets all change detection options to defaults.
     */
    public BitbucketServerRepository defaultChangeDetection() {
        this.vcsChangeDetection = null;
        return this;
    }

    /**
     * Sets change detection options for this repository.
     *
     * @see VcsChangeDetection
     */
    public BitbucketServerRepository changeDetection(@NotNull VcsChangeDetection vcsChangeDetection) {
        checkNotNull("vcsChangeDetection", vcsChangeDetection);
        this.vcsChangeDetection = EntityPropertiesBuilders.build(vcsChangeDetection);
        return this;
    }

    /**
     * Sets mirror to be used for cloning repository.
     * Leave it empty if not using mirrors.
     */
    public BitbucketServerRepository mirror(@NotNull BitbucketServerMirror mirror) {
        checkNotNull("mirror", mirror);
        this.mirror = EntityPropertiesBuilders.build(mirror);
        return this;
    }

    @Override
    protected BitbucketServerRepositoryProperties build() {
        return new BitbucketServerRepositoryProperties(name,
                oid,
                description,
                parent,
                repositoryViewer,
                server,
                projectKey,
                repositorySlug,
                sshPublicKey,
                sshPrivateKey,
                sshCloneUrl,
                branch,
                project,
                vcsChangeDetection,
                useShallowClones,
                useRemoteAgentCache,
                useSubmodules,
                useSubmodulesWithShallowClones,
                sshKeyAppliesToSubmodules,
                commandTimeout,
                verboseLogs,
                fetchWholeRepository,
                useLfs,
                mirror);
    }
}
