package com.atlassian.bamboo.specs.model.repository.bitbucket.cloud;

import com.atlassian.bamboo.specs.api.codegen.annotations.ConstructFrom;
import com.atlassian.bamboo.specs.api.validators.common.ValidationContext;
import com.atlassian.bamboo.specs.api.validators.common.ValidationProblem;
import com.atlassian.bamboo.specs.model.repository.git.AuthenticationProperties;

import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNoErrors;
import static com.atlassian.bamboo.specs.api.validators.common.ImporterUtils.checkNotNull;
import static com.atlassian.bamboo.specs.api.validators.common.ValidationUtils.containsBambooVariable;
import static com.atlassian.bamboo.specs.api.validators.common.ValidationUtils.validateNotContainsRelaxedXssRelatedCharacters;

@Immutable
@ConstructFrom("email")
public class EmailApiTokenAuthenticationProperties implements AuthenticationProperties {
    private final String email;
    private final String apiToken;

    private EmailApiTokenAuthenticationProperties() {
        this.email = null;
        this.apiToken = null;
    }

    public EmailApiTokenAuthenticationProperties(String email, String apiToken) {
        this.email = email;
        this.apiToken = apiToken;

        validate();
    }

    public String getEmail() {
        return email;
    }

    public String getApiToken() {
        return apiToken;
    }

    public void validate() {
        checkNotNull("email", email);

        final ValidationContext context = ValidationContext.of("User-token authentication");
        final List<ValidationProblem> errors = new ArrayList<>();

        if (!containsBambooVariable(email)) {
            validateNotContainsRelaxedXssRelatedCharacters(context.with("email"), email)
                    .ifPresent(errors::add);
        }

        checkNoErrors(errors);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        EmailApiTokenAuthenticationProperties that = (EmailApiTokenAuthenticationProperties) o;
        return Objects.equals(email, that.email) && Objects.equals(apiToken, that.apiToken);
    }

    @Override
    public int hashCode() {
        return Objects.hash(email, apiToken);
    }
}
