package com.autonomousapps.tasks

import com.autonomousapps.TASK_GROUP_DEP_INTERNAL
import com.autonomousapps.advice.Advice
import com.autonomousapps.advice.ComprehensiveAdvice
import com.autonomousapps.advice.PluginAdvice
import com.autonomousapps.extension.Behavior
import com.autonomousapps.internal.advice.SeverityHandler
import com.autonomousapps.internal.utils.*
import org.gradle.api.DefaultTask
import org.gradle.api.file.RegularFile
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.*

@CacheableTask
abstract class AdviceSubprojectAggregationTask : DefaultTask() {

  init {
    group = TASK_GROUP_DEP_INTERNAL
    description = "Aggregates advice from a project's variant-specific advice tasks"
  }

  /*
   * Inputs
   */

  @get:Input
  val projectPath = project.path

  @get:PathSensitive(PathSensitivity.RELATIVE)
  @get:InputFiles
  abstract val dependencyAdvice: ListProperty<RegularFile>

  @get:PathSensitive(PathSensitivity.RELATIVE)
  @get:InputFiles
  abstract val redundantJvmAdvice: ListProperty<RegularFile>

  @get:PathSensitive(PathSensitivity.RELATIVE)
  @get:InputFiles
  abstract val redundantKaptAdvice: ListProperty<RegularFile>

  /*
   * Severity
   */

  @get:Input
  abstract val anyBehavior: Property<Behavior>

  @get:Input
  abstract val unusedDependenciesBehavior: Property<Behavior>

  @get:Input
  abstract val usedTransitiveDependenciesBehavior: Property<Behavior>

  @get:Input
  abstract val incorrectConfigurationBehavior: Property<Behavior>

  @get:Input
  abstract val compileOnlyBehavior: Property<Behavior>

  @get:Input
  abstract val unusedProcsBehavior: Property<Behavior>

  @get:Input
  abstract val redundantPluginsBehavior: Property<Behavior>

  /*
   * Outputs
   */

  @get:OutputFile
  abstract val output: RegularFileProperty

  @get:OutputFile
  abstract val outputPretty: RegularFileProperty

  @TaskAction fun action() {
    // Outputs
    val outputFile = output.getAndDelete()
    val outputPrettyFile = outputPretty.getAndDelete()

    // Inputs
    val dependencyAdvice = dependencyAdvice.get().flatMapToOrderedSet { it.fromJsonSet<Advice>() }
    val pluginAdvice = redundantJvmAdvice.toPluginAdvice() + redundantKaptAdvice.toPluginAdvice()

    val severityHandler = SeverityHandler(
      anyBehavior = anyBehavior.get(),
      unusedDependenciesBehavior = unusedDependenciesBehavior.get(),
      usedTransitiveDependenciesBehavior = usedTransitiveDependenciesBehavior.get(),
      incorrectConfigurationBehavior = incorrectConfigurationBehavior.get(),
      compileOnlyBehavior = compileOnlyBehavior.get(),
      unusedProcsBehavior = unusedProcsBehavior.get(),
      redundantPluginsBehavior = redundantPluginsBehavior.get()
    )
    val shouldFailDeps = severityHandler.shouldFailDeps(dependencyAdvice)
    val shouldFailPlugins = severityHandler.shouldFailPlugins(pluginAdvice)

    // Combine
    val comprehensiveAdvice = ComprehensiveAdvice(
      projectPath = projectPath,
      dependencyAdvice = dependencyAdvice,
      pluginAdvice = pluginAdvice,
      shouldFail = shouldFailDeps || shouldFailPlugins
    )

    outputFile.writeText(comprehensiveAdvice.toJson())
    outputPrettyFile.writeText(comprehensiveAdvice.toPrettyString())
  }

  private fun ListProperty<RegularFile>.toPluginAdvice(): Set<PluginAdvice> = get()
    .flatMapToSet {
      val file = it.asFile
      if (file.exists()) {
        file.readText().fromJsonSet()
      } else {
        emptySet()
      }
    }
}
