package com.autonomousapps.tasks

import com.autonomousapps.TASK_GROUP_DEP_INTERNAL
import com.autonomousapps.internal.ClassFilesParser
import com.autonomousapps.internal.utils.filterToClassFiles
import com.autonomousapps.internal.utils.getAndDelete
import com.autonomousapps.internal.utils.toJson
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.ProjectLayout
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.tasks.CacheableTask
import org.gradle.api.tasks.Classpath
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import org.gradle.workers.WorkAction
import org.gradle.workers.WorkParameters
import org.gradle.workers.WorkerExecutor
import javax.inject.Inject

abstract class ByteCodeSourceExploderTask : DefaultTask() {

  @get:OutputFile
  abstract val output: RegularFileProperty
}

@CacheableTask
abstract class ClassListExploderTask @Inject constructor(
  private val workerExecutor: WorkerExecutor,
  private val layout: ProjectLayout
) : ByteCodeSourceExploderTask() {

  init {
    group = TASK_GROUP_DEP_INTERNAL
    description = "Produces a report of all classes referenced by a given set of class files"
  }

  /**
   * Class files generated by Kotlin source. May be empty.
   */
  @get:Classpath
  @get:InputFiles
  abstract val kotlinClasses: ConfigurableFileCollection

  /**
   * Class files generated by Java source. May be empty.
   */
  @get:Classpath
  @get:InputFiles
  abstract val javaClasses: ConfigurableFileCollection

  @TaskAction fun action() {
    workerExecutor.noIsolation().submit(ClassListExploderWorkAction::class.java) {
      classFiles.setFrom(
        javaClasses.asFileTree.plus(kotlinClasses)
          .filterToClassFiles()
          .files
      )
      buildDir.set(layout.buildDirectory)
      output.set(this@ClassListExploderTask.output)
    }
  }

  interface ClassListExploderParameters : WorkParameters {
    val classFiles: ConfigurableFileCollection
    val buildDir: DirectoryProperty
    val output: RegularFileProperty
  }

  abstract class ClassListExploderWorkAction : WorkAction<ClassListExploderParameters> {

    override fun execute() {
      val output = parameters.output.getAndDelete()

      val usedClasses = ClassFilesParser(
        classes = parameters.classFiles.asFileTree.files,
        buildDir = parameters.buildDir.get().asFile
      ).analyze()

      output.writeText(usedClasses.toJson())
    }
  }
}
