// Copyright (c) 2024. Tony Robalik.
// SPDX-License-Identifier: Apache-2.0
package com.autonomousapps.model.declaration

import com.autonomousapps.model.declaration.internal.Configurations
import com.squareup.moshi.JsonClass

/**
 * A "Variant" has two meanings depending on context:
 * 1. For the JVM, it is simply the source set (e.g., "main" and "test").
 * 2. For Android, it is the combination of _variant_ (e.g., "debug" and "release") and [SourceSetKind] ("main" and
 * "test").
 */
@JsonClass(generateAdapter = false)
data class Variant(
  /** The name of the source set (e.g., "main", "test", "debug", "release", etc.) */
  val variant: String,
  val kind: SourceSetKind
) : Comparable<Variant> {

  override fun compareTo(other: Variant): Int = compareBy(Variant::kind)
    .thenBy { it.variant }
    .compare(this, other)

  /** See [SourceSetKind.asBaseVariant]. */
  fun base(): Variant = kind.asBaseVariant(if (kind == SourceSetKind.CUSTOM_JVM) variant else null)

  companion object {
    const val MAIN_NAME = "main"
    const val TEST_NAME = "test"
    const val ANDROID_TEST_NAME = "androidTest"

    val MAIN = Variant(MAIN_NAME, SourceSetKind.MAIN)
    //val TEST = Variant(TEST_NAME, SourceSetKind.TEST)
    //val ANDROID_TEST = Variant(ANDROID_TEST_NAME, SourceSetKind.ANDROID_TEST)

    @JvmStatic
    fun of(configurationName: String, supportedSourceSets: Set<String>, hasCustomSourceSets: Boolean): Variant? =
      Configurations.variantFrom(configurationName, supportedSourceSets, hasCustomSourceSets)

    fun String.toVariant(kind: SourceSetKind): Variant {
      return if (isNotBlank()) {
        Variant(this, kind)
      } else {
        Variant(MAIN_NAME, kind)
      }
    }
  }
}
