package com.aventstack.extentreports.model;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.atomic.AtomicInteger;

import org.bson.types.ObjectId;

import com.aventstack.extentreports.ExtentReports;
import com.aventstack.extentreports.RunResult;
import com.aventstack.extentreports.Status;
import com.aventstack.extentreports.gherkin.model.IGherkinFormatterModel;
import com.google.gson.annotations.Expose;

public class Test implements Serializable, RunResult, BasicMongoReportElement {

	private static final long serialVersionUID = -8681630689550647312L;
	private static final AtomicInteger atomicInt = new AtomicInteger(0);

	/**
	 * An instance of {@link ExtentReports}
	 */
	private transient ExtentReports extent;

	/**
	 * Level describes the hierarchy of the test in the tree. A level 0 indicates a
	 * parent test. Level 1 indicates an immediate child of the parent, level 2
	 * indicates an immediate child of the Child and so on.. The bottom-most test in
	 * the hierarchy is considered a leaf
	 */
	private int level = 0;

	/**
	 * A unique ID, generated by AtomicInteger
	 */
	private transient int id = atomicInt.incrementAndGet();
	
	/**
     * An assigned {@link ObjectId}  
     */
    private ObjectId objectId;

	/**
	 * If this Test is at the top-most level, or in other words, has a level value
	 * of 0, parent will be null. This field will only contain a value if the Test
	 * is a child, or has a level 1 or greater.
	 */
	private transient Test parent;

	/**
	 * {@link Status} of this test, defaults to PASS
	 */
	private Status status = Status.PASS;

	/**
	 * A structure containing all nodes of this test
	 */
	private AbstractStructure<Test> nodeContext;

	/**
	 * A structure containing all logs/events of this test
	 */
	private AbstractStructure<Log> logContext;

	/**
	 * A structure containing all categories assigned to this test
	 */
	private AbstractStructure<Attribute> categoryContext;

	/**
	 * A structure containing all authors assigned to this test
	 */
	private AbstractStructure<Attribute> authorContext;

	/**
	 * A structure containing all devices assigned to this test
	 */
	private AbstractStructure<Attribute> deviceContext;

	/**
	 * Time the test was started
	 */
	private Date startTime = Calendar.getInstance().getTime();

	/**
	 * Time the test was ended
	 */
	private Date endTime = Calendar.getInstance().getTime();

	/**
	 * Type of BDD object represented by {@link IGherkinFormatterModel}. It can have
	 * one of the following values:
	 * 
	 * <ul>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.Feature}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.Background}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.Scenario}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.Given}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.When}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.Then}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.And}</li>
	 * <li>{@link com.aventstack.extentreports.gherkin.model.But}</li>
	 * </ul>
	 */
	private transient Class<? extends IGherkinFormatterModel> bddType;
	
	@Expose
	private String bddTypeName;

	/**
	 * A list of {@link ScreenCapture}
	 */
	private transient AbstractStructure<ScreenCapture> screenCaptureContext;

	/**
	 * A list of {@link ExceptionInfo}
	 */
	private transient AbstractStructure<ExceptionInfo> exceptionInfoContext;

	/**
	 * Name of the test
	 */
	private String name;

	/**
	 * Description
	 */
	private String description;

	/**
	 * This setting allows setting test with your own time-stamps. With this
	 * enabled, Extent does not use time-stamps for tests at the time they were
	 * created
	 */
	private boolean usesManualConfiguration = false;

	public ExtentReports getExtent() {
		return extent;
	}

	public void setExtent(ExtentReports extent) {
		this.extent = extent;
	}

	public int getLevel() {
		return level;
	}

	public void setLevel(int level) {
		this.level = level;
	}

	public int getId() {
		return id;
	}

	public void setId(int id) {
		this.id = id;
	}

	public Test getParent() {
		return parent;
	}

	public void setParent(Test parent) {
		this.parent = parent;
	}

	public Status getStatus() {
		return status;
	}

	public void setStatus(Status status) {
		this.status = status;
	}

	public AbstractStructure<Test> getNodeContext() {
		if (nodeContext == null) {
			nodeContext = new AbstractStructure<>();
		}
		return nodeContext;
	}

	public void setNodeContext(AbstractStructure<Test> node) {
		this.nodeContext = node;
	}

	public AbstractStructure<Log> getLogContext() {
		if (logContext == null) {
			logContext = new AbstractStructure<>();
		}
		return logContext;
	}

	public void setLogContext(AbstractStructure<Log> log) {
		this.logContext = log;
	}

	public AbstractStructure<Attribute> getCategoryContext() {
		if (categoryContext == null) {
			categoryContext = new AbstractStructure<>();
		}
		return categoryContext;
	}

	public void setCategoryContext(AbstractStructure<Attribute> category) {
		this.categoryContext = category;
	}

	public AbstractStructure<Attribute> getAuthorContext() {
		if (authorContext == null) {
			authorContext = new AbstractStructure<>();
		}
		return authorContext;
	}

	public void setAuthorContext(AbstractStructure<Attribute> author) {
		this.authorContext = author;
	}

	public AbstractStructure<Attribute> getDeviceContext() {
		if (deviceContext == null) {
			deviceContext = new AbstractStructure<>();
		}
		return deviceContext;
	}

	public void setDeviceContext(AbstractStructure<Attribute> device) {
		this.deviceContext = device;
	}

	public Date getStartTime() {
		return startTime;
	}

	public void setStartTime(Date startTime) {
		this.startTime = startTime;
	}

	public Date getEndTime() {
		return endTime;
	}

	public void setEndTime(Date endTime) {
		this.endTime = endTime;
	}

	public Class<? extends IGherkinFormatterModel> getBddType() {
		return bddType;
	}

	public void setBddType(Class<? extends IGherkinFormatterModel> bddType) {
		this.bddType = bddType;
		bddTypeName = getBehaviorDrivenTypeName();
	}

	public AbstractStructure<ScreenCapture> getScreenCaptureContext() {
		if (screenCaptureContext == null) {
			screenCaptureContext = new AbstractStructure<>();
		}
		return screenCaptureContext;
	}

	public void setScreenCaptureContext(AbstractStructure<ScreenCapture> screenCapture) {
		this.screenCaptureContext = screenCapture;
	}

	public AbstractStructure<ExceptionInfo> getExceptionInfoContext() {
		if (exceptionInfoContext == null) {
			exceptionInfoContext = new AbstractStructure<>();
		}
		return exceptionInfoContext;
	}

	public void setExceptionInfoContext(AbstractStructure<ExceptionInfo> exception) {
		this.exceptionInfoContext = exception;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public boolean isUsesManualConfiguration() {
		return usesManualConfiguration;
	}

	public void setUsesManualConfiguration(boolean usesManualConfiguration) {
		this.usesManualConfiguration = usesManualConfiguration;
	}

	private synchronized void updateStatus(Status logStatus) {
		int logStatusIndex = Status.getStatusHierarchy().indexOf(logStatus);
		int testStatusIndex = Status.getStatusHierarchy().indexOf(status);

		status = logStatusIndex < testStatusIndex ? logStatus : status;
	}

	public void end() {
		updateTestStatusRecursive(this);
		endChildrenRecursive(this);
		status = (status == Status.INFO || status == Status.DEBUG) ? Status.PASS : status;
		if (!usesManualConfiguration) {
			computeEndTimeFromChildren();
		}
	}

	private synchronized void updateTestStatusRecursive(Test test) {
		test.getLogContext().getAll().forEach(x -> updateStatus(x.getStatus()));

		if (!test.getNodeContext().isEmpty()) {
			test.getNodeContext().getAll().forEach(this::updateTestStatusRecursive);
		}

		// if not all children are marked SKIP, then:
		// ensure the parent has a status that is not SKIP
		if (test.bddType == null) {
			boolean hasNodeNotSkipped = test.getNodeContext().getAll().stream()
					.anyMatch(x -> x.getStatus() != Status.SKIP);

			if (status == Status.SKIP && hasNodeNotSkipped) {
				// reset status
				status = Status.PASS;
				// compute new status
				test.getNodeContext().getAll().stream().filter(x -> x.getStatus() != Status.SKIP)
						.forEach(this::updateTestStatusRecursive);
			}
		}
	}

	private void endChildrenRecursive(Test test) {
		test.getNodeContext().getAll().forEach(Test::end);
	}

	public void computeEndTimeFromChildren() {
		if (!getNodeContext().isEmpty()) {
			setStartTime(getNodeContext().getFirst().getStartTime());
			setEndTime(getNodeContext().getLast().getEndTime());
		} else if (!getLogContext().isEmpty()) {
			Date lastLogEndTime = getLogContext().getLast().getTimestamp();
			setEndTime(lastLogEndTime);
		}
	}

	public String getBehaviorDrivenTypeName() {
		if (bddTypeName != null) {
			return bddTypeName;
		}
		try {
			Method method = bddType.getMethod("getGherkinName");
			Object o = method.invoke(null, (Object[]) null);
			return o.toString();
		} catch (Exception e) {
			return null;
		}
	}

	public Boolean isBehaviorDrivenType() {
		return getBddType() != null;
	}

	@Override
	public ObjectId getObjectId() {
		return objectId;
	}

	@Override
	public void setObjectId(ObjectId id) {
		this.objectId = id;
	}

}
