package com.avioconsulting.mule.opentelemetry.internal.processor;

import com.avioconsulting.mule.opentelemetry.api.processor.ProcessorComponent;
import com.avioconsulting.mule.opentelemetry.api.traces.TraceComponent;
import com.avioconsulting.mule.opentelemetry.internal.processor.service.ComponentRegistryService;
import com.avioconsulting.mule.opentelemetry.internal.processor.util.TraceComponentManager;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.propagation.TextMapGetter;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.notification.EnrichedServerNotification;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.*;

import static com.avioconsulting.mule.opentelemetry.api.sdk.SemanticAttributes.*;
import static com.avioconsulting.mule.opentelemetry.internal.util.BatchHelperUtil.addBatchTags;
import static com.avioconsulting.mule.opentelemetry.internal.util.OpenTelemetryUtil.getEventTransactionId;

public abstract class AbstractProcessorComponent implements ProcessorComponent {

  static final String NAMESPACE_URI_MULE = "http://www.mulesoft.org/schema/mule/core";
  public static final String NAMESPACE_MULE = "mule";
  public static final String FLOW = "flow";

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractProcessorComponent.class);

  protected ExpressionManager expressionManager;
  protected ComponentRegistryService componentRegistryService;
  protected TraceComponentManager traceComponentManager = TraceComponentManager.getInstance();

  public ProcessorComponent withComponentRegistryService(ComponentRegistryService componentRegistryService) {
    this.componentRegistryService = componentRegistryService;
    return this;
  }

  @Override
  public ProcessorComponent withExpressionManager(ExpressionManager expressionManager) {
    this.expressionManager = expressionManager;
    return this;
  }

  protected abstract String getNamespace();

  protected abstract List<String> getOperations();

  protected abstract List<String> getSources();

  protected SpanKind getSpanKind() {
    return SpanKind.INTERNAL;
  }

  @Override
  public boolean canHandle(ComponentIdentifier componentIdentifier) {
    return getNamespace().equalsIgnoreCase(componentIdentifier.getNamespace())
        && (getOperations().contains(componentIdentifier.getName().toLowerCase())
            || getSources().contains(componentIdentifier.getName().toLowerCase()));
  }

  protected boolean namespaceSupported(ComponentIdentifier componentIdentifier) {
    return getNamespace().equalsIgnoreCase(componentIdentifier.getNamespace().toLowerCase());
  }

  protected boolean operationSupported(ComponentIdentifier componentIdentifier) {
    return getOperations().contains(componentIdentifier.getName().toLowerCase());
  }

  protected boolean sourceSupported(ComponentIdentifier componentIdentifier) {
    return getSources().contains(componentIdentifier.getName().toLowerCase());
  }

  @Override
  public TraceComponent getEndTraceComponent(EnrichedServerNotification notification) {
    return getTraceComponentBuilderFor(notification);
  }

  protected TraceComponent getTraceComponentBuilderFor(EnrichedServerNotification notification) {
    TraceComponent traceComponent = TraceComponentManager.getInstance()
        .createTraceComponent(getTransactionId(notification), notification.getResourceIdentifier(),
            notification.getComponent().getLocation())
        .withErrorMessage(
            notification.getEvent().getError().map(Error::getDescription).orElse(null));
    addBatchTags(traceComponent, notification.getEvent());
    return traceComponent;
  }

  protected String getTransactionId(EnrichedServerNotification notification) {
    return getEventTransactionId(notification.getEvent());
  }

  protected void addProcessorCommonTags(Component component, TraceComponent collector) {
    ComponentWrapper componentWrapper = componentRegistryService.getComponentWrapper(component);
    collector.addAllTags(componentWrapper.staticParametersAsReadOnlyMap());
  }

  protected ComponentIdentifier getSourceIdentifier(EnrichedServerNotification notification) {
    ComponentIdentifier sourceIdentifier = null;
    if (notification.getEvent() != null
        && notification.getEvent().getContext().getOriginatingLocation() != null
        && notification.getResourceIdentifier().equalsIgnoreCase(
            notification.getEvent().getContext().getOriginatingLocation().getRootContainerName())) {
      sourceIdentifier = notification
          .getEvent()
          .getContext()
          .getOriginatingLocation()
          .getComponentIdentifier()
          .getIdentifier();
    }
    return sourceIdentifier;
  }

  /**
   * Adds attributes to collector if needed
   * 
   * @param component
   * @param attributes
   * @param collector
   * @param <A>
   */
  protected <A> void addAttributes(Component component, TypedValue<A> attributes,
      final TraceComponent collector) {
    // nothing to do in the abstract
  }

  @Override
  public TraceComponent getStartTraceComponent(EnrichedServerNotification notification) {
    return getStartTraceComponent(notification.getComponent(), notification.getEvent());
  }

  /**
   * Create a start trace component without the notification object. This is
   * mostly consumed by interceptors.
   *
   * @param component
   *            {@link Component}
   * @param event
   *            {@link Message}
   * @return TraceComponent
   */
  public TraceComponent getStartTraceComponent(Component component, Event event) {
    ComponentWrapper componentWrapper = componentRegistryService.getComponentWrapper(component);
    TraceComponent traceComponent = TraceComponentManager.getInstance()
        .createTraceComponent(getEventTransactionId(event), component)
        .withSpanName(componentWrapper.getDefaultSpanName())
        .withSpanKind(getSpanKind())
        .withEventContextId(event.getContext().getId());

    addProcessorCommonTags(component, traceComponent);
    traceComponent.addTag(MULE_CORRELATION_ID.getKey(), event.getCorrelationId());
    addAttributes(component, event.getMessage().getAttributes(), traceComponent);

    addBatchTags(traceComponent, event);
    return traceComponent;
  }

  protected void addTagIfPresent(Map<String, String> sourceMap, String sourceKey, TraceComponent targetMap,
      String targetKey) {
    if (sourceMap.containsKey(sourceKey))
      targetMap.addTag(targetKey, sourceMap.get(sourceKey));
  }

  protected Component getSourceComponent(EnrichedServerNotification notification) {
    return componentRegistryService.findComponentByLocation(
        notification.getEvent().getContext().getOriginatingLocation().getLocation());
  }

  protected enum ContextMapGetter implements TextMapGetter<Map<String, ? super String>> {
    INSTANCE;

    @Override
    public Iterable<String> keys(Map<String, ? super String> map) {
      return map.keySet();
    }

    @Nullable
    @Override
    public String get(@Nullable Map<String, ? super String> map, String s) {
      return (map == null || map.get(s) == null) ? null : map.get(s).toString();
    }
  }
}
