package com.avioconsulting.mule.opentelemetry.internal.util;

import com.avioconsulting.mule.opentelemetry.api.traces.TraceComponent;
import com.avioconsulting.mule.opentelemetry.internal.opentelemetry.sdk.AttributesKeyCache;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.AttributesBuilder;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanBuilder;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.event.EventContext;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

import static com.avioconsulting.mule.opentelemetry.api.store.TransactionStore.OTEL_BATCH_PARENT_CONTEXT_ID;
import static com.avioconsulting.mule.opentelemetry.internal.util.BatchHelperUtil.*;

public class OpenTelemetryUtil {

  private static final Logger LOGGER = LoggerFactory.getLogger(OpenTelemetryUtil.class);
  private static final AttributesKeyCache attributesKeyCache = new AttributesKeyCache();

  /**
   * <pre>
   * Extract any attributes defined via system properties (see {@link System#getProperties()}) for provided <code>configName</code>.
   *
   * It uses `{configName}.otel.{attributeKey}` pattern to identify relevant system properties. Key matching is case-insensitive.
   * </pre>
   *
   * @param configName
   *            {@link String} name of the component's global configuration
   *            element
   * @param sourceMap
   *            {@link Map} contains all properties to search in
   * @return Map of a key-value pair resolved from given source map
   */
  public static Map<String, String> getGlobalConfigSystemAttributes(String configName,
      Map<String, String> sourceMap) {
    if (configName == null || configName.trim().isEmpty()
        || sourceMap == null || sourceMap.isEmpty()) {
      return Collections.emptyMap();
    }
    String configRef = configName.toLowerCase();
    String replaceVal = configRef + ".otel.";
    Map<String, String> newTags = new HashMap<>();
    for (Map.Entry<String, String> e : sourceMap.entrySet()) {
      if (e.getKey().startsWith(configRef)) {
        String propKey = e.getKey().substring(replaceVal.length());
        newTags.put(propKey, e.getValue());
      }
    }
    return newTags;
  }

  /**
   * This method uses {@link EventContext#getId()} for extracting the unique id
   * for current event processing.
   *
   * @param event
   *            {@link Event} to extract id from
   * @return String id for the current event
   */
  public static String getEventTransactionId(Event event) {
    String transactionId = null;
    if (event.getVariables().containsKey(OTEL_BATCH_PARENT_CONTEXT_ID)
        || (transactionId = getBatchJobInstanceId(event)) == null) {
      transactionId = getEventTransactionId(event.getContext().getId());
    }
    return transactionId;
  }

  /**
   * Creates a unique id for current event processing.
   *
   * @param eventId
   *            {@link Event} to extract id from
   * @return String id for the current event
   */
  public static String getEventTransactionId(String eventId) {
    // For child contexts, the primary id is appended with "_{timeInMillis}".
    // We remove time part to get a unique id across the event processing.
    int index = eventId.indexOf('_');
    return (index != -1) ? eventId.substring(0, index) : eventId;
  }

  /**
   * If given system property exists, this will set its value as an attribute to
   * provided {@link AttributesBuilder}.
   *
   * @param property
   *            Name of the system property to search for.
   * @param builder
   *            {@link AttributesBuilder} instance to add attribute
   * @param attributeKey
   *            {@link AttributeKey} to use for setting attribute in given
   *            {@link AttributesBuilder}
   */
  public static void addAttribute(String property, AttributesBuilder builder,
      AttributeKey<String> attributeKey) {
    String value = PropertiesUtil.getProperty(property);
    if (value != null) {
      builder.put(attributeKey, value);
    }
  }

  /**
   * Resolves any expressions in the TraceComponent's spanName and tags using the
   * provided ExpressionManager
   * based on the given Event.
   *
   * @param traceComponent
   *            the TraceComponent containing spanName and tags to resolve
   * @param expressionManager
   *            the ExpressionManager used to evaluate expressions
   * @param event
   *            the Event used for context in expression evaluation
   */
  public static void resolveExpressions(TraceComponent traceComponent, ExpressionManager expressionManager,
      Event event) {
    try {
      if (expressionManager
          .isExpression(traceComponent.getSpanName())) {
        String value = resolveExpression(traceComponent.getSpanName(), expressionManager, event);
        if (value != null) {
          traceComponent.withSpanName(value);
        }
      }
      traceComponent.forEachTagEntry(entry -> {
        if (expressionManager.isExpression(entry.getValue())) {
          try {
            entry.setValue(resolveExpression(entry.getValue(), expressionManager, event));
          } catch (Exception ignored) {
          }
        }
      });
    } catch (Exception ignored) {
    }
  }

  public static String resolveExpression(String expression, ExpressionManager expressionManager, Event event)
      throws Exception {
    TypedValue evaluate = expressionManager.evaluate(expression, DataType.STRING, event.asBindingContext());
    return typedValueToString(evaluate);
  }

  /**
   * Converts a given TypedValue into a String representation.
   * The method handles different types of input including CursorStreamProvider,
   * InputStream,
   * and other generic object types.
   *
   * @param typedValue
   *            The TypedValue object to be converted to a String.
   * @return A String representation of the TypedValue.
   * @throws Exception
   *             If an error occurs during the conversion process, such as IO
   *             errors.
   */
  public static String typedValueToString(TypedValue typedValue) throws Exception {
    String value = "";
    Object input = typedValue.getValue();
    if (input instanceof CursorStreamProvider) {
      value = IOUtils.toString(((CursorStreamProvider) input).openCursor());
    } else if (input instanceof InputStream) {
      value = IOUtils.toString((InputStream) input);
    } else {
      value = TypedValue.unwrap(typedValue).toString();
    }
    return value;
  }

  public static void tagsToAttributes(TraceComponent traceComponent, SpanBuilder spanBuilder) {
    if (!traceComponent.hasTags()) {
      return;
    }
    traceComponent.forEachTagEntry(entry -> {
      AttributeKey attributeKey = attributesKeyCache.getAttributeKey(entry.getKey());
      spanBuilder.setAttribute(attributeKey, attributesKeyCache.convertValue(attributeKey, entry.getValue()));
    });
  }

  public static void tagsToAttributes(TraceComponent traceComponent, Span span) {
    if (!traceComponent.hasTags()) {
      return;
    }
    traceComponent.forEachTagEntry(entry -> {
      AttributeKey attributeKey = attributesKeyCache.getAttributeKey(entry.getKey());
      span.setAttribute(attributeKey, attributesKeyCache.convertValue(attributeKey, entry.getValue()));
    });
  }
}
