/*
 * Decompiled with CFR 0.152.
 */
package com.avioconsulting.mule.opentelemetry.tools;

import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SystemPropertyExtractor {
    private static final Map<String, PropertyInfo> systemProperties = new LinkedHashMap<String, PropertyInfo>();

    public static void main(String[] args) {
        try {
            String baseDir = args[0];
            System.out.println("Extracting system properties from project at: " + baseDir + " ...");
            Path projectRoot = Paths.get(baseDir, "src/main/java");
            SystemPropertyExtractor.extractSystemProperties(projectRoot);
            SystemPropertyExtractor.generateDocumentation(baseDir);
        }
        catch (IOException e) {
            System.err.println("Error extracting system properties: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private static void extractSystemProperties(Path projectRoot) throws IOException {
        systemProperties.clear();
        Files.walk(projectRoot, new FileVisitOption[0]).filter(path -> path.toString().endsWith(".java")).forEach(SystemPropertyExtractor::extractFromFile);
    }

    private static void addProperty(String name, String defaultValue, String description, String location) {
        PropertyInfo prop = new PropertyInfo(name, location);
        prop.defaultValue = defaultValue;
        prop.description = description;
        systemProperties.put(name, prop);
    }

    private static void extractFromFile(Path filePath) {
        try {
            if (filePath.endsWith("SystemPropertyExtractor.java")) {
                return;
            }
            String content = new String(Files.readAllBytes(filePath));
            Pattern propertyWithJavadocPattern = Pattern.compile("/\\*\\*([^*]*(?:\\*(?!/)[^*]*)*)\\*/\\s*(private|public)\\s+static\\s+final\\s+String\\s+\\w+\\s*=\\s*\"([a-z][a-z0-9._-]+)\"", 8);
            Matcher javadocMatcher = propertyWithJavadocPattern.matcher(content);
            while (javadocMatcher.find()) {
                String javadoc = javadocMatcher.group(1);
                String potentialProperty = javadocMatcher.group(3);
                if (!potentialProperty.startsWith("mule.otel.")) continue;
                String defaultValue = SystemPropertyExtractor.extractDefaultValue(javadoc);
                String cleanJavadoc = SystemPropertyExtractor.cleanJavadoc(javadoc);
                if (!systemProperties.containsKey(potentialProperty)) {
                    PropertyInfo prop = new PropertyInfo(potentialProperty, filePath.toString());
                    prop.description = "Property found in codebase";
                    prop.javadocDescription = cleanJavadoc;
                    prop.javadocDefaultValue = defaultValue;
                    systemProperties.put(potentialProperty, prop);
                    continue;
                }
                PropertyInfo existingProp = systemProperties.get(potentialProperty);
                if (existingProp.javadocDescription == null || existingProp.javadocDescription.isEmpty()) {
                    existingProp.javadocDescription = cleanJavadoc;
                }
                if (existingProp.javadocDefaultValue != null && !existingProp.javadocDefaultValue.isEmpty()) continue;
                existingProp.javadocDefaultValue = defaultValue;
            }
            Pattern propertyPattern = Pattern.compile("\"([a-z][a-z0-9._-]+)\"");
            Matcher matcher = propertyPattern.matcher(content);
            while (matcher.find()) {
                String potentialProperty = matcher.group(1);
                if (!potentialProperty.startsWith("mule.otel.") || systemProperties.containsKey(potentialProperty)) continue;
                SystemPropertyExtractor.addProperty(potentialProperty, "", "Property found in codebase", filePath.toString());
            }
        }
        catch (IOException e) {
            System.err.println("Error reading file " + filePath + ": " + e.getMessage());
        }
    }

    private static String extractDefaultValue(String javadoc) {
        if (javadoc == null) {
            return "";
        }
        Pattern defaultPattern = Pattern.compile("@[Dd]efault\\s+([^\\s\\*]+)", 2);
        Matcher matcher = defaultPattern.matcher(javadoc);
        if (matcher.find()) {
            return matcher.group(1).trim();
        }
        return "";
    }

    private static String cleanJavadoc(String javadoc) {
        if (javadoc == null) {
            return "";
        }
        return javadoc.replaceAll("@[Dd]efault\\s+[^\\s\\*]+", "").replaceAll("\\*", "").replaceAll("\\n\\s*", " ").replaceAll("\\s+", " ").trim();
    }

    private static void generateDocumentation(String baseDir) {
        try {
            Path outputPath = Paths.get(baseDir, "src/docs/asciidoc/system-properties-reference.adoc");
            Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
            try (PrintWriter writer = new PrintWriter(Files.newBufferedWriter(outputPath, new OpenOption[0]));){
                writer.println("=== System Properties Reference");
                writer.println();
                writer.println("The following table lists all system properties supported by the Mule OpenTelemetry Module:");
                writer.println();
                writer.println("NOTE: *Environment variables take precedence over system properties*. System properties take precedence over configuration values.");
                writer.println();
                writer.println(".System Properties");
                writer.println("|===");
                writer.println("|System Property |Environment Variable |Description |Default Value");
                writer.println();
                for (PropertyInfo prop : systemProperties.values()) {
                    if (!prop.name.startsWith("mule.otel.")) continue;
                    writer.printf("|%s%n", prop.name);
                    writer.printf("|%s%n", prop.environmentVariable);
                    writer.printf("|%s%n", prop.getEffectiveDescription());
                    String effectiveDefault = prop.getEffectiveDefaultValue();
                    writer.printf("|%s%n", effectiveDefault == null || effectiveDefault.isEmpty() ? "Not set" : "`" + effectiveDefault + "`");
                    writer.println();
                }
                writer.println("|===");
                writer.println();
                System.out.println("Generated system properties documentation at: " + outputPath.toAbsolutePath());
                System.out.println("Found " + systemProperties.size() + " system properties");
            }
        }
        catch (IOException e) {
            System.err.println("Error writing documentation: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private static class PropertyInfo {
        String name;
        String defaultValue;
        String description;
        String javadocDescription;
        String javadocDefaultValue;
        String location;
        String environmentVariable;

        PropertyInfo(String name, String location) {
            this.name = name;
            this.location = location;
            this.environmentVariable = this.toEnvName(name);
        }

        private String toEnvName(String propertyName) {
            return propertyName.toUpperCase(Locale.ROOT).replace('.', '_').replace('-', '_');
        }

        public String getEffectiveDescription() {
            return this.javadocDescription != null && !this.javadocDescription.isEmpty() ? this.javadocDescription : this.description;
        }

        public String getEffectiveDefaultValue() {
            return this.javadocDefaultValue != null && !this.javadocDefaultValue.isEmpty() ? this.javadocDefaultValue : this.defaultValue;
        }
    }
}

