// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.ai.openai.responses.implementation;

import com.azure.core.annotation.BodyParam;
import com.azure.core.annotation.Delete;
import com.azure.core.annotation.ExpectedResponses;
import com.azure.core.annotation.Get;
import com.azure.core.annotation.HeaderParam;
import com.azure.core.annotation.Host;
import com.azure.core.annotation.HostParam;
import com.azure.core.annotation.PathParam;
import com.azure.core.annotation.Post;
import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceInterface;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.annotation.UnexpectedResponseExceptionType;
import com.azure.core.exception.ClientAuthenticationException;
import com.azure.core.exception.HttpResponseException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.rest.RequestOptions;
import com.azure.core.http.rest.Response;
import com.azure.core.http.rest.RestProxy;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.serializer.SerializerAdapter;
import reactor.core.publisher.Mono;

/**
 * Implementation for calling Non-Azure OpenAI Responses Service
 */
public final class NonAzureResponsesClientImpl {
    /**
     * The proxy service used to perform REST calls.
     */
    private final NonAzureResponsesClientService service;

    /**
     * The HTTP pipeline to send requests through.
     */
    private final HttpPipeline httpPipeline;

    /**
     * Gets The HTTP pipeline to send requests through.
     *
     * @return the httpPipeline value.
     */
    public HttpPipeline getHttpPipeline() {
        return this.httpPipeline;
    }

    /**
     * The serializer to serialize an object into a string.
     */
    private final SerializerAdapter serializerAdapter;

    /**
     * Gets The serializer to serialize an object into a string.
     *
     * @return the serializerAdapter value.
     */
    public SerializerAdapter getSerializerAdapter() {
        return this.serializerAdapter;
    }

    /**
     * This is the endpoint that non-azure OpenAI supports. Currently, it has only v1 version.
     */
    public static final String OPEN_AI_ENDPOINT = "https://api.openai.com/v1";

    /**
     * Initializes an instance of NonAzureResponsesClientImpl client.
     *
     * @param httpPipeline      The HTTP pipeline to send requests through.
     * @param serializerAdapter The serializer to serialize an object into a string.
     */
    public NonAzureResponsesClientImpl(HttpPipeline httpPipeline, SerializerAdapter serializerAdapter) {
        this.httpPipeline = httpPipeline;
        this.serializerAdapter = serializerAdapter;
        this.service
            = RestProxy.create(NonAzureResponsesClientService.class, this.httpPipeline, this.getSerializerAdapter());
    }

    /**
     * The interface defining all the services for NonAzureResponsesClientImpl to be used by the proxy service to perform REST calls.
     */
    @Host("{endpoint}")
    @ServiceInterface(name = "NonAzureResponsesCli")
    public interface NonAzureResponsesClientService {
        @Post("/responses")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> createResponse(@HostParam("endpoint") String endpoint,
            @HeaderParam("accept") String accept, @HeaderParam("Content-Type") String contentType,
            @BodyParam("application/json") BinaryData requestBody, RequestOptions requestOptions, Context context);

        @Post("/responses")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> createResponseSync(@HostParam("endpoint") String endpoint,
            @HeaderParam("accept") String accept, @HeaderParam("Content-Type") String contentType,
            @BodyParam("application/json") BinaryData requestBody, RequestOptions requestOptions, Context context);

        @Get("/responses/{response_id}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getResponse(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/responses/{response_id}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getResponseSync(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Delete("/responses/{response_id}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> deleteResponse(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Delete("/responses/{response_id}")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> deleteResponseSync(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/responses/{response_id}/input_items")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> listInputItems(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/responses/{response_id}/input_items")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> listInputItemsSync(@HostParam("endpoint") String endpoint,
            @PathParam("response_id") String responseId, @HeaderParam("Accept") String accept,
            RequestOptions requestOptions, Context context);
    }

    /**
     * Creates a model response.
     * <p><strong>Request Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     model: String(o3-mini/o3-mini-2025-01-31/o1/o1-2024-12-17/o1-preview/o1-preview-2024-09-12/o1-mini/o1-mini-2024-09-12/computer-use-preview/computer-use-preview-2025-02-04/computer-use-preview-2025-03-11/gpt-4.5-preview/gpt-4.5-preview-2025-02-27/gpt-4o/gpt-4o-2024-11-20/gpt-4o-2024-08-06/gpt-4o-2024-05-13/gpt-4o-audio-preview/gpt-4o-audio-preview-2024-10-01/gpt-4o-audio-preview-2024-12-17/gpt-4o-mini-audio-preview/gpt-4o-mini-audio-preview-2024-12-17/chatgpt-4o-latest/gpt-4o-mini/gpt-4o-mini-2024-07-18/gpt-4-turbo/gpt-4-turbo-2024-04-09/gpt-4-0125-preview/gpt-4-turbo-preview/gpt-4-1106-preview/gpt-4-vision-preview/gpt-4/gpt-4-0314/gpt-4-0613/gpt-4-32k/gpt-4-32k-0314/gpt-4-32k-0613/gpt-3.5-turbo/gpt-3.5-turbo-16k/gpt-3.5-turbo-0301/gpt-3.5-turbo-0613/gpt-3.5-turbo-1106/gpt-3.5-turbo-0125/gpt-3.5-turbo-16k-0613) (Required)
     *     metadata (Optional): {
     *         String: String (Required)
     *     }
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     previous_response_id: String (Optional)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     max_output_tokens: Integer (Optional)
     *     instructions: String (Optional)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tools (Optional): [
     *          (Optional){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     tool_choice: BinaryData (Optional)
     *     truncation: String(auto/disabled) (Optional)
     *     user: String (Optional)
     *     input: BinaryData (Required)
     *     include (Optional): [
     *         String(file_search_call.results/computer_call_output.output.image_url/message.input_image.image_url) (Optional)
     *     ]
     *     parallel_tool_calls: Boolean (Optional)
     *     store: Boolean (Optional)
     *     stream: Boolean (Optional)
     * }
     * }
     * </pre>
     *
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     id: String (Required)
     *     object: String (Required)
     *     created_at: long (Required)
     *     status: String(completed/in_progress/failed/incomplete) (Optional)
     *     error (Required): {
     *         code: String (Required)
     *         message: String (Required)
     *     }
     *     incomplete_details (Required): {
     *         reason: String(max_output_tokens/content_filter) (Optional)
     *     }
     *     instructions: String (Required)
     *     max_output_tokens: Integer (Optional)
     *     model: String (Required)
     *     output (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     parallel_tool_calls: boolean (Required)
     *     previous_response_id: String (Required)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     temperature: double (Required)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tool_choice: BinaryData (Required)
     *     tools (Required): [
     *          (Required){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     top_p: double (Required)
     *     truncation: String(auto/disabled) (Optional)
     *     usage (Optional): {
     *         input_tokens: int (Required)
     *         output_tokens: int (Required)
     *         total_tokens: int (Required)
     *         output_tokens_details (Required): {
     *             reasoning_tokens: int (Required)
     *         }
     *     }
     *     user: String (Optional)
     *     metadata (Required): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     *
     * @param accept The accept parameter. Allowed values: "application/json", "text/event-stream".
     * @param requestBody The requestBody parameter.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> createResponseWithResponseAsync(String accept, BinaryData requestBody,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        return FluxUtil.withContext(context -> service.createResponse(OPEN_AI_ENDPOINT, accept, contentType,
            requestBody, requestOptions, context));
    }

    /**
     * Creates a model response.
     * <p><strong>Request Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     model: String(o3-mini/o3-mini-2025-01-31/o1/o1-2024-12-17/o1-preview/o1-preview-2024-09-12/o1-mini/o1-mini-2024-09-12/computer-use-preview/computer-use-preview-2025-02-04/computer-use-preview-2025-03-11/gpt-4.5-preview/gpt-4.5-preview-2025-02-27/gpt-4o/gpt-4o-2024-11-20/gpt-4o-2024-08-06/gpt-4o-2024-05-13/gpt-4o-audio-preview/gpt-4o-audio-preview-2024-10-01/gpt-4o-audio-preview-2024-12-17/gpt-4o-mini-audio-preview/gpt-4o-mini-audio-preview-2024-12-17/chatgpt-4o-latest/gpt-4o-mini/gpt-4o-mini-2024-07-18/gpt-4-turbo/gpt-4-turbo-2024-04-09/gpt-4-0125-preview/gpt-4-turbo-preview/gpt-4-1106-preview/gpt-4-vision-preview/gpt-4/gpt-4-0314/gpt-4-0613/gpt-4-32k/gpt-4-32k-0314/gpt-4-32k-0613/gpt-3.5-turbo/gpt-3.5-turbo-16k/gpt-3.5-turbo-0301/gpt-3.5-turbo-0613/gpt-3.5-turbo-1106/gpt-3.5-turbo-0125/gpt-3.5-turbo-16k-0613) (Required)
     *     metadata (Optional): {
     *         String: String (Required)
     *     }
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     previous_response_id: String (Optional)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     max_output_tokens: Integer (Optional)
     *     instructions: String (Optional)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tools (Optional): [
     *          (Optional){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     tool_choice: BinaryData (Optional)
     *     truncation: String(auto/disabled) (Optional)
     *     user: String (Optional)
     *     input: BinaryData (Required)
     *     include (Optional): [
     *         String(file_search_call.results/computer_call_output.output.image_url/message.input_image.image_url) (Optional)
     *     ]
     *     parallel_tool_calls: Boolean (Optional)
     *     store: Boolean (Optional)
     *     stream: Boolean (Optional)
     * }
     * }
     * </pre>
     *
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     id: String (Required)
     *     object: String (Required)
     *     created_at: long (Required)
     *     status: String(completed/in_progress/failed/incomplete) (Optional)
     *     error (Required): {
     *         code: String (Required)
     *         message: String (Required)
     *     }
     *     incomplete_details (Required): {
     *         reason: String(max_output_tokens/content_filter) (Optional)
     *     }
     *     instructions: String (Required)
     *     max_output_tokens: Integer (Optional)
     *     model: String (Required)
     *     output (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     parallel_tool_calls: boolean (Required)
     *     previous_response_id: String (Required)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     temperature: double (Required)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tool_choice: BinaryData (Required)
     *     tools (Required): [
     *          (Required){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     top_p: double (Required)
     *     truncation: String(auto/disabled) (Optional)
     *     usage (Optional): {
     *         input_tokens: int (Required)
     *         output_tokens: int (Required)
     *         total_tokens: int (Required)
     *         output_tokens_details (Required): {
     *             reasoning_tokens: int (Required)
     *         }
     *     }
     *     user: String (Optional)
     *     metadata (Required): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     *
     * @param accept The accept parameter. Allowed values: "application/json", "text/event-stream".
     * @param requestBody The requestBody parameter.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> createResponseWithResponse(String accept, BinaryData requestBody,
        RequestOptions requestOptions) {
        final String contentType = "application/json";
        return service.createResponseSync(OPEN_AI_ENDPOINT, accept, contentType, requestBody, requestOptions,
            Context.NONE);
    }

    /**
     * Retrieves a model response with the given ID.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>include[]</td><td>List&lt;String&gt;</td><td>No</td><td>The includables parameter. Call
     * {@link RequestOptions#addQueryParam} to add string to array.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     id: String (Required)
     *     object: String (Required)
     *     created_at: long (Required)
     *     status: String(completed/in_progress/failed/incomplete) (Optional)
     *     error (Required): {
     *         code: String (Required)
     *         message: String (Required)
     *     }
     *     incomplete_details (Required): {
     *         reason: String(max_output_tokens/content_filter) (Optional)
     *     }
     *     instructions: String (Required)
     *     max_output_tokens: Integer (Optional)
     *     model: String (Required)
     *     output (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     parallel_tool_calls: boolean (Required)
     *     previous_response_id: String (Required)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     temperature: double (Required)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tool_choice: BinaryData (Required)
     *     tools (Required): [
     *          (Required){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     top_p: double (Required)
     *     truncation: String(auto/disabled) (Optional)
     *     usage (Optional): {
     *         input_tokens: int (Required)
     *         output_tokens: int (Required)
     *         total_tokens: int (Required)
     *         output_tokens_details (Required): {
     *             reasoning_tokens: int (Required)
     *         }
     *     }
     *     user: String (Optional)
     *     metadata (Required): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     *
     * @param responseId The ID of the response to retrieve.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getResponseWithResponseAsync(String responseId, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getResponse(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, context));
    }

    /**
     * Retrieves a model response with the given ID.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>include[]</td><td>List&lt;String&gt;</td><td>No</td><td>The includables parameter. Call
     * {@link RequestOptions#addQueryParam} to add string to array.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     id: String (Required)
     *     object: String (Required)
     *     created_at: long (Required)
     *     status: String(completed/in_progress/failed/incomplete) (Optional)
     *     error (Required): {
     *         code: String (Required)
     *         message: String (Required)
     *     }
     *     incomplete_details (Required): {
     *         reason: String(max_output_tokens/content_filter) (Optional)
     *     }
     *     instructions: String (Required)
     *     max_output_tokens: Integer (Optional)
     *     model: String (Required)
     *     output (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     parallel_tool_calls: boolean (Required)
     *     previous_response_id: String (Required)
     *     reasoning (Optional): {
     *         effort: String(low/medium/high) (Required)
     *         generate_summary: String(concise/detailed) (Optional)
     *     }
     *     temperature: double (Required)
     *     text (Optional): {
     *         format (Optional): {
     *             type: String(text/json_object/json_schema) (Required)
     *         }
     *     }
     *     tool_choice: BinaryData (Required)
     *     tools (Required): [
     *          (Required){
     *             type: String(function/file_search/web_search_preview/computer_use_preview) (Required)
     *         }
     *     ]
     *     top_p: double (Required)
     *     truncation: String(auto/disabled) (Optional)
     *     usage (Optional): {
     *         input_tokens: int (Required)
     *         output_tokens: int (Required)
     *         total_tokens: int (Required)
     *         output_tokens_details (Required): {
     *             reasoning_tokens: int (Required)
     *         }
     *     }
     *     user: String (Optional)
     *     metadata (Required): {
     *         String: String (Required)
     *     }
     * }
     * }
     * </pre>
     *
     * @param responseId The ID of the response to retrieve.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getResponseWithResponse(String responseId, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getResponseSync(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, Context.NONE);
    }

    /**
     * Deletes a response by ID.
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     object: String (Required)
     *     id: String (Required)
     *     deleted: boolean (Required)
     * }
     * }
     * </pre>
     *
     * @param responseId The responseId parameter.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> deleteResponseWithResponseAsync(String responseId,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.deleteResponse(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, context));
    }

    /**
     * Deletes a response by ID.
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     object: String (Required)
     *     id: String (Required)
     *     deleted: boolean (Required)
     * }
     * }
     * </pre>
     *
     * @param responseId The responseId parameter.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> deleteResponseWithResponse(String responseId, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.deleteResponseSync(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, Context.NONE);
    }

    /**
     * Returns a list of input items for a given response.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>limit</td><td>Integer</td><td>No</td><td>A limit on the number of objects to be returned. Limit can range
     * between 1 and 100, and the
     * default is 20.</td></tr>
     * <tr><td>order</td><td>String</td><td>No</td><td>Sort order by the `created_at` timestamp of the objects. `asc`
     * for ascending order and`desc`
     * for descending order. Allowed values: "asc", "desc".</td></tr>
     * <tr><td>after</td><td>String</td><td>No</td><td>A cursor for use in pagination. `after` is an object ID that
     * defines your place in the list.
     * For instance, if you make a list request and receive 100 objects, ending with obj_foo, your
     * subsequent call can include after=obj_foo in order to fetch the next page of the list.</td></tr>
     * <tr><td>before</td><td>String</td><td>No</td><td>A cursor for use in pagination. `before` is an object ID that
     * defines your place in the list.
     * For instance, if you make a list request and receive 100 objects, ending with obj_foo, your
     * subsequent call can include before=obj_foo in order to fetch the previous page of the list.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     object: String (Required)
     *     data (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     first_id: String (Required)
     *     last_id: String (Required)
     *     has_more: boolean (Required)
     * }
     * }
     * </pre>
     *
     * @param responseId The ID of the response to retrieve.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> listInputItemsWithResponseAsync(String responseId,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.listInputItems(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, context));
    }

    /**
     * Returns a list of input items for a given response.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>limit</td><td>Integer</td><td>No</td><td>A limit on the number of objects to be returned. Limit can range
     * between 1 and 100, and the
     * default is 20.</td></tr>
     * <tr><td>order</td><td>String</td><td>No</td><td>Sort order by the `created_at` timestamp of the objects. `asc`
     * for ascending order and`desc`
     * for descending order. Allowed values: "asc", "desc".</td></tr>
     * <tr><td>after</td><td>String</td><td>No</td><td>A cursor for use in pagination. `after` is an object ID that
     * defines your place in the list.
     * For instance, if you make a list request and receive 100 objects, ending with obj_foo, your
     * subsequent call can include after=obj_foo in order to fetch the next page of the list.</td></tr>
     * <tr><td>before</td><td>String</td><td>No</td><td>A cursor for use in pagination. `before` is an object ID that
     * defines your place in the list.
     * For instance, if you make a list request and receive 100 objects, ending with obj_foo, your
     * subsequent call can include before=obj_foo in order to fetch the previous page of the list.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Response Body Schema</strong></p>
     *
     * <pre>
     * {@code
     * {
     *     object: String (Required)
     *     data (Required): [
     *          (Required){
     *             type: String(message/file_search_call/function_call/function_call_output/computer_call/computer_call_output/web_search_call/item_reference/reasoning) (Required)
     *             id: String (Optional)
     *         }
     *     ]
     *     first_id: String (Required)
     *     last_id: String (Required)
     *     has_more: boolean (Required)
     * }
     * }
     * </pre>
     *
     * @param responseId The ID of the response to retrieve.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> listInputItemsWithResponse(String responseId, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.listInputItemsSync(OPEN_AI_ENDPOINT, responseId, accept, requestOptions, Context.NONE);
    }
}
