// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
// Code generated by Microsoft (R) TypeSpec Code Generator.

package com.azure.ai.openai.implementation;

import com.azure.ai.openai.OpenAIServiceVersion;
import com.azure.core.annotation.BodyParam;
import com.azure.core.annotation.ExpectedResponses;
import com.azure.core.annotation.HeaderParam;
import com.azure.core.annotation.Host;
import com.azure.core.annotation.HostParam;
import com.azure.core.annotation.PathParam;
import com.azure.core.annotation.Post;
import com.azure.core.annotation.QueryParam;
import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceInterface;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.annotation.UnexpectedResponseExceptionType;
import com.azure.core.exception.ClientAuthenticationException;
import com.azure.core.exception.HttpResponseException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.HttpPipelineBuilder;
import com.azure.core.http.policy.RetryPolicy;
import com.azure.core.http.policy.UserAgentPolicy;
import com.azure.core.http.rest.RequestOptions;
import com.azure.core.http.rest.Response;
import com.azure.core.http.rest.RestProxy;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.serializer.JacksonAdapter;
import com.azure.core.util.serializer.SerializerAdapter;
import reactor.core.publisher.Mono;

/**
 * Initializes a new instance of the OpenAIClient type.
 */
public final class OpenAIClientImpl {
    /**
     * The proxy service used to perform REST calls.
     */
    private final OpenAIClientService service;

    /**
     * Supported Cognitive Services endpoints (protocol and hostname, for example:
     * https://westus.api.cognitive.microsoft.com).
     */
    private final String endpoint;

    /**
     * Gets Supported Cognitive Services endpoints (protocol and hostname, for example:
     * https://westus.api.cognitive.microsoft.com).
     * 
     * @return the endpoint value.
     */
    public String getEndpoint() {
        return this.endpoint;
    }

    /**
     * Service version.
     */
    private final OpenAIServiceVersion serviceVersion;

    /**
     * Gets Service version.
     * 
     * @return the serviceVersion value.
     */
    public OpenAIServiceVersion getServiceVersion() {
        return this.serviceVersion;
    }

    /**
     * The HTTP pipeline to send requests through.
     */
    private final HttpPipeline httpPipeline;

    /**
     * Gets The HTTP pipeline to send requests through.
     * 
     * @return the httpPipeline value.
     */
    public HttpPipeline getHttpPipeline() {
        return this.httpPipeline;
    }

    /**
     * The serializer to serialize an object into a string.
     */
    private final SerializerAdapter serializerAdapter;

    /**
     * Gets The serializer to serialize an object into a string.
     * 
     * @return the serializerAdapter value.
     */
    public SerializerAdapter getSerializerAdapter() {
        return this.serializerAdapter;
    }

    /**
     * Initializes an instance of OpenAIClient client.
     * 
     * @param endpoint Supported Cognitive Services endpoints (protocol and hostname, for example:
     * https://westus.api.cognitive.microsoft.com).
     * @param serviceVersion Service version.
     */
    public OpenAIClientImpl(String endpoint, OpenAIServiceVersion serviceVersion) {
        this(new HttpPipelineBuilder().policies(new UserAgentPolicy(), new RetryPolicy()).build(),
            JacksonAdapter.createDefaultSerializerAdapter(), endpoint, serviceVersion);
    }

    /**
     * Initializes an instance of OpenAIClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param endpoint Supported Cognitive Services endpoints (protocol and hostname, for example:
     * https://westus.api.cognitive.microsoft.com).
     * @param serviceVersion Service version.
     */
    public OpenAIClientImpl(HttpPipeline httpPipeline, String endpoint, OpenAIServiceVersion serviceVersion) {
        this(httpPipeline, JacksonAdapter.createDefaultSerializerAdapter(), endpoint, serviceVersion);
    }

    /**
     * Initializes an instance of OpenAIClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param serializerAdapter The serializer to serialize an object into a string.
     * @param endpoint Supported Cognitive Services endpoints (protocol and hostname, for example:
     * https://westus.api.cognitive.microsoft.com).
     * @param serviceVersion Service version.
     */
    public OpenAIClientImpl(HttpPipeline httpPipeline, SerializerAdapter serializerAdapter, String endpoint,
        OpenAIServiceVersion serviceVersion) {
        this.httpPipeline = httpPipeline;
        this.serializerAdapter = serializerAdapter;
        this.endpoint = endpoint;
        this.serviceVersion = serviceVersion;
        this.service = RestProxy.create(OpenAIClientService.class, this.httpPipeline, this.getSerializerAdapter());
    }

    /**
     * The interface defining all the services for OpenAIClient to be used by the proxy service to perform REST calls.
     */
    @Host("{endpoint}/openai")
    @ServiceInterface(name = "OpenAIClient")
    public interface OpenAIClientService {
        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/transcriptions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getAudioTranscriptionAsPlainText(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranscriptionOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/transcriptions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getAudioTranscriptionAsPlainTextSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranscriptionOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/transcriptions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getAudioTranscriptionAsResponseObject(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranscriptionOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/transcriptions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getAudioTranscriptionAsResponseObjectSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranscriptionOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/translations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getAudioTranslationAsPlainText(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranslationOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/translations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getAudioTranslationAsPlainTextSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranslationOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/translations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getAudioTranslationAsResponseObject(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranslationOptions, RequestOptions requestOptions,
            Context context);

        // @Multipart not supported by RestProxy
        @Post("/deployments/{deploymentId}/audio/translations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getAudioTranslationAsResponseObjectSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("content-type") String contentType, @HeaderParam("accept") String accept,
            @BodyParam("multipart/form-data") BinaryData audioTranslationOptions, RequestOptions requestOptions,
            Context context);

        @Post("/deployments/{deploymentId}/completions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getCompletions(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData completionsOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/completions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getCompletionsSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData completionsOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/chat/completions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getChatCompletions(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData chatCompletionsOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/chat/completions")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getChatCompletionsSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData chatCompletionsOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/images/generations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getImageGenerations(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData imageGenerationOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/images/generations")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getImageGenerationsSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData imageGenerationOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/audio/speech")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> generateSpeechFromText(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData speechGenerationOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/audio/speech")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> generateSpeechFromTextSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData speechGenerationOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/embeddings")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getEmbeddings(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData embeddingsOptions,
            RequestOptions requestOptions, Context context);

        @Post("/deployments/{deploymentId}/embeddings")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getEmbeddingsSync(@HostParam("endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @PathParam("deploymentId") String deploymentOrModelName,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData embeddingsOptions,
            RequestOptions requestOptions, Context context);
    }

    /**
     * Gets transcribed text and associated metadata from provided spoken audio data. Audio will be transcribed in the
     * written language corresponding to the language it was spoken in.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * String
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranscriptionOptions The configuration information for an audio transcription request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return transcribed text and associated metadata from provided spoken audio data along with {@link Response} on
     * successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getAudioTranscriptionAsPlainTextWithResponseAsync(String deploymentOrModelName,
        BinaryData audioTranscriptionOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "text/plain, application/json";
        return FluxUtil.withContext(context -> service.getAudioTranscriptionAsPlainText(this.getEndpoint(),
            this.getServiceVersion().getVersion(), deploymentOrModelName, contentType, accept,
            audioTranscriptionOptions, requestOptions, context));
    }

    /**
     * Gets transcribed text and associated metadata from provided spoken audio data. Audio will be transcribed in the
     * written language corresponding to the language it was spoken in.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * String
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranscriptionOptions The configuration information for an audio transcription request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return transcribed text and associated metadata from provided spoken audio data along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getAudioTranscriptionAsPlainTextWithResponse(String deploymentOrModelName,
        BinaryData audioTranscriptionOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "text/plain, application/json";
        return service.getAudioTranscriptionAsPlainTextSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, contentType, accept, audioTranscriptionOptions, requestOptions, Context.NONE);
    }

    /**
     * Gets transcribed text and associated metadata from provided spoken audio data. Audio will be transcribed in the
     * written language corresponding to the language it was spoken in.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     text: String (Required)
     *     task: String(transcribe/translate) (Optional)
     *     language: String (Optional)
     *     duration: Double (Optional)
     *     segments (Optional): [
     *          (Optional){
     *             id: int (Required)
     *             start: double (Required)
     *             end: double (Required)
     *             text: String (Required)
     *             temperature: double (Required)
     *             avg_logprob: double (Required)
     *             compression_ratio: double (Required)
     *             no_speech_prob: double (Required)
     *             tokens (Required): [
     *                 int (Required)
     *             ]
     *             seek: int (Required)
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranscriptionOptions The configuration information for an audio transcription request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return transcribed text and associated metadata from provided spoken audio data along with {@link Response} on
     * successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getAudioTranscriptionAsResponseObjectWithResponseAsync(
        String deploymentOrModelName, BinaryData audioTranscriptionOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.getAudioTranscriptionAsResponseObject(this.getEndpoint(),
            this.getServiceVersion().getVersion(), deploymentOrModelName, contentType, accept,
            audioTranscriptionOptions, requestOptions, context));
    }

    /**
     * Gets transcribed text and associated metadata from provided spoken audio data. Audio will be transcribed in the
     * written language corresponding to the language it was spoken in.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     text: String (Required)
     *     task: String(transcribe/translate) (Optional)
     *     language: String (Optional)
     *     duration: Double (Optional)
     *     segments (Optional): [
     *          (Optional){
     *             id: int (Required)
     *             start: double (Required)
     *             end: double (Required)
     *             text: String (Required)
     *             temperature: double (Required)
     *             avg_logprob: double (Required)
     *             compression_ratio: double (Required)
     *             no_speech_prob: double (Required)
     *             tokens (Required): [
     *                 int (Required)
     *             ]
     *             seek: int (Required)
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranscriptionOptions The configuration information for an audio transcription request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return transcribed text and associated metadata from provided spoken audio data along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getAudioTranscriptionAsResponseObjectWithResponse(String deploymentOrModelName,
        BinaryData audioTranscriptionOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "application/json";
        return service.getAudioTranscriptionAsResponseObjectSync(this.getEndpoint(),
            this.getServiceVersion().getVersion(), deploymentOrModelName, contentType, accept,
            audioTranscriptionOptions, requestOptions, Context.NONE);
    }

    /**
     * Gets English language transcribed text and associated metadata from provided spoken audio data.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * String
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranslationOptions The configuration information for an audio translation request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return english language transcribed text and associated metadata from provided spoken audio data along with
     * {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getAudioTranslationAsPlainTextWithResponseAsync(String deploymentOrModelName,
        BinaryData audioTranslationOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "text/plain, application/json";
        return FluxUtil.withContext(
            context -> service.getAudioTranslationAsPlainText(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, contentType, accept, audioTranslationOptions, requestOptions, context));
    }

    /**
     * Gets English language transcribed text and associated metadata from provided spoken audio data.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * String
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranslationOptions The configuration information for an audio translation request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return english language transcribed text and associated metadata from provided spoken audio data along with
     * {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getAudioTranslationAsPlainTextWithResponse(String deploymentOrModelName,
        BinaryData audioTranslationOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "text/plain, application/json";
        return service.getAudioTranslationAsPlainTextSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, contentType, accept, audioTranslationOptions, requestOptions, Context.NONE);
    }

    /**
     * Gets English language transcribed text and associated metadata from provided spoken audio data.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     text: String (Required)
     *     task: String(transcribe/translate) (Optional)
     *     language: String (Optional)
     *     duration: Double (Optional)
     *     segments (Optional): [
     *          (Optional){
     *             id: int (Required)
     *             start: double (Required)
     *             end: double (Required)
     *             text: String (Required)
     *             temperature: double (Required)
     *             avg_logprob: double (Required)
     *             compression_ratio: double (Required)
     *             no_speech_prob: double (Required)
     *             tokens (Required): [
     *                 int (Required)
     *             ]
     *             seek: int (Required)
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranslationOptions The configuration information for an audio translation request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return english language transcribed text and associated metadata from provided spoken audio data along with
     * {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getAudioTranslationAsResponseObjectWithResponseAsync(String deploymentOrModelName,
        BinaryData audioTranslationOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.getAudioTranslationAsResponseObject(this.getEndpoint(),
            this.getServiceVersion().getVersion(), deploymentOrModelName, contentType, accept, audioTranslationOptions,
            requestOptions, context));
    }

    /**
     * Gets English language transcribed text and associated metadata from provided spoken audio data.
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     text: String (Required)
     *     task: String(transcribe/translate) (Optional)
     *     language: String (Optional)
     *     duration: Double (Optional)
     *     segments (Optional): [
     *          (Optional){
     *             id: int (Required)
     *             start: double (Required)
     *             end: double (Required)
     *             text: String (Required)
     *             temperature: double (Required)
     *             avg_logprob: double (Required)
     *             compression_ratio: double (Required)
     *             no_speech_prob: double (Required)
     *             tokens (Required): [
     *                 int (Required)
     *             ]
     *             seek: int (Required)
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param audioTranslationOptions The configuration information for an audio translation request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return english language transcribed text and associated metadata from provided spoken audio data along with
     * {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getAudioTranslationAsResponseObjectWithResponse(String deploymentOrModelName,
        BinaryData audioTranslationOptions, RequestOptions requestOptions) {
        final String contentType = "multipart/form-data";
        final String accept = "application/json";
        return service.getAudioTranslationAsResponseObjectSync(this.getEndpoint(),
            this.getServiceVersion().getVersion(), deploymentOrModelName, contentType, accept, audioTranslationOptions,
            requestOptions, Context.NONE);
    }

    /**
     * Gets completions for the provided input prompts.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     prompt (Required): [
     *         String (Required)
     *     ]
     *     max_tokens: Integer (Optional)
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     logit_bias (Optional): {
     *         String: int (Required)
     *     }
     *     user: String (Optional)
     *     n: Integer (Optional)
     *     logprobs: Integer (Optional)
     *     suffix: String (Optional)
     *     echo: Boolean (Optional)
     *     stop (Optional): [
     *         String (Optional)
     *     ]
     *     presence_penalty: Double (Optional)
     *     frequency_penalty: Double (Optional)
     *     best_of: Integer (Optional)
     *     stream: Boolean (Optional)
     *     model: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     id: String (Required)
     *     created: long (Required)
     *     prompt_filter_results (Optional): [
     *          (Optional){
     *             prompt_index: int (Required)
     *             content_filter_results (Required): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 custom_blocklists (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                         filtered: boolean (Required)
     *                     }
     *                 ]
     *                 error (Optional): {
     *                     code: String (Required)
     *                     message: String (Required)
     *                     target: String (Optional)
     *                     details (Optional): [
     *                         (recursive schema, see above)
     *                     ]
     *                     innererror (Optional): {
     *                         code: String (Optional)
     *                         innererror (Optional): (recursive schema, see innererror above)
     *                     }
     *                 }
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     *     choices (Required): [
     *          (Required){
     *             text: String (Required)
     *             index: int (Required)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): (recursive schema, see profanity above)
     *                 custom_blocklists (Optional): [
     *                     (recursive schema, see above)
     *                 ]
     *                 error (Optional): (recursive schema, see error above)
     *                 protected_material_text (Optional): (recursive schema, see protected_material_text above)
     *                 protected_material_code (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                     URL: String (Optional)
     *                     license: String (Required)
     *                 }
     *             }
     *             logprobs (Required): {
     *                 tokens (Required): [
     *                     String (Required)
     *                 ]
     *                 token_logprobs (Required): [
     *                     double (Required)
     *                 ]
     *                 top_logprobs (Required): [
     *                      (Required){
     *                         String: Double (Optional)
     *                     }
     *                 ]
     *                 text_offset (Required): [
     *                     int (Required)
     *                 ]
     *             }
     *             finish_reason: String(stop/length/content_filter/function_call/tool_calls) (Required)
     *         }
     *     ]
     *     usage (Required): {
     *         completion_tokens: int (Required)
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param completionsOptions The configuration information for a completions request.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return completions for the provided input prompts.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getCompletionsWithResponseAsync(String deploymentOrModelName,
        BinaryData completionsOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getCompletions(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, accept, completionsOptions, requestOptions, context));
    }

    /**
     * Gets completions for the provided input prompts.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     prompt (Required): [
     *         String (Required)
     *     ]
     *     max_tokens: Integer (Optional)
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     logit_bias (Optional): {
     *         String: int (Required)
     *     }
     *     user: String (Optional)
     *     n: Integer (Optional)
     *     logprobs: Integer (Optional)
     *     suffix: String (Optional)
     *     echo: Boolean (Optional)
     *     stop (Optional): [
     *         String (Optional)
     *     ]
     *     presence_penalty: Double (Optional)
     *     frequency_penalty: Double (Optional)
     *     best_of: Integer (Optional)
     *     stream: Boolean (Optional)
     *     model: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     id: String (Required)
     *     created: long (Required)
     *     prompt_filter_results (Optional): [
     *          (Optional){
     *             prompt_index: int (Required)
     *             content_filter_results (Required): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 custom_blocklists (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                         filtered: boolean (Required)
     *                     }
     *                 ]
     *                 error (Optional): {
     *                     code: String (Required)
     *                     message: String (Required)
     *                     target: String (Optional)
     *                     details (Optional): [
     *                         (recursive schema, see above)
     *                     ]
     *                     innererror (Optional): {
     *                         code: String (Optional)
     *                         innererror (Optional): (recursive schema, see innererror above)
     *                     }
     *                 }
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     *     choices (Required): [
     *          (Required){
     *             text: String (Required)
     *             index: int (Required)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): (recursive schema, see profanity above)
     *                 custom_blocklists (Optional): [
     *                     (recursive schema, see above)
     *                 ]
     *                 error (Optional): (recursive schema, see error above)
     *                 protected_material_text (Optional): (recursive schema, see protected_material_text above)
     *                 protected_material_code (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                     URL: String (Optional)
     *                     license: String (Required)
     *                 }
     *             }
     *             logprobs (Required): {
     *                 tokens (Required): [
     *                     String (Required)
     *                 ]
     *                 token_logprobs (Required): [
     *                     double (Required)
     *                 ]
     *                 top_logprobs (Required): [
     *                      (Required){
     *                         String: Double (Optional)
     *                     }
     *                 ]
     *                 text_offset (Required): [
     *                     int (Required)
     *                 ]
     *             }
     *             finish_reason: String(stop/length/content_filter/function_call/tool_calls) (Required)
     *         }
     *     ]
     *     usage (Required): {
     *         completion_tokens: int (Required)
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param completionsOptions The configuration information for a completions request.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return completions for the provided input prompts.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getCompletionsWithResponse(String deploymentOrModelName, BinaryData completionsOptions,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getCompletionsSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, accept, completionsOptions, requestOptions, Context.NONE);
    }

    /**
     * Gets chat completions for the provided chat messages.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     messages (Required): [
     *          (Required){
     *         }
     *     ]
     *     functions (Optional): [
     *          (Optional){
     *             name: String (Required)
     *             description: String (Optional)
     *             parameters: Object (Optional)
     *         }
     *     ]
     *     function_call: BinaryData (Optional)
     *     max_tokens: Integer (Optional)
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     logit_bias (Optional): {
     *         String: int (Required)
     *     }
     *     user: String (Optional)
     *     n: Integer (Optional)
     *     stop (Optional): [
     *         String (Optional)
     *     ]
     *     presence_penalty: Double (Optional)
     *     frequency_penalty: Double (Optional)
     *     stream: Boolean (Optional)
     *     model: String (Optional)
     *     data_sources (Optional): [
     *          (Optional){
     *         }
     *     ]
     *     enhancements (Optional): {
     *         grounding (Optional): {
     *             enabled: boolean (Required)
     *         }
     *         ocr (Optional): {
     *             enabled: boolean (Required)
     *         }
     *     }
     *     seed: Long (Optional)
     *     logprobs: Boolean (Optional)
     *     top_logprobs: Integer (Optional)
     *     response_format (Optional): {
     *     }
     *     tools (Optional): [
     *          (Optional){
     *         }
     *     ]
     *     tool_choice: BinaryData (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     id: String (Required)
     *     created: long (Required)
     *     choices (Required): [
     *          (Required){
     *             message (Optional): {
     *                 role: String(system/assistant/user/function/tool) (Required)
     *                 content: String (Required)
     *                 tool_calls (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                     }
     *                 ]
     *                 function_call (Optional): {
     *                     name: String (Required)
     *                     arguments: String (Required)
     *                 }
     *                 context (Optional): {
     *                     citations (Optional): [
     *                          (Optional){
     *                             content: String (Required)
     *                             title: String (Optional)
     *                             url: String (Optional)
     *                             filepath: String (Optional)
     *                             chunk_id: String (Optional)
     *                         }
     *                     ]
     *                     intent: String (Optional)
     *                 }
     *             }
     *             logprobs (Required): {
     *                 content (Required): [
     *                      (Required){
     *                         token: String (Required)
     *                         logprob: double (Required)
     *                         bytes (Required): [
     *                             int (Required)
     *                         ]
     *                         top_logprobs (Required): [
     *                              (Required){
     *                                 token: String (Required)
     *                                 logprob: double (Required)
     *                                 bytes (Required): [
     *                                     int (Required)
     *                                 ]
     *                             }
     *                         ]
     *                     }
     *                 ]
     *             }
     *             index: int (Required)
     *             finish_reason: String(stop/length/content_filter/function_call/tool_calls) (Required)
     *             finish_details (Optional): {
     *             }
     *             delta (Optional): (recursive schema, see delta above)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 custom_blocklists (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                         filtered: boolean (Required)
     *                     }
     *                 ]
     *                 error (Optional): {
     *                     code: String (Required)
     *                     message: String (Required)
     *                     target: String (Optional)
     *                     details (Optional): [
     *                         (recursive schema, see above)
     *                     ]
     *                     innererror (Optional): {
     *                         code: String (Optional)
     *                         innererror (Optional): (recursive schema, see innererror above)
     *                     }
     *                 }
     *                 protected_material_text (Optional): (recursive schema, see protected_material_text above)
     *                 protected_material_code (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                     URL: String (Optional)
     *                     license: String (Required)
     *                 }
     *             }
     *             enhancements (Optional): {
     *                 grounding (Optional): {
     *                     lines (Required): [
     *                          (Required){
     *                             text: String (Required)
     *                             spans (Required): [
     *                                  (Required){
     *                                     text: String (Required)
     *                                     offset: int (Required)
     *                                     length: int (Required)
     *                                     polygon (Required): [
     *                                          (Required){
     *                                             x: double (Required)
     *                                             y: double (Required)
     *                                         }
     *                                     ]
     *                                 }
     *                             ]
     *                         }
     *                     ]
     *                 }
     *             }
     *         }
     *     ]
     *     prompt_filter_results (Optional): [
     *          (Optional){
     *             prompt_index: int (Required)
     *             content_filter_results (Required): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): (recursive schema, see profanity above)
     *                 custom_blocklists (Optional): [
     *                     (recursive schema, see above)
     *                 ]
     *                 error (Optional): (recursive schema, see error above)
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     *     system_fingerprint: String (Optional)
     *     usage (Required): {
     *         completion_tokens: int (Required)
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param chatCompletionsOptions The configuration information for a chat completions request.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return chat completions for the provided chat messages.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getChatCompletionsWithResponseAsync(String deploymentOrModelName,
        BinaryData chatCompletionsOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.getChatCompletions(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, accept, chatCompletionsOptions, requestOptions, context));
    }

    /**
     * Gets chat completions for the provided chat messages.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     messages (Required): [
     *          (Required){
     *         }
     *     ]
     *     functions (Optional): [
     *          (Optional){
     *             name: String (Required)
     *             description: String (Optional)
     *             parameters: Object (Optional)
     *         }
     *     ]
     *     function_call: BinaryData (Optional)
     *     max_tokens: Integer (Optional)
     *     temperature: Double (Optional)
     *     top_p: Double (Optional)
     *     logit_bias (Optional): {
     *         String: int (Required)
     *     }
     *     user: String (Optional)
     *     n: Integer (Optional)
     *     stop (Optional): [
     *         String (Optional)
     *     ]
     *     presence_penalty: Double (Optional)
     *     frequency_penalty: Double (Optional)
     *     stream: Boolean (Optional)
     *     model: String (Optional)
     *     data_sources (Optional): [
     *          (Optional){
     *         }
     *     ]
     *     enhancements (Optional): {
     *         grounding (Optional): {
     *             enabled: boolean (Required)
     *         }
     *         ocr (Optional): {
     *             enabled: boolean (Required)
     *         }
     *     }
     *     seed: Long (Optional)
     *     logprobs: Boolean (Optional)
     *     top_logprobs: Integer (Optional)
     *     response_format (Optional): {
     *     }
     *     tools (Optional): [
     *          (Optional){
     *         }
     *     ]
     *     tool_choice: BinaryData (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     id: String (Required)
     *     created: long (Required)
     *     choices (Required): [
     *          (Required){
     *             message (Optional): {
     *                 role: String(system/assistant/user/function/tool) (Required)
     *                 content: String (Required)
     *                 tool_calls (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                     }
     *                 ]
     *                 function_call (Optional): {
     *                     name: String (Required)
     *                     arguments: String (Required)
     *                 }
     *                 context (Optional): {
     *                     citations (Optional): [
     *                          (Optional){
     *                             content: String (Required)
     *                             title: String (Optional)
     *                             url: String (Optional)
     *                             filepath: String (Optional)
     *                             chunk_id: String (Optional)
     *                         }
     *                     ]
     *                     intent: String (Optional)
     *                 }
     *             }
     *             logprobs (Required): {
     *                 content (Required): [
     *                      (Required){
     *                         token: String (Required)
     *                         logprob: double (Required)
     *                         bytes (Required): [
     *                             int (Required)
     *                         ]
     *                         top_logprobs (Required): [
     *                              (Required){
     *                                 token: String (Required)
     *                                 logprob: double (Required)
     *                                 bytes (Required): [
     *                                     int (Required)
     *                                 ]
     *                             }
     *                         ]
     *                     }
     *                 ]
     *             }
     *             index: int (Required)
     *             finish_reason: String(stop/length/content_filter/function_call/tool_calls) (Required)
     *             finish_details (Optional): {
     *             }
     *             delta (Optional): (recursive schema, see delta above)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 custom_blocklists (Optional): [
     *                      (Optional){
     *                         id: String (Required)
     *                         filtered: boolean (Required)
     *                     }
     *                 ]
     *                 error (Optional): {
     *                     code: String (Required)
     *                     message: String (Required)
     *                     target: String (Optional)
     *                     details (Optional): [
     *                         (recursive schema, see above)
     *                     ]
     *                     innererror (Optional): {
     *                         code: String (Optional)
     *                         innererror (Optional): (recursive schema, see innererror above)
     *                     }
     *                 }
     *                 protected_material_text (Optional): (recursive schema, see protected_material_text above)
     *                 protected_material_code (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                     URL: String (Optional)
     *                     license: String (Required)
     *                 }
     *             }
     *             enhancements (Optional): {
     *                 grounding (Optional): {
     *                     lines (Required): [
     *                          (Required){
     *                             text: String (Required)
     *                             spans (Required): [
     *                                  (Required){
     *                                     text: String (Required)
     *                                     offset: int (Required)
     *                                     length: int (Required)
     *                                     polygon (Required): [
     *                                          (Required){
     *                                             x: double (Required)
     *                                             y: double (Required)
     *                                         }
     *                                     ]
     *                                 }
     *                             ]
     *                         }
     *                     ]
     *                 }
     *             }
     *         }
     *     ]
     *     prompt_filter_results (Optional): [
     *          (Optional){
     *             prompt_index: int (Required)
     *             content_filter_results (Required): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): (recursive schema, see profanity above)
     *                 custom_blocklists (Optional): [
     *                     (recursive schema, see above)
     *                 ]
     *                 error (Optional): (recursive schema, see error above)
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     *     system_fingerprint: String (Optional)
     *     usage (Required): {
     *         completion_tokens: int (Required)
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param chatCompletionsOptions The configuration information for a chat completions request.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return chat completions for the provided chat messages.
     * Completions support a wide variety of tasks and generate text that continues from or "completes"
     * provided prompt data along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getChatCompletionsWithResponse(String deploymentOrModelName,
        BinaryData chatCompletionsOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getChatCompletionsSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, accept, chatCompletionsOptions, requestOptions, Context.NONE);
    }

    /**
     * Creates an image given a prompt.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     model: String (Optional)
     *     prompt: String (Required)
     *     n: Integer (Optional)
     *     size: String(256x256/512x512/1024x1024/1792x1024/1024x1792) (Optional)
     *     response_format: String(url/b64_json) (Optional)
     *     quality: String(standard/hd) (Optional)
     *     style: String(natural/vivid) (Optional)
     *     user: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     created: long (Required)
     *     data (Required): [
     *          (Required){
     *             url: String (Optional)
     *             b64_json: String (Optional)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *             }
     *             revised_prompt: String (Optional)
     *             prompt_filter_results (Optional): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param imageGenerationOptions Represents the request data used to generate images.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the result of a successful image generation operation along with {@link Response} on successful
     * completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getImageGenerationsWithResponseAsync(String deploymentOrModelName,
        BinaryData imageGenerationOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(
            context -> service.getImageGenerations(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, accept, imageGenerationOptions, requestOptions, context));
    }

    /**
     * Creates an image given a prompt.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     model: String (Optional)
     *     prompt: String (Required)
     *     n: Integer (Optional)
     *     size: String(256x256/512x512/1024x1024/1792x1024/1024x1792) (Optional)
     *     response_format: String(url/b64_json) (Optional)
     *     quality: String(standard/hd) (Optional)
     *     style: String(natural/vivid) (Optional)
     *     user: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     created: long (Required)
     *     data (Required): [
     *          (Required){
     *             url: String (Optional)
     *             b64_json: String (Optional)
     *             content_filter_results (Optional): {
     *                 sexual (Optional): {
     *                     severity: String(safe/low/medium/high) (Required)
     *                     filtered: boolean (Required)
     *                 }
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *             }
     *             revised_prompt: String (Optional)
     *             prompt_filter_results (Optional): {
     *                 sexual (Optional): (recursive schema, see sexual above)
     *                 violence (Optional): (recursive schema, see violence above)
     *                 hate (Optional): (recursive schema, see hate above)
     *                 self_harm (Optional): (recursive schema, see self_harm above)
     *                 profanity (Optional): {
     *                     filtered: boolean (Required)
     *                     detected: boolean (Required)
     *                 }
     *                 jailbreak (Optional): (recursive schema, see jailbreak above)
     *             }
     *         }
     *     ]
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param imageGenerationOptions Represents the request data used to generate images.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the result of a successful image generation operation along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getImageGenerationsWithResponse(String deploymentOrModelName,
        BinaryData imageGenerationOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getImageGenerationsSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, accept, imageGenerationOptions, requestOptions, Context.NONE);
    }

    /**
     * Generates text-to-speech audio from the input text.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     input: String (Required)
     *     voice: String(alloy/echo/fable/onyx/nova/shimmer) (Required)
     *     response_format: String(mp3/opus/aac/flac) (Optional)
     *     speed: Double (Optional)
     *     model: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * BinaryData
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param speechGenerationOptions A representation of the request options that control the behavior of a
     * text-to-speech operation.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> generateSpeechFromTextWithResponseAsync(String deploymentOrModelName,
        BinaryData speechGenerationOptions, RequestOptions requestOptions) {
        final String accept = "application/octet-stream, application/json";
        return FluxUtil.withContext(
            context -> service.generateSpeechFromText(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, accept, speechGenerationOptions, requestOptions, context));
    }

    /**
     * Generates text-to-speech audio from the input text.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     input: String (Required)
     *     voice: String(alloy/echo/fable/onyx/nova/shimmer) (Required)
     *     response_format: String(mp3/opus/aac/flac) (Optional)
     *     speed: Double (Optional)
     *     model: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * BinaryData
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param speechGenerationOptions A representation of the request options that control the behavior of a
     * text-to-speech operation.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> generateSpeechFromTextWithResponse(String deploymentOrModelName,
        BinaryData speechGenerationOptions, RequestOptions requestOptions) {
        final String accept = "application/octet-stream, application/json";
        return service.generateSpeechFromTextSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, accept, speechGenerationOptions, requestOptions, Context.NONE);
    }

    /**
     * Return the embeddings for a given prompt.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     user: String (Optional)
     *     model: String (Optional)
     *     input (Required): [
     *         String (Required)
     *     ]
     *     input_type: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     data (Required): [
     *          (Required){
     *             embedding (Required): [
     *                 double (Required)
     *             ]
     *             index: int (Required)
     *         }
     *     ]
     *     usage (Required): {
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param embeddingsOptions The configuration information for an embeddings request.
     * Embeddings measure the relatedness of text strings and are commonly used for search, clustering,
     * recommendations, and other similar scenarios.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return representation of the response data from an embeddings request.
     * Embeddings measure the relatedness of text strings and are commonly used for search, clustering,
     * recommendations, and other similar scenarios along with {@link Response} on successful completion of
     * {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getEmbeddingsWithResponseAsync(String deploymentOrModelName,
        BinaryData embeddingsOptions, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil
            .withContext(context -> service.getEmbeddings(this.getEndpoint(), this.getServiceVersion().getVersion(),
                deploymentOrModelName, accept, embeddingsOptions, requestOptions, context));
    }

    /**
     * Return the embeddings for a given prompt.
     * <p>
     * <strong>Request Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     user: String (Optional)
     *     model: String (Optional)
     *     input (Required): [
     *         String (Required)
     *     ]
     *     input_type: String (Optional)
     * }
     * }</pre>
     * <p>
     * <strong>Response Body Schema</strong>
     * </p>
     * <pre>{@code
     * {
     *     data (Required): [
     *          (Required){
     *             embedding (Required): [
     *                 double (Required)
     *             ]
     *             index: int (Required)
     *         }
     *     ]
     *     usage (Required): {
     *         prompt_tokens: int (Required)
     *         total_tokens: int (Required)
     *     }
     * }
     * }</pre>
     * 
     * @param deploymentOrModelName Specifies either the model deployment name (when using Azure OpenAI) or model name
     * (when using non-Azure OpenAI) to use for this request.
     * @param embeddingsOptions The configuration information for an embeddings request.
     * Embeddings measure the relatedness of text strings and are commonly used for search, clustering,
     * recommendations, and other similar scenarios.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return representation of the response data from an embeddings request.
     * Embeddings measure the relatedness of text strings and are commonly used for search, clustering,
     * recommendations, and other similar scenarios along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getEmbeddingsWithResponse(String deploymentOrModelName, BinaryData embeddingsOptions,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getEmbeddingsSync(this.getEndpoint(), this.getServiceVersion().getVersion(),
            deploymentOrModelName, accept, embeddingsOptions, requestOptions, Context.NONE);
    }
}
