// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
// Code generated by Microsoft (R) TypeSpec Code Generator.

package com.azure.ai.translation.text.implementation;

import com.azure.ai.translation.text.TextTranslationServiceVersion;
import com.azure.core.annotation.BodyParam;
import com.azure.core.annotation.ExpectedResponses;
import com.azure.core.annotation.Get;
import com.azure.core.annotation.HeaderParam;
import com.azure.core.annotation.Host;
import com.azure.core.annotation.HostParam;
import com.azure.core.annotation.Post;
import com.azure.core.annotation.QueryParam;
import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceInterface;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.annotation.UnexpectedResponseExceptionType;
import com.azure.core.exception.ClientAuthenticationException;
import com.azure.core.exception.HttpResponseException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.HttpPipelineBuilder;
import com.azure.core.http.policy.RetryPolicy;
import com.azure.core.http.policy.UserAgentPolicy;
import com.azure.core.http.rest.RequestOptions;
import com.azure.core.http.rest.Response;
import com.azure.core.http.rest.RestProxy;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.serializer.JacksonAdapter;
import com.azure.core.util.serializer.SerializerAdapter;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import reactor.core.publisher.Mono;

/**
 * Initializes a new instance of the TextTranslationClient type.
 */
public final class TextTranslationClientImpl {
    /**
     * The proxy service used to perform REST calls.
     */
    private final TextTranslationClientService service;

    /**
     * Supported Text Translation endpoints (protocol and hostname, for example:
     * https://api.cognitive.microsofttranslator.com).
     */
    private final String endpoint;

    /**
     * Gets Supported Text Translation endpoints (protocol and hostname, for example:
     * https://api.cognitive.microsofttranslator.com).
     * 
     * @return the endpoint value.
     */
    public String getEndpoint() {
        return this.endpoint;
    }

    /**
     * Service version.
     */
    private final TextTranslationServiceVersion serviceVersion;

    /**
     * Gets Service version.
     * 
     * @return the serviceVersion value.
     */
    public TextTranslationServiceVersion getServiceVersion() {
        return this.serviceVersion;
    }

    /**
     * The HTTP pipeline to send requests through.
     */
    private final HttpPipeline httpPipeline;

    /**
     * Gets The HTTP pipeline to send requests through.
     * 
     * @return the httpPipeline value.
     */
    public HttpPipeline getHttpPipeline() {
        return this.httpPipeline;
    }

    /**
     * The serializer to serialize an object into a string.
     */
    private final SerializerAdapter serializerAdapter;

    /**
     * Gets The serializer to serialize an object into a string.
     * 
     * @return the serializerAdapter value.
     */
    public SerializerAdapter getSerializerAdapter() {
        return this.serializerAdapter;
    }

    /**
     * Initializes an instance of TextTranslationClient client.
     * 
     * @param endpoint Supported Text Translation endpoints (protocol and hostname, for example:
     * https://api.cognitive.microsofttranslator.com).
     * @param serviceVersion Service version.
     */
    public TextTranslationClientImpl(String endpoint, TextTranslationServiceVersion serviceVersion) {
        this(new HttpPipelineBuilder().policies(new UserAgentPolicy(), new RetryPolicy()).build(),
            JacksonAdapter.createDefaultSerializerAdapter(), endpoint, serviceVersion);
    }

    /**
     * Initializes an instance of TextTranslationClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param endpoint Supported Text Translation endpoints (protocol and hostname, for example:
     * https://api.cognitive.microsofttranslator.com).
     * @param serviceVersion Service version.
     */
    public TextTranslationClientImpl(HttpPipeline httpPipeline, String endpoint,
        TextTranslationServiceVersion serviceVersion) {
        this(httpPipeline, JacksonAdapter.createDefaultSerializerAdapter(), endpoint, serviceVersion);
    }

    /**
     * Initializes an instance of TextTranslationClient client.
     * 
     * @param httpPipeline The HTTP pipeline to send requests through.
     * @param serializerAdapter The serializer to serialize an object into a string.
     * @param endpoint Supported Text Translation endpoints (protocol and hostname, for example:
     * https://api.cognitive.microsofttranslator.com).
     * @param serviceVersion Service version.
     */
    public TextTranslationClientImpl(HttpPipeline httpPipeline, SerializerAdapter serializerAdapter, String endpoint,
        TextTranslationServiceVersion serviceVersion) {
        this.httpPipeline = httpPipeline;
        this.serializerAdapter = serializerAdapter;
        this.endpoint = endpoint;
        this.serviceVersion = serviceVersion;
        this.service
            = RestProxy.create(TextTranslationClientService.class, this.httpPipeline, this.getSerializerAdapter());
    }

    /**
     * The interface defining all the services for TextTranslationClient to be used by the proxy service to perform REST
     * calls.
     */
    @Host("{Endpoint}")
    @ServiceInterface(name = "TextTranslationClien")
    public interface TextTranslationClientService {
        @Get("/languages")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> getSupportedLanguages(@HostParam("Endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            RequestOptions requestOptions, Context context);

        @Get("/languages")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> getSupportedLanguagesSync(@HostParam("Endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            RequestOptions requestOptions, Context context);

        @Post("/translate")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> translate(@HostParam("Endpoint") String endpoint,
            @QueryParam(value = "to", multipleQueryParams = true) List<String> targetLanguages,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/translate")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> translateSync(@HostParam("Endpoint") String endpoint,
            @QueryParam(value = "to", multipleQueryParams = true) List<String> targetLanguages,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/transliterate")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> transliterate(@HostParam("Endpoint") String endpoint,
            @QueryParam("language") String language, @QueryParam("fromScript") String sourceLanguageScript,
            @QueryParam("toScript") String targetLanguageScript, @QueryParam("api-version") String apiVersion,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData body,
            RequestOptions requestOptions, Context context);

        @Post("/transliterate")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> transliterateSync(@HostParam("Endpoint") String endpoint,
            @QueryParam("language") String language, @QueryParam("fromScript") String sourceLanguageScript,
            @QueryParam("toScript") String targetLanguageScript, @QueryParam("api-version") String apiVersion,
            @HeaderParam("accept") String accept, @BodyParam("application/json") BinaryData body,
            RequestOptions requestOptions, Context context);

        @Post("/breaksentence")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> findSentenceBoundaries(@HostParam("Endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/breaksentence")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> findSentenceBoundariesSync(@HostParam("Endpoint") String endpoint,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/dictionary/lookup")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> lookupDictionaryEntries(@HostParam("Endpoint") String endpoint,
            @QueryParam("from") String sourceLanguage, @QueryParam("to") String targetLanguage,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/dictionary/lookup")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> lookupDictionaryEntriesSync(@HostParam("Endpoint") String endpoint,
            @QueryParam("from") String sourceLanguage, @QueryParam("to") String targetLanguage,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/dictionary/examples")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Mono<Response<BinaryData>> lookupDictionaryExamples(@HostParam("Endpoint") String endpoint,
            @QueryParam("from") String sourceLanguage, @QueryParam("to") String targetLanguage,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);

        @Post("/dictionary/examples")
        @ExpectedResponses({ 200 })
        @UnexpectedResponseExceptionType(value = ClientAuthenticationException.class, code = { 401 })
        @UnexpectedResponseExceptionType(value = ResourceNotFoundException.class, code = { 404 })
        @UnexpectedResponseExceptionType(value = ResourceModifiedException.class, code = { 409 })
        @UnexpectedResponseExceptionType(HttpResponseException.class)
        Response<BinaryData> lookupDictionaryExamplesSync(@HostParam("Endpoint") String endpoint,
            @QueryParam("from") String sourceLanguage, @QueryParam("to") String targetLanguage,
            @QueryParam("api-version") String apiVersion, @HeaderParam("accept") String accept,
            @BodyParam("application/json") BinaryData body, RequestOptions requestOptions, Context context);
    }

    /**
     * Gets the set of languages currently supported by other operations of the Translator.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>scope</td><td>String</td><td>No</td><td>A comma-separated list of names defining the group of languages
     * to return.
     * Allowed group names are: `translation`, `transliteration` and `dictionary`.
     * If no scope is given, then all groups are returned, which is equivalent to passing
     * `scope=translation,transliteration,dictionary`. To decide which set of supported languages
     * is appropriate for your scenario, see the description of the [response object](#response-body).</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * <tr><td>Accept-Language</td><td>String</td><td>No</td><td>The language to use for user interface strings. Some of
     * the fields in the response are names of languages or
     * names of regions. Use this parameter to define the language in which these names are returned.
     * The language is specified by providing a well-formed BCP 47 language tag. For instance, use the value `fr`
     * to request names in French or use the value `zh-Hant` to request names in Chinese Traditional.
     * Names are provided in the English language when a target language is not specified or when localization
     * is not available.</td></tr>
     * <tr><td>If-None-Match</td><td>String</td><td>No</td><td>Passing the value of the ETag response header in an
     * If-None-Match field will allow the service to optimize the response.
     * If the resource has not been modified, the service will return status code 304 and an empty response
     * body.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * {
     *     translation (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             dir: String(ltr/rtl) (Required)
     *         }
     *     }
     *     transliteration (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             scripts (Required): [
     *                  (Required){
     *                     code: String (Required)
     *                     name: String (Required)
     *                     nativeName: String (Required)
     *                     dir: String(ltr/rtl) (Required)
     *                     toScripts (Required): [
     *                          (Required){
     *                             code: String (Required)
     *                             name: String (Required)
     *                             nativeName: String (Required)
     *                             dir: String(ltr/rtl) (Required)
     *                         }
     *                     ]
     *                 }
     *             ]
     *         }
     *     }
     *     dictionary (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             dir: String(ltr/rtl) (Required)
     *             translations (Required): [
     *                  (Required){
     *                     name: String (Required)
     *                     nativeName: String (Required)
     *                     dir: String(ltr/rtl) (Required)
     *                     code: String (Required)
     *                 }
     *             ]
     *         }
     *     }
     * }
     * }</pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the set of languages currently supported by other operations of the Translator along with
     * {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> getSupportedLanguagesWithResponseAsync(RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.getSupportedLanguages(this.getEndpoint(),
            this.getServiceVersion().getVersion(), accept, requestOptions, context));
    }

    /**
     * Gets the set of languages currently supported by other operations of the Translator.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>scope</td><td>String</td><td>No</td><td>A comma-separated list of names defining the group of languages
     * to return.
     * Allowed group names are: `translation`, `transliteration` and `dictionary`.
     * If no scope is given, then all groups are returned, which is equivalent to passing
     * `scope=translation,transliteration,dictionary`. To decide which set of supported languages
     * is appropriate for your scenario, see the description of the [response object](#response-body).</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * <tr><td>Accept-Language</td><td>String</td><td>No</td><td>The language to use for user interface strings. Some of
     * the fields in the response are names of languages or
     * names of regions. Use this parameter to define the language in which these names are returned.
     * The language is specified by providing a well-formed BCP 47 language tag. For instance, use the value `fr`
     * to request names in French or use the value `zh-Hant` to request names in Chinese Traditional.
     * Names are provided in the English language when a target language is not specified or when localization
     * is not available.</td></tr>
     * <tr><td>If-None-Match</td><td>String</td><td>No</td><td>Passing the value of the ETag response header in an
     * If-None-Match field will allow the service to optimize the response.
     * If the resource has not been modified, the service will return status code 304 and an empty response
     * body.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * {
     *     translation (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             dir: String(ltr/rtl) (Required)
     *         }
     *     }
     *     transliteration (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             scripts (Required): [
     *                  (Required){
     *                     code: String (Required)
     *                     name: String (Required)
     *                     nativeName: String (Required)
     *                     dir: String(ltr/rtl) (Required)
     *                     toScripts (Required): [
     *                          (Required){
     *                             code: String (Required)
     *                             name: String (Required)
     *                             nativeName: String (Required)
     *                             dir: String(ltr/rtl) (Required)
     *                         }
     *                     ]
     *                 }
     *             ]
     *         }
     *     }
     *     dictionary (Optional): {
     *         String (Required): {
     *             name: String (Required)
     *             nativeName: String (Required)
     *             dir: String(ltr/rtl) (Required)
     *             translations (Required): [
     *                  (Required){
     *                     name: String (Required)
     *                     nativeName: String (Required)
     *                     dir: String(ltr/rtl) (Required)
     *                     code: String (Required)
     *                 }
     *             ]
     *         }
     *     }
     * }
     * }</pre>
     * 
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the set of languages currently supported by other operations of the Translator along with
     * {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> getSupportedLanguagesWithResponse(RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.getSupportedLanguagesSync(this.getEndpoint(), this.getServiceVersion().getVersion(), accept,
            requestOptions, Context.NONE);
    }

    /**
     * Translate Text.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>from</td><td>String</td><td>No</td><td>Specifies the language of the input text. Find which languages are
     * available to translate from by
     * looking up supported languages using the translation scope. If the from parameter isn't specified,
     * automatic language detection is applied to determine the source language.
     * 
     * You must use the from parameter rather than autodetection when using the dynamic dictionary feature.
     * Note: the dynamic dictionary feature is case-sensitive.</td></tr>
     * <tr><td>textType</td><td>String</td><td>No</td><td>Defines whether the text being translated is plain text or
     * HTML text. Any HTML needs to be a well-formed,
     * complete element. Possible values are: plain (default) or html. Allowed values: "Plain", "Html".</td></tr>
     * <tr><td>category</td><td>String</td><td>No</td><td>A string specifying the category (domain) of the translation.
     * This parameter is used to get translations
     * from a customized system built with Custom Translator. Add the Category ID from your Custom Translator
     * project details to this parameter to use your deployed customized system. Default value is: general.</td></tr>
     * <tr><td>profanityAction</td><td>String</td><td>No</td><td>Specifies how profanities should be treated in
     * translations.
     * Possible values are: NoAction (default), Marked or Deleted. Allowed values: "NoAction", "Marked",
     * "Deleted".</td></tr>
     * <tr><td>profanityMarker</td><td>String</td><td>No</td><td>Specifies how profanities should be marked in
     * translations.
     * Possible values are: Asterisk (default) or Tag. . Allowed values: "Asterisk", "Tag".</td></tr>
     * <tr><td>includeAlignment</td><td>Boolean</td><td>No</td><td>Specifies whether to include alignment projection
     * from source text to translated text.
     * Possible values are: true or false (default).</td></tr>
     * <tr><td>includeSentenceLength</td><td>Boolean</td><td>No</td><td>Specifies whether to include sentence boundaries
     * for the input text and the translated text.
     * Possible values are: true or false (default).</td></tr>
     * <tr><td>suggestedFrom</td><td>String</td><td>No</td><td>Specifies a fallback language if the language of the
     * input text can't be identified.
     * Language autodetection is applied when the from parameter is omitted. If detection fails,
     * the suggestedFrom language will be assumed.</td></tr>
     * <tr><td>fromScript</td><td>String</td><td>No</td><td>Specifies the script of the input text.</td></tr>
     * <tr><td>toScript</td><td>String</td><td>No</td><td>Specifies the script of the translated text.</td></tr>
     * <tr><td>allowFallback</td><td>Boolean</td><td>No</td><td>Specifies that the service is allowed to fall back to a
     * general system when a custom system doesn't exist.
     * Possible values are: true (default) or false.
     * 
     * allowFallback=false specifies that the translation should only use systems trained for the category specified
     * by the request. If a translation for language X to language Y requires chaining through a pivot language E,
     * then all the systems in the chain (X → E and E → Y) will need to be custom and have the same category.
     * If no system is found with the specific category, the request will return a 400 status code. allowFallback=true
     * specifies that the service is allowed to fall back to a general system when a custom system doesn't
     * exist.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         detectedLanguage (Optional): {
     *             language: String (Required)
     *             score: double (Required)
     *         }
     *         translations (Required): [
     *              (Required){
     *                 to: String (Required)
     *                 text: String (Required)
     *                 transliteration (Optional): {
     *                     text: String (Required)
     *                     script: String (Required)
     *                 }
     *                 alignment (Optional): {
     *                     proj: String (Required)
     *                 }
     *                 sentLen (Optional): {
     *                     srcSentLen (Required): [
     *                         int (Required)
     *                     ]
     *                     transSentLen (Required): [
     *                         int (Required)
     *                     ]
     *                 }
     *             }
     *         ]
     *         sourceText (Optional): {
     *             text: String (Required)
     *         }
     *     }
     * ]
     * }</pre>
     * 
     * @param targetLanguages Specifies the language of the output text. The target language must be one of the
     * supported languages included
     * in the translation scope. For example, use to=de to translate to German.
     * It's possible to translate to multiple languages simultaneously by repeating the parameter in the query string.
     * For example, use to=de&amp;to=it to translate to German and Italian.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> translateWithResponseAsync(List<String> targetLanguages, BinaryData body,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        List<String> targetLanguagesConverted
            = targetLanguages.stream().map(item -> Objects.toString(item, "")).collect(Collectors.toList());
        return FluxUtil.withContext(context -> service.translate(this.getEndpoint(), targetLanguagesConverted,
            this.getServiceVersion().getVersion(), accept, body, requestOptions, context));
    }

    /**
     * Translate Text.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>from</td><td>String</td><td>No</td><td>Specifies the language of the input text. Find which languages are
     * available to translate from by
     * looking up supported languages using the translation scope. If the from parameter isn't specified,
     * automatic language detection is applied to determine the source language.
     * 
     * You must use the from parameter rather than autodetection when using the dynamic dictionary feature.
     * Note: the dynamic dictionary feature is case-sensitive.</td></tr>
     * <tr><td>textType</td><td>String</td><td>No</td><td>Defines whether the text being translated is plain text or
     * HTML text. Any HTML needs to be a well-formed,
     * complete element. Possible values are: plain (default) or html. Allowed values: "Plain", "Html".</td></tr>
     * <tr><td>category</td><td>String</td><td>No</td><td>A string specifying the category (domain) of the translation.
     * This parameter is used to get translations
     * from a customized system built with Custom Translator. Add the Category ID from your Custom Translator
     * project details to this parameter to use your deployed customized system. Default value is: general.</td></tr>
     * <tr><td>profanityAction</td><td>String</td><td>No</td><td>Specifies how profanities should be treated in
     * translations.
     * Possible values are: NoAction (default), Marked or Deleted. Allowed values: "NoAction", "Marked",
     * "Deleted".</td></tr>
     * <tr><td>profanityMarker</td><td>String</td><td>No</td><td>Specifies how profanities should be marked in
     * translations.
     * Possible values are: Asterisk (default) or Tag. . Allowed values: "Asterisk", "Tag".</td></tr>
     * <tr><td>includeAlignment</td><td>Boolean</td><td>No</td><td>Specifies whether to include alignment projection
     * from source text to translated text.
     * Possible values are: true or false (default).</td></tr>
     * <tr><td>includeSentenceLength</td><td>Boolean</td><td>No</td><td>Specifies whether to include sentence boundaries
     * for the input text and the translated text.
     * Possible values are: true or false (default).</td></tr>
     * <tr><td>suggestedFrom</td><td>String</td><td>No</td><td>Specifies a fallback language if the language of the
     * input text can't be identified.
     * Language autodetection is applied when the from parameter is omitted. If detection fails,
     * the suggestedFrom language will be assumed.</td></tr>
     * <tr><td>fromScript</td><td>String</td><td>No</td><td>Specifies the script of the input text.</td></tr>
     * <tr><td>toScript</td><td>String</td><td>No</td><td>Specifies the script of the translated text.</td></tr>
     * <tr><td>allowFallback</td><td>Boolean</td><td>No</td><td>Specifies that the service is allowed to fall back to a
     * general system when a custom system doesn't exist.
     * Possible values are: true (default) or false.
     * 
     * allowFallback=false specifies that the translation should only use systems trained for the category specified
     * by the request. If a translation for language X to language Y requires chaining through a pivot language E,
     * then all the systems in the chain (X → E and E → Y) will need to be custom and have the same category.
     * If no system is found with the specific category, the request will return a 400 status code. allowFallback=true
     * specifies that the service is allowed to fall back to a general system when a custom system doesn't
     * exist.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         detectedLanguage (Optional): {
     *             language: String (Required)
     *             score: double (Required)
     *         }
     *         translations (Required): [
     *              (Required){
     *                 to: String (Required)
     *                 text: String (Required)
     *                 transliteration (Optional): {
     *                     text: String (Required)
     *                     script: String (Required)
     *                 }
     *                 alignment (Optional): {
     *                     proj: String (Required)
     *                 }
     *                 sentLen (Optional): {
     *                     srcSentLen (Required): [
     *                         int (Required)
     *                     ]
     *                     transSentLen (Required): [
     *                         int (Required)
     *                     ]
     *                 }
     *             }
     *         ]
     *         sourceText (Optional): {
     *             text: String (Required)
     *         }
     *     }
     * ]
     * }</pre>
     * 
     * @param targetLanguages Specifies the language of the output text. The target language must be one of the
     * supported languages included
     * in the translation scope. For example, use to=de to translate to German.
     * It's possible to translate to multiple languages simultaneously by repeating the parameter in the query string.
     * For example, use to=de&amp;to=it to translate to German and Italian.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> translateWithResponse(List<String> targetLanguages, BinaryData body,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        List<String> targetLanguagesConverted
            = targetLanguages.stream().map(item -> Objects.toString(item, "")).collect(Collectors.toList());
        return service.translateSync(this.getEndpoint(), targetLanguagesConverted,
            this.getServiceVersion().getVersion(), accept, body, requestOptions, Context.NONE);
    }

    /**
     * Transliterate Text.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *         script: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * @param language Specifies the language of the text to convert from one script to another.
     * Possible languages are listed in the transliteration scope obtained by querying the service
     * for its supported languages.
     * @param sourceLanguageScript Specifies the script used by the input text. Look up supported languages using the
     * transliteration scope,
     * to find input scripts available for the selected language.
     * @param targetLanguageScript Specifies the output script. Look up supported languages using the transliteration
     * scope, to find output
     * scripts available for the selected combination of input language and input script.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> transliterateWithResponseAsync(String language, String sourceLanguageScript,
        String targetLanguageScript, BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.transliterate(this.getEndpoint(), language, sourceLanguageScript,
            targetLanguageScript, this.getServiceVersion().getVersion(), accept, body, requestOptions, context));
    }

    /**
     * Transliterate Text.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *         script: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * @param language Specifies the language of the text to convert from one script to another.
     * Possible languages are listed in the transliteration scope obtained by querying the service
     * for its supported languages.
     * @param sourceLanguageScript Specifies the script used by the input text. Look up supported languages using the
     * transliteration scope,
     * to find input scripts available for the selected language.
     * @param targetLanguageScript Specifies the output script. Look up supported languages using the transliteration
     * scope, to find output
     * scripts available for the selected combination of input language and input script.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> transliterateWithResponse(String language, String sourceLanguageScript,
        String targetLanguageScript, BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.transliterateSync(this.getEndpoint(), language, sourceLanguageScript, targetLanguageScript,
            this.getServiceVersion().getVersion(), accept, body, requestOptions, Context.NONE);
    }

    /**
     * Find Sentence Boundaries.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>language</td><td>String</td><td>No</td><td>Language tag identifying the language of the input text.
     * If a code isn't specified, automatic language detection will be applied.</td></tr>
     * <tr><td>script</td><td>String</td><td>No</td><td>Script tag identifying the script used by the input text.
     * If a script isn't specified, the default script of the language will be assumed.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         detectedLanguage (Optional): {
     *             language: String (Required)
     *             score: double (Required)
     *         }
     *         sentLen (Required): [
     *             int (Required)
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> findSentenceBoundariesWithResponseAsync(BinaryData body,
        RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.findSentenceBoundaries(this.getEndpoint(),
            this.getServiceVersion().getVersion(), accept, body, requestOptions, context));
    }

    /**
     * Find Sentence Boundaries.
     * <p><strong>Query Parameters</strong></p>
     * <table border="1">
     * <caption>Query Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>language</td><td>String</td><td>No</td><td>Language tag identifying the language of the input text.
     * If a code isn't specified, automatic language detection will be applied.</td></tr>
     * <tr><td>script</td><td>String</td><td>No</td><td>Script tag identifying the script used by the input text.
     * If a script isn't specified, the default script of the language will be assumed.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addQueryParam}
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         detectedLanguage (Optional): {
     *             language: String (Required)
     *             score: double (Required)
     *         }
     *         sentLen (Required): [
     *             int (Required)
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> findSentenceBoundariesWithResponse(BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.findSentenceBoundariesSync(this.getEndpoint(), this.getServiceVersion().getVersion(), accept,
            body, requestOptions, Context.NONE);
    }

    /**
     * Lookup Dictionary Entries.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         normalizedSource: String (Required)
     *         displaySource: String (Required)
     *         translations (Required): [
     *              (Required){
     *                 normalizedTarget: String (Required)
     *                 displayTarget: String (Required)
     *                 posTag: String (Required)
     *                 confidence: double (Required)
     *                 prefixWord: String (Required)
     *                 backTranslations (Required): [
     *                      (Required){
     *                         normalizedText: String (Required)
     *                         displayText: String (Required)
     *                         numExamples: int (Required)
     *                         frequencyCount: int (Required)
     *                     }
     *                 ]
     *             }
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param sourceLanguage Specifies the language of the input text.
     * The source language must be one of the supported languages included in the dictionary scope.
     * @param targetLanguage Specifies the language of the output text.
     * The target language must be one of the supported languages included in the dictionary scope.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> lookupDictionaryEntriesWithResponseAsync(String sourceLanguage,
        String targetLanguage, BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.lookupDictionaryEntries(this.getEndpoint(), sourceLanguage,
            targetLanguage, this.getServiceVersion().getVersion(), accept, body, requestOptions, context));
    }

    /**
     * Lookup Dictionary Entries.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         normalizedSource: String (Required)
     *         displaySource: String (Required)
     *         translations (Required): [
     *              (Required){
     *                 normalizedTarget: String (Required)
     *                 displayTarget: String (Required)
     *                 posTag: String (Required)
     *                 confidence: double (Required)
     *                 prefixWord: String (Required)
     *                 backTranslations (Required): [
     *                      (Required){
     *                         normalizedText: String (Required)
     *                         displayText: String (Required)
     *                         numExamples: int (Required)
     *                         frequencyCount: int (Required)
     *                     }
     *                 ]
     *             }
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param sourceLanguage Specifies the language of the input text.
     * The source language must be one of the supported languages included in the dictionary scope.
     * @param targetLanguage Specifies the language of the output text.
     * The target language must be one of the supported languages included in the dictionary scope.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> lookupDictionaryEntriesWithResponse(String sourceLanguage, String targetLanguage,
        BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.lookupDictionaryEntriesSync(this.getEndpoint(), sourceLanguage, targetLanguage,
            this.getServiceVersion().getVersion(), accept, body, requestOptions, Context.NONE);
    }

    /**
     * Lookup Dictionary Examples.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *         translation: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         normalizedSource: String (Required)
     *         normalizedTarget: String (Required)
     *         examples (Required): [
     *              (Required){
     *                 sourcePrefix: String (Required)
     *                 sourceTerm: String (Required)
     *                 sourceSuffix: String (Required)
     *                 targetPrefix: String (Required)
     *                 targetTerm: String (Required)
     *                 targetSuffix: String (Required)
     *             }
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param sourceLanguage Specifies the language of the input text.
     * The source language must be one of the supported languages included in the dictionary scope.
     * @param targetLanguage Specifies the language of the output text.
     * The target language must be one of the supported languages included in the dictionary scope.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response} on successful completion of {@link Mono}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Mono<Response<BinaryData>> lookupDictionaryExamplesWithResponseAsync(String sourceLanguage,
        String targetLanguage, BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return FluxUtil.withContext(context -> service.lookupDictionaryExamples(this.getEndpoint(), sourceLanguage,
            targetLanguage, this.getServiceVersion().getVersion(), accept, body, requestOptions, context));
    }

    /**
     * Lookup Dictionary Examples.
     * <p><strong>Header Parameters</strong></p>
     * <table border="1">
     * <caption>Header Parameters</caption>
     * <tr><th>Name</th><th>Type</th><th>Required</th><th>Description</th></tr>
     * <tr><td>X-ClientTraceId</td><td>String</td><td>No</td><td>A client-generated GUID to uniquely identify the
     * request.</td></tr>
     * </table>
     * You can add these to a request with {@link RequestOptions#addHeader}
     * <p><strong>Request Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         text: String (Required)
     *         translation: String (Required)
     *     }
     * ]
     * }</pre>
     * 
     * <p><strong>Response Body Schema</strong></p>
     * 
     * <pre>{@code
     * [
     *      (Required){
     *         normalizedSource: String (Required)
     *         normalizedTarget: String (Required)
     *         examples (Required): [
     *              (Required){
     *                 sourcePrefix: String (Required)
     *                 sourceTerm: String (Required)
     *                 sourceSuffix: String (Required)
     *                 targetPrefix: String (Required)
     *                 targetTerm: String (Required)
     *                 targetSuffix: String (Required)
     *             }
     *         ]
     *     }
     * ]
     * }</pre>
     * 
     * @param sourceLanguage Specifies the language of the input text.
     * The source language must be one of the supported languages included in the dictionary scope.
     * @param targetLanguage Specifies the language of the output text.
     * The target language must be one of the supported languages included in the dictionary scope.
     * @param body Defines the content of the request.
     * @param requestOptions The options to configure the HTTP request before HTTP client sends it.
     * @throws HttpResponseException thrown if the request is rejected by server.
     * @throws ClientAuthenticationException thrown if the request is rejected by server on status code 401.
     * @throws ResourceNotFoundException thrown if the request is rejected by server on status code 404.
     * @throws ResourceModifiedException thrown if the request is rejected by server on status code 409.
     * @return the response body along with {@link Response}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<BinaryData> lookupDictionaryExamplesWithResponse(String sourceLanguage, String targetLanguage,
        BinaryData body, RequestOptions requestOptions) {
        final String accept = "application/json";
        return service.lookupDictionaryExamplesSync(this.getEndpoint(), sourceLanguage, targetLanguage,
            this.getServiceVersion().getVersion(), accept, body, requestOptions, Context.NONE);
    }
}
