//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.azure.ai.vision.common;

import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.DiagnosticsJNI;
import com.azure.ai.vision.common.VisionServiceOptions;

/**
 * Native logging and other diagnostics
 */
public final class Diagnostics {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Starts console logging
     * @param logToStderr Whether the log should be emitted to standard error or standard out.
     */
    public final static void startConsoleLogging(boolean logToStderr) {
        DiagnosticsJNI.startConsoleLogging(logToStderr);
    }

    /**
     * Starts console (for Android, logcat) logging
     */
    public final static void startConsoleLogging() {
        DiagnosticsJNI.startConsoleLogging();
    }

    /**
     * Stops console logging
     */
    public final static void stopConsoleLogging() {
        DiagnosticsJNI.stopConsoleLogging();
    }

    /**
     * Enables the native memory logger
     */
    public final static void startMemoryLogging() {
        DiagnosticsJNI.startMemoryLogging();
    }

    /**
     * Stops the native memory logger
     */
    public final static void stopMemoryLogging() {
        DiagnosticsJNI.stopMemoryLogging();
    }

    /**
     * Dumps the contents of the memory logger
     * @param filename The name of the file to write to. Set this to an empty string if not needed
     * @param linePrefix The prefix to apply to each line of logged memory content, e.g. "CRBN"
     * @param emitToStdOut Whether the log should be emitted to standard output in addition to any other targets
     * @param emitToStdErr Whether the log should be emitted to standard error in addition to any other targets
     */
    public final static void dumpMemoryLog(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr) {
        DiagnosticsJNI.dumpMemoryLog(filename, linePrefix, emitToStdOut, emitToStdErr);
    }

    /**
     * Dumps the contents of the memory logger on exit
     * @param filename The name of the file to write to. Set this to an empty string if not needed
     * @param linePrefix The prefix to apply to each line of logged memory content, e.g. "CRBN"
     * @param emitToStdOut Whether the log should be emitted to standard output in addition to any other targets
     * @param emitToStdErr Whether the log should be emitted to standard error in addition to any other targets
     */
    public final static void dumpMemoryLogOnExit(String filename, String linePrefix, boolean emitToStdOut, boolean emitToStdErr) {
        DiagnosticsJNI.dumpMemoryLogOnExit(filename, linePrefix, emitToStdOut, emitToStdErr);
    }

    /**
     * Enables the native file logger
     * @param filename The name of resulting log file
     * @param append If true will append to file (if already exists). Otherwise will create a new file
     */
    public final static void startFileLogging(String filename, boolean append) {
        DiagnosticsJNI.startFileLogging(filename, append);
    }

    /**
     * Stops the native file logger
     */
    public final static void stopFileLogging() {
        DiagnosticsJNI.stopFileLogging();
    }

    /**
     * Log specific trace message
     * @param message The trace message
     */
    public final static void logMessage(String message) {
        DiagnosticsJNI.logMessage(message);
    }

}
