//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.nio.ByteBuffer;

import com.azure.ai.vision.common.FrameFormat;
import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.FrameJNI;
import com.azure.ai.vision.common.internal.implementation.IntRef;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;
import com.azure.ai.vision.common.VisionServiceOptions;

/**
 * Frame class
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class Frame implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Constructs an instance of Frame from given byte buffer.
     * @param buffer The byte buffer containing pixels of the frame
     */
    public Frame(ByteBuffer buffer) {
        frameBuffer = buffer;
        propertyCollection = new PropertyCollection(PropertiesJNI.createPropertiesHandle());
    }

    /**
     * Constructs an instance of Frame from given byte buffer.
     * @param buffer The byte buffer containing pixels of the frame
     * @param frameFormat The frame format
     */
    public Frame(ByteBuffer buffer, FrameFormat frameFormat) {
        frameBuffer = buffer;
        this.propertyCollection = new PropertyCollection(PropertiesJNI.createPropertiesHandle());
        this.propertyCollection.setProperty("frame.format.pixel_format", String.valueOf(frameFormat.getFourCCAsInt()));
        this.propertyCollection.setProperty("frame.format.width", String.valueOf(frameFormat.getWidth()));
        this.propertyCollection.setProperty("frame.format.height", String.valueOf(frameFormat.getHeight()));
        this.propertyCollection.setProperty("frame.format.stride", String.valueOf(frameFormat.getStride()));
        this.propertyCollection.setProperty("frame.format.source_kind", "SourceKind_Color");
    }

    Frame(SafeHandle handle) {
        Contracts.throwIfNull(handle, "handle");
        frameHandle = handle;
        frameBuffer = null;
        propertyCollection = new PropertyCollection(FrameJNI.getFramePropertiesHandle(handle));
    }

    /**
     * Get the collection of additional frame properties.
     * @return A property collection instance
     */
    public final PropertyCollection getProperties() {
        return propertyCollection;
    }

    /**
     * Get the byte buffer associated with the frame.
     * @return The buffer
     */
    public final ByteBuffer getData() {
        return frameBuffer;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public final void close() {

        if (this.propertyCollection != null) {
            this.propertyCollection.close();
            this.propertyCollection = null;
        }
        if (this.frameHandle != null) {
            this.frameHandle.close();
            this.frameHandle = null;
        }
    }

    private ByteBuffer frameBuffer;
    private PropertyCollection propertyCollection;
    private SafeHandle frameHandle;
}
