//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.VisionServiceOptions;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.IntRef;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.FrameFormatJNI;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;

/**
 * FrameFormat class
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class FrameFormat implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    FrameFormat(SafeHandle handle) {
        formatHandle = handle;
        propertyCollection = new PropertyCollection(FrameFormatJNI.getFrameFormatPropertiesHandle(handle));
    }

    /**
     * Creates FrameFormat instance from FourCCFormat
     * @param ch1 First character of the FourCC
     * @param ch2 Second character of the FourCC
     * @param ch3 Third character of the FourCC
     * @param ch4 Fourth character of the FourCC
     * @param width Width of the frame, in pixels
     * @param height Height of the frame, in pixels
     * @param stride Width of the frame, in bytes
     * @return A FrameFormat instance
     */
    public static FrameFormat createFourCCFormat(char ch1, char ch2, char ch3, char ch4, int width, int height, int stride) {

        try (SafeHandle propertiesHandle = PropertiesJNI.createPropertiesHandle();
             PropertyCollection props = new PropertyCollection(propertiesHandle)) {

            props.setProperty("frame.format.width", String.valueOf(width));
            props.setProperty("frame.format.height", String.valueOf(height));
            if (stride > 0) {
                props.setProperty("frame.format.stride", String.valueOf(stride));
            }
            String fourcc = "" + ch1 + ch2 + ch3 + ch4;
            props.setProperty("frame.format.fourcc", fourcc);

            SafeHandle formatHandle = FrameFormatJNI.createFrameFormatHandle(ch1, ch2, ch3, ch4, propertiesHandle);
            return new FrameFormat(formatHandle);
        }
    }

    /**
     * Creates FrameFormat instance from RGB format
     * @param bitsPerPixel Number of bits describing each pixel
     * @param width Width of the frame, in pixels
     * @param height Height of the frame, in pixels
     * @param stride Width of the frame, in bytes
     * @return A FrameFormat instance
     */
    public static FrameFormat createRGBFormat(int bitsPerPixel, int width, int height, int stride) {
        FrameFormat format = createFourCCFormat('R', 'G', 'B', ' ', width, height, stride);
        format.getProperties().setProperty("frame.format.bits.per.pixel", String.valueOf(bitsPerPixel));
        return format;
    }

    /**
     * Gets the image format's FOURCC as ASCII string
     * @return The FOURCC format code as ASCII string
     */
    public String getFourCCAsString() {
        return this.propertyCollection.getProperty("frame.format.fourcc");
    }

    /**
     * Gets the image format's FOURCC as integer value
     * @return The FOURCC format code as a integer
     */
    public int getFourCCAsInt() {
        String fourCCStr = this.propertyCollection.getProperty("frame.format.fourcc");
        int fourCC = intFourcc(fourCCStr);
        return fourCC;
    }

    /**
     * Gets the image format's pixel height.
     * @return The image pixel height.
     */
    public int getHeight() {
        String height = this.propertyCollection.getProperty("frame.format.height");
        return Integer.parseInt(height);
    }

    /**
     * Gets the image format's pixel width.
     * @return The image pixel width.
     */
    public int getWidth() {
        String width = this.propertyCollection.getProperty("frame.format.width");
        return Integer.parseInt(width);
    }

    /**
     * Gets the image format's pixel stride.
     * @return The image pixel stride.
     */
    public int getStride() {
        String stride = this.propertyCollection.getProperty("frame.format.stride");
        return Integer.parseInt(stride);
    }

    /**
     * Gets the image format's bits per pixel value.
     * @return The image's bits per pixel value.
     */
    public int getBitsPerPixel() {
        String bitsPerPixel = this.propertyCollection.getProperty("frame.format.bits.per.pixel");
        return Integer.parseInt(bitsPerPixel);
    }

    /**
     * Sets the image format's bits per pixel value.
     * @param value The image's bits per pixel value.
     */
    public void setBitsPerPixel(int value) {
        this.propertyCollection.setProperty("frame.format.bits.per.pixel", String.valueOf(value));
    }

    /**
     * Gets the collection of additional FrameFormat properties
     * @return A property collection instance
     */
    public PropertyCollection getProperties() {
        return propertyCollection;
    }

    /*! \cond INTERNAL */

    /**
     * Gets the internal native handle.
     * TODO This is internal use only
     * @return A SafeHandle instance
     */
    public SafeHandle getHandle() {
        return formatHandle;
    }

    /*! \endcond */

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {

        if (this.propertyCollection != null) {
            this.propertyCollection.close();
            this.propertyCollection = null;
        }

        if (this.formatHandle != null) {
            this.formatHandle.close();
            this.formatHandle = null;
        }
    }

    private static int makeInt(byte b3, byte b2, byte b1, byte b0) {
        return (((b3) << 24) |
            ((b2 & 0xff) << 16) |
            ((b1 & 0xff) <<  8) |
            ((b0 & 0xff)));
    }

    private static int intFourcc(String string) {
        byte[] b = string.getBytes();
        return makeInt(b[3], b[2], b[1], b[0]);
    }

    private SafeHandle formatHandle = null;
    private PropertyCollection propertyCollection;
}
