//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;

import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.EventSource;
import com.azure.ai.vision.common.internal.implementation.FrameReaderJNI;
import com.azure.ai.vision.common.internal.implementation.VisionResultJNI;
import com.azure.ai.vision.common.FrameFormat;
import com.azure.ai.vision.common.VisionSource;
import com.azure.ai.vision.common.VisionServiceOptions;

/**
 * FrameReader class
 * Represents the ability to read image frame data for input Vision AI scenarios.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class FrameReader implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    FrameReader(SafeHandle handle) {
        Contracts.throwIfNull(handle, "handle");
        frameReaderHandle = handle;
    }

    /**
     * Creates FrameReader from vision source
     * @param source The vision source
     * @return A FrameReader instance
     */
    public static FrameReader fromVisionSource(VisionSource source) {
        throw new java.lang.UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Creates FrameReader from the result
     * @param resultHandle SafeHandle to the result
     * @return A FrameReader instance
     */
    public static FrameReader fromResult(SafeHandle resultHandle) {
        Contracts.throwIfNull(resultHandle, "resultHandle");
        try (SafeHandle handle = VisionResultJNI.getFrameReaderHandle(resultHandle)) {
            return new FrameReader(handle);
        }
    }

    /**
     * Reads a single frame of image data from the underlying frame.
     * @param position The position of the frame in frames
     * @return A Frame instance
     */
    public Frame readFrame(long position) {
        Contracts.throwIfNull(frameReaderHandle, "frameReaderHandle");
        SafeHandle frameHandle = FrameReaderJNI.read(frameReaderHandle, position, 0);
        return new Frame(frameHandle);
    }

    /**
     * Gets the frame format associated to frame reader
     * @return A FrameFormat instance
     */
    public FrameFormat getFrameFormat() {
        return frameFormat;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {

        if (this.frameReaderHandle != null) {
            this.frameReaderHandle.close();
            this.frameReaderHandle = null;
        }

        if (this.frameFormat != null) {
            this.frameFormat.close();
            this.frameFormat = null;
        }
    }

    private SafeHandle frameReaderHandle;
    private FrameFormat frameFormat;

}
